/** This file is part of PeerHood.
*
*   PeerHood is free software: you can redistribute it and/or modify
*   it under the terms of the GNU Lesser General Public License 
*   version 2 as published by the Free Software Foundation.
*
*   PeerHood is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU Lesser General Public License for more details.
*
*   You should have received a copy of the GNU Lesser General Public
*   License along with PeerHood. If not, see <http://www.gnu.org/licenses/>.
*/

/**
 * Copyright 2008 LUT. .
 *
 * @name AbstractListenerCreator.h
 * @memo Definition of a pure virtual creator that is used by listener factory
 * to address concrete creators of the listeners 
 *
 * @version 0.03
 * date     25.08.2008
 * change   18.02.2008
 */

#ifndef __ABSTRACTLISTENERCREATOR_H__
#define __ABSTRACTLISTENERCREATOR_H__

#include "AbstractStateConverter.h"
#include "AbstractListener.h"

/**
 * @memo Defines the PeerHood Listener creator interface (pure virtual creator)
 * @doc This pure virtual class defines the interface for creating listeners for various
 * purposes without the need to know anything about underlying technology. Concrete
 * listeners are created via listener factory by using the concrete creator class which
 * has to implement this class. All created listeners will register to component that
 * created it through AbstractStateConverter -interface (that will have to be implemented
 * by the component that is going to use listeners), a reference to this given as one creation
 * parameter, listeners use RegisterListener()-function to register itself to that component.
 */

class MAbstractListenerCreator
{
public:
	
	/**
	 * @memo Default destructor
	 * @doc Default destructor, does nothing.
	 */
	virtual ~MAbstractListenerCreator() {};
	
	/**
	 * @memo Transparently create listener objects based on their prototype names.
	 * @doc Creates a new listener based on the given name, used only through listener
	 * factory.
	 * 
	 * @param aName a string representation of the concrete listener prototype
	 * @param aConverter reference to the owner object, CANNOT BE NULL!
	 * 
	 * @returns Created concrete listener object or NULL if the given name doesn't match 
	 * the prototype name. 
	 */
	virtual MAbstractListener* CreateListener(const std::string& aName, MAbstractStateConverter* aConverter) = 0;
};

#endif /*__ABSTRACTLISTENERCREATOR_H__*/
