/** This file is part of PeerHood.
*
*   PeerHood is free software: you can redistribute it and/or modify
*   it under the terms of the GNU Lesser General Public License 
*   version 2 as published by the Free Software Foundation.
*
*   PeerHood is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU Lesser General Public License for more details.
*
*   You should have received a copy of the GNU Lesser General Public
*   License along with PeerHood. If not, see <http://www.gnu.org/licenses/>.
*/

// Emacs, hail the mighty -*-C++-*-

/**
 * Copyright 2003 LUT. .
 *
 * @name PHIterator.h
 * @memo PeerHood iterator template.
 *
 * @version 0.1
 * date     17.04.2003
 * change   17.04.2003
 */

#ifndef __PHITERATOR_H__
#define __PHITERATOR_H__

#include "PHListItem.h"

#ifdef __cplusplus

#include <cassert>

/**
 * @memo PeerHood iterator template class.
 * @memo PeerHood iterator template class. This iterator is used internally by
 * the PeerHood system when delivering neighborhood info via the PeerHood
 * library. Users should not use this class directly!
 *
 * @see CPHList
 * @see CPHListItem
 */
template <class T>
class CPHIterator
{
 public:
  CPHIterator() {};
  T& operator*();
  CPHIterator operator++(int);
  CPHIterator& operator++();
  bool operator!=(const CPHIterator& aItem);

 private:
  friend class CPHList<T>;

  CPHIterator(CPHListItem<T>* aOriginal);
  CPHListItem<T>* iCurrent;
};


/**
 * @memo Copy constructor.
 * @doc Copy constructor. Creates a new iterator that points to the same
 * element as the original one.
 *
 * @param aOriginal The original iterator.
 *
 * @return none
 */
template <class T>
CPHIterator<T>::CPHIterator(CPHListItem<T>* aOriginal)
{
  iCurrent = aOriginal;
}


/**
 * @memo The '*' operator.
 * @doc The '*' operator. This operator returns a pointer to the data element
 * pointed by the current iterator.
 *
 * @return pointer to the current data element
 */
template <class T>
T& CPHIterator<T>::operator*()
{
  return iCurrent->iData;
}


/**
 * @memo The suffix '++' operator.
 * @doc The suffix '++' operator. This operator increases the current iterator
 * by one so it starts to point to the next element on the list. If the current
 * iterator is the last one then an assertion takes place.
 *
 * @return iterator pointing to the next list element
 */
template <class T>
CPHIterator<T> CPHIterator<T>::operator++(int)
{       
  assert(iCurrent != NULL);
  CPHIterator<T> temp = *this;
  ++(*this);
  return temp;
}


/**
 * @memo the prefix '++' operator.
 * @doc the prefix '++' operator. This operator increases the current iterator
 * by one so it starts to point to the next element on the list. If the current
 * iterator is the last one then an assertion takes place.
 *
 * @return iterator pointing to the next list element
 */
template <class T>
CPHIterator<T>& CPHIterator<T>::operator++()
{
  assert(iCurrent != NULL);
  iCurrent = iCurrent->iNext;
  return *this;
}


/**
 * @memo The '!=' operator.
 * @doc The '!=' operator. This operator test if the current iterator and
 * another iterator are the same.
 *
 * @param aItem The iterator to compare with.
 *
 * @return true if the two iterators are the same
 */
template <class T>
bool CPHIterator<T>::operator!=(const CPHIterator<T>& aItem) 
{
  return iCurrent != aItem.iCurrent;
}
#else

#endif

#endif
