/** This file is part of PeerHood.
*
*   PeerHood is free software: you can redistribute it and/or modify
*   it under the terms of the GNU Lesser General Public License 
*   version 2 as published by the Free Software Foundation.
*
*   PeerHood is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU Lesser General Public License for more details.
*
*   You should have received a copy of the GNU Lesser General Public
*   License along with PeerHood. If not, see <http://www.gnu.org/licenses/>.
*/

/**
 * Copyright 2003 LUT. .
 *
 * @name PHObject.cc
 * @memo Implementation of the PeerHood object class.
 *
 * @version 0.1
 * date     03.04.2003
 * change   03.04.2003
 */

#include <PHObject.h>


/**
 * @memo Standard constructor.
 * @doc Standard constructor.
 *
 * @return none
 */
CPHObject::CPHObject()
{
  iName = NULL;
  iConnection = NULL;
}


/**
 * @memo Destructor.
 * @doc Destructor. If a connection is associated with the object then it is
 * closed and deleted before the object is allowed to cease.
 *
 * @return none
 */
CPHObject::~CPHObject()
{
  if (iConnection) {
    if (iConnection->IsConnected()) iConnection->Disconnect();
    delete iConnection;
  }

  delete[] iName;
}


/**
 * @memo Returns the name of the object.
 * @doc Returns the name of the object.
 *
 * @return object's name
 */
const char* CPHObject::GetName()
{
  return iName;
}


/**
 * @memo Returns the prototype of object's desired networking technology.
 * @doc Returns the prototype of object's desired networking technology.
 *
 * @return the prototype of object's desired networking technology
 */
const std::string& CPHObject::GetPrototype() const
{
  return iPrototype;
}


/**
 * @memo Returns object's address.
 * @doc Returns the address that is associated with the object. This is also
 * the address of the device that hosts the object.
 *
 * @return the address associated with the object
 */
const std::string& CPHObject::GetAddress() const
{
  return iAddress;
}


/**
 * @memo Returns object's port.
 * @doc Returns the port part of object's address. The port is used by the 
 * PeerHood library when contacting the service mapped to the object.
 *
 * @return port part of the address associated with the object
 */
const int CPHObject::GetPort() const
{
  return iPort;
}


/**
 * @memo Returns a pointer to object's connection.
 * @doc Returns a pointer to objects connection. 
 *
 * @return pointer to object's connection object or NULL if there's no 
 * associated connection
 */
MAbstractConnection* CPHObject::GetConnection()
{
  return iConnection;
}


/**
 * @memo Associates a connection with the object.
 * @doc Associates a connection with the object. If another connection is 
 * already associated with the object then it's closed and deleted before the
 * new connection is associated.
 *
 * @return none
 */
void CPHObject::SetConnection(MAbstractConnection* aConnection)
{
  if (iConnection) {
    if (iConnection->IsConnected()) iConnection->Disconnect();
    delete iConnection;
  }

  iConnection = aConnection;
}
