/** This file is part of PeerHood.
*
*   PeerHood is free software: you can redistribute it and/or modify
*   it under the terms of the GNU Lesser General Public License 
*   version 2 as published by the Free Software Foundation.
*
*   PeerHood is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU Lesser General Public License for more details.
*
*   You should have received a copy of the GNU Lesser General Public
*   License along with PeerHood. If not, see <http://www.gnu.org/licenses/>.
*/

/**
 * Copyright 2009,2010 LUT.
 *
 * @name IteratorWrapper.h
 * @memo Header for iterator wrappers (device and service), needed for C API.
 *
 * @version 0.3
 * date     20.11.2009
 * change   18.02.2010
 */

#ifndef __ITERATORWRAPPER_H_
#define __ITERATORWRAPPER_H_

#include "PeerHood.h"

#ifdef __cplusplus

/**
 * @memo Wrapper class for TDeviceIterator.
 * @doc Wrapper for TDeviceIterator, needed for C API.
 */
class DeviceIterator
{
	public:
		/**
		 * @memo Constructor for DeviceIterator
		 * @doc Create new DeviceIterator and initialize the actual iterator.
		 */
		DeviceIterator(TDeviceIterator aDeviceIterator);
		
		/**
		 * @memo Destructor for DeviceIterator
		 * @doc Delete the DeviceIterator
		 */
		~DeviceIterator();
		
		/**
		 * @memo Get the iterator.
		 * @doc Returns the content of this wrapper.
		 *
		 * @return The content of the wrapper
		 */
		TDeviceIterator GetIterator();
		
		/**
		 * @memo Set a specific iterator as content.
		 * @doc Assign the given iterator object to be as the content of this
		 * iterator.
		 */
		void SetIterator(TDeviceIterator aDeviceIterator);
		
		/**
		 * @memo Increase the iterator,
		 * @doc Move on to the next actual iterator.
		 */
		void NextIterator();
		
	private:
		/**
		 * @memo Default constructor - private!
		 * @doc Default constructor usage prevented
		 */
		DeviceIterator();
		
		// Iterator holder
		// TODO a workaround to use TDeviceIterator as pointer? 
		TDeviceIterator iIterator;
};

/**
 * @memo Wrapper class for TServiceIterator.
 * @doc Wrapper for TServiceIterator, needed for C API.
 */
class ServiceIterator
{
	public:
		
		/**
		 * @memo Constructor for ServiceIterator
		 * @doc Create new ServiceIterator and initialize the actual iterator.
		 */
		ServiceIterator(TServiceIterator aServiceIterator);
		
		/**
		 * @memo Destructor for ServiceIterator
		 * @doc Delete the ServiceIterator
		 */
		~ServiceIterator();
		
		/**
		 * @memo Get the iterator.
		 * @doc Returns the content of this wrapper.
		 *
		 * @return The content of the wrapper
		 */
		TServiceIterator GetIterator();
		
		/**
		 * @memo Set a specific iterator as content.
		 * @doc Assign the given iterator object to be as the content of this
		 * iterator.
		 */
		void SetIterator(TServiceIterator aServiceIterator);
		
		/**
		 * @memo Increase the iterator,
		 * @doc Move on to the next actual iterator.
		 */
		void NextIterator();
		
	private:
		/**
		 * @memo Default constructor - private!
		 * @doc Default constructor usage prevented
		 */
		ServiceIterator();
		
		// Iterator holder
		// TODO a workaround to use TServiceIterator as pointer? 
		TServiceIterator iIterator;
};

#else
// DeviceIterator for C
typedef
	struct DeviceIterator
		DeviceIterator;
// ServiceIterator for C
typedef
	struct ServiceIterator
		ServiceIterator;
#endif //__cplusplus

#ifdef __cplusplus
extern "C" {
#endif

DeviceIterator* ph_c_new_device_iterator(TDeviceList* aDeviceList);

void ph_c_delete_device_iterator(DeviceIterator* aWrapper);

MAbstractDevice* ph_c_device_iterator_get_device(DeviceIterator* aWrapper);

void ph_c_device_iterator_next_iterator(DeviceIterator* aWrapper);

int ph_c_device_iterator_is_last(DeviceIterator* aWrapper, TDeviceList* aDeviceList);

ServiceIterator* ph_c_new_service_iterator(TServiceList* aServiceList);

void ph_c_delete_service_iterator(ServiceIterator* aWrapper);

CService* ph_c_service_iterator_get_service(ServiceIterator* aWrapper);

void ph_c_service_iterator_next_iterator(ServiceIterator* aWrapper);

int ph_c_service_iterator_is_last(ServiceIterator* aWrapper, TServiceList* aServiceList);

#ifdef __cplusplus
}
#endif

#endif //__ITERATORWRAPPER_H_
