/** This file is part of PeerHood.
*
*   PeerHood is free software: you can redistribute it and/or modify
*   it under the terms of the GNU Lesser General Public License 
*   version 2 as published by the Free Software Foundation.
*
*   PeerHood is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU Lesser General Public License for more details.
*
*   You should have received a copy of the GNU Lesser General Public
*   License along with PeerHood. If not, see <http://www.gnu.org/licenses/>.
*/

/**
 * Copyright 2003 LUT. .
 *
 * @name WLANCreator.cc
 * @memo WLAN implementation of the MAbstractCreator interface.
 *
 * @version 0.1
 * date     30.06.2003
 * change   30.06.2003
 */

#include <fstream>
#include "WLANCreator.h"
#include <TCPConnection.h>
#include "WLANPinger.h"
#include "WLANMonitor.h"
#include <Factory.h>

/**
 * @memo Default constructor.
 * @doc Default constructor, sets internal variables to their default values
 * and insert creator to the PeerHood factory.
 *
 * @return none
 */
CWLANCreator::CWLANCreator()
{
  iProto = std::string("wlan-base");
  Factory::Register(this);
}


/**
 * @memo Creates a new WLAN specific connection object.
 * @doc Creates a new WLAN specific connection object. The object is
 * created dynamically so the caller is responsible of deletion of the object.
 *
 * @param aProto Prototype of the desired object's type.
 *
 * @return new CWLAConnection object or NULL if the prototype was not recognized
 */
MAbstractConnection* CWLANCreator::CreateConnectionL(const std::string& aProto)
{
  std::string nDevice = "WLANInterface";
  if(!LoadParameter(nDevice))
    {
      // ERR("CGPRSPlugin::WLANCreator : Interface entry not found from .phconfig!");
      return NULL;
    }

  if (iProto.compare(aProto) == 0) {
    return new CTCPConnection(nDevice, "WLAN");
  }

  return NULL;
}


/**
 * @memo Creates a new WLAN specific pinger object.
 * @doc Creates a new WLAN specific pinger object. The object is
 * created dynamically so the caller is responsible of deletion of the object.
 *
 * @param aProto Prototype of the desired object's type.
 * @param aAddress Remote device's address.
 *
 * @return new CWLANPinger object or NULL if the prototype was not recognized
 */
MAbstractPinger* CWLANCreator::CreatePingerL(const std::string& aProto, const std::string& aAddress)
{
  if (iProto.compare(aProto) == 0) {
    return new CWLANPinger(aAddress);
  }

  return NULL;
}

/**
 * @memo Creates a new WLAN specific monitor object.
 * @doc Creates a new WLAN specific monitor object. The object is
 * created dynamically so the caller is responsible of deletion of the object.
 *
 * @param aProto Prototype of the desired object's type.
 * @param aAddress Remote device's address.
 *
 * @return new CWLANPinger object or NULL if the prototype was not recognized
 */
MAbstractMonitor* CWLANCreator::CreateMonitorL(const std::string& aProto, const std::string& aAddress)
{
  if (iProto.compare(aProto) == 0) {
    return new CWLANMonitor(aAddress);
  }

  return NULL;
}


/**
 * @memo Returns creator's unique id string.
 * @doc Returns creator's unique id string. WLAN creator's id <i>wlan-base
 * </i> is the same as WLAN plugin's id.
 *
 * @return creator's unique id string
 */
const std::string& CWLANCreator::GetPrototype()
{
  return iProto;
}

/**
 * @memo Loads a parameter from the configuration file.
 * @doc Loads a parameter from the configuration file. If the requested
 * parameter is found then the argument <code>aParameter</code> is replaced
 * with parameter's value.
 *
 * @param aParameter Parameter's name.
 *
 * @return true if the requested parameter was found
 */
bool CWLANCreator::LoadParameter(std::string& aParameter)
{
  char* home;
  std::string line;

  /*
   * TODO: This use should be from library.
   */
  if ((home = getenv("HOME")) == NULL) {
    return false;
  }

  std::string path(home);
  path.append("/.phconfig");

  std::ifstream in(path.c_str());

  if (!in.is_open()) {
    // ERR("Failed to open the configuration file $HOME/.phconfig");
    in.close();
    return false;
  }

  while (!in.eof()) {
    getline(in, line);
    if (line[0] == '#') continue;
    std::string::size_type position = line.find_first_of(':');

    if (position != std::string::npos) {
      std::string parameter = line.substr(0, position);
      //  DBG("parameter is %s", parameter.c_str());
      if (parameter.compare(aParameter) == 0) {
        aParameter = line.substr(++position);
        in.close();
        return true;
      }
    }
  }

  in.close();
  return false;
}

static CWLANCreator wlancreator;
