/** This file is part of PeerHood.
*
*   PeerHood is free software: you can redistribute it and/or modify
*   it under the terms of the GNU Lesser General Public License 
*   version 2 as published by the Free Software Foundation.
*
*   PeerHood is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU Lesser General Public License for more details.
*
*   You should have received a copy of the GNU Lesser General Public
*   License along with PeerHood. If not, see <http://www.gnu.org/licenses/>.
*/

/**
 * Copyright 2003 LUT. .
 *
 * @name IFSearch.cc
 * @memo Implementation of the PeerHood IFSearch class.
 *
 * @version 0.1
 * date     22.07.2003
 * change   22.07.2003
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <signal.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <sys/select.h>
#include <errno.h>
#include <sys/uio.h>
#include <net/if.h>
#include <linux/sockios.h>
#include "IFSearch.h"

#include <iostream>

#warning "Temporary debug"
//temp
#include <syslog.h>

#define ERR(format, msg...) syslog(LOG_ERR, "ERROR: " format "\n" , ## msg)

#ifdef PH_DEBUG
#define DBG(format, msg...) syslog(LOG_DEBUG, format "\n" , ## msg)
#else
#define DBG( A... )
#endif
//temp

/**
 * @memo Default constructor.
 * @doc Default constructor, Does nothing
 *
 * @return none
 */
CIFSearch::CIFSearch() {
}

/*
 * TODO: NULL should checked on every GetIFInfo, causes segfault.
 */
/**
 * @memo Returns all the found network interfaces
 * @doc Returns all the found network interfaces.
 *
 * @return Pointer to the CIFInfo linked list which
 * contains information about found interfaces. Return NULL if no interfaces
 * are found.
 */
struct IFInfo *CIFSearch::GetIFInfo(const std::string& aIface)
{
 int sockfd, len, lastlen, flags, myflags;
  char *ptr, *buf, lastname[IFNAMSIZ+48], *cptr;
  struct ifconf	ifc;
  struct ifreq *ifr, ifrcopy;
  struct sockaddr_in *sinptr;
  int family = AF_INET;
  int doaliases = 1;
  char str[16];
  struct IFInfo *ifaces = new struct IFInfo;
  bool found = false;

  sockfd = socket(AF_INET, SOCK_DGRAM, 0);

  lastlen = 0;
  len = 100 * sizeof(struct ifreq);// initial buffer size guess
  for ( ; ; ) {
    buf = new char[len];
    ifc.ifc_len = len;
    ifc.ifc_buf = buf;

    if (ioctl(sockfd, SIOCGIFCONF, &ifc) < 0) {
      if (errno != EINVAL || lastlen != 0)
	{
	  ERR("CIFSearch::GetIFInfo : ioctl error");
	  delete ifaces;
	  close(sockfd);
	  delete[] buf;
	  return false;
	}
    } else {
      if (ifc.ifc_len == lastlen)
	break;		// success, len has not changed
      lastlen = ifc.ifc_len;
    }
    len += 10 * sizeof(struct ifreq);	// increment
    delete[] buf;
  }

  lastname[0] = 0;

  for (ptr = buf; ptr < buf + ifc.ifc_len; ) {
    ifr = (struct ifreq *) ptr;

#ifdef	HAVE_SOCKADDR_SA_LEN
    len = max(sizeof(struct sockaddr), ifr->ifr_addr.sa_len);
#else
    switch (ifr->ifr_addr.sa_family) {
#ifdef	IPV6
    case AF_INET6:
      len = sizeof(struct sockaddr_in6);
      break;
#endif
    case AF_INET:
    default:
      len = sizeof(struct sockaddr);
      break;
    }
#endif	// HAVE_SOCKADDR_SA_LEN
    ptr += sizeof(ifr->ifr_name) + len;	// for next one in buffer

    if (ifr->ifr_addr.sa_family != family)
      continue;	// ignore if not desired address family

    myflags = 0;
    if ( (cptr = strchr(ifr->ifr_name, ':')) != NULL)
      //*cptr = 0;		// replace colon will null
    if (strncmp(lastname, ifr->ifr_name, IFNAMSIZ+48) == 0) {
      if (doaliases == 0)
	continue;	// already processed this interface
      myflags = IFI_ALIAS;
    }
    memcpy(lastname, ifr->ifr_name, IFNAMSIZ+48);

    ifrcopy = *ifr;
    ioctl(sockfd, SIOCGIFFLAGS, &ifrcopy);

    flags = ifrcopy.ifr_flags;
    if ((flags & IFF_UP) == 0)
      continue;	// ignore if interface not up

    memcpy(str, ifr->ifr_name, IFI_NAME);

    str[IFI_NAME-1] = '\0';

    if(aIface == str)
      {
	//unicast
	sinptr = (struct sockaddr_in *) &ifr->ifr_addr;
	memcpy(&ifaces->iUcast, sinptr, sizeof(struct sockaddr_in));

	//bcast
#ifdef	SIOCGIFBRDADDR
	if (flags & IFF_BROADCAST) {
	  ioctl(sockfd, SIOCGIFBRDADDR, &ifrcopy);
	  sinptr = (struct sockaddr_in *) &ifrcopy.ifr_broadaddr;
	  memcpy(&ifaces->iBcast, sinptr, sizeof(struct sockaddr_in));
	}
#endif
	found = true;
      }
  }

  close(sockfd);
  delete[] buf;

  if(!found)
    {
      delete ifaces;
      return NULL;
    }
  return ifaces;
}



