#ifndef _CTF_H_
#define _CTF_H_

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <time.h>

#include "ctf_io.h"

#define CTF_LOG_LEVEL_DEFAULT 0x00
#define CTF_LOG_LEVEL_SILENT  0x01
#define CTF_LOG_LEVEL_VERBOSE 0x02

#define CTF_TEST_NAME_LENGTH  256

#define GLOBAL() \
int  CTF_log_level = CTF_LOG_LEVEL_DEFAULT; \
char CTF_current_test_case_name[CTF_TEST_NAME_LENGTH]; \
int  CTF_passed = 0; /* Number of passed tests */      \
int  CTF_failed = 0; /* Number of failed tests */      \
int  CTF_total  = 0; /* Total number of tests  */

#define GLOBAL_DECL() \
extern int  CTF_log_level; \
extern char CTF_current_test_case_name[]; \
extern int  CTF_passed; \
extern int  CTF_failed; \
extern int  CTF_total;

#define CTF_TEST_NAME(test_name) \
    strncpy(CTF_current_test_case_name, test_name, CTF_TEST_NAME_LENGTH);

#define CTF_TEST_CASE(test_case_name) \
    void test_case_name()

#define CTF_CHECK(expr) \
    if((expr)) { PASS(); } else { FAIL(); }

#define CTF_CHECK_EQUAL(expr1, expr2) \
    if((expr1) == (expr2)) { PASS(); } else { FAIL(); }

#define CTF_CHECK_EQUAL_MSG(expr1, expr2, msg) \
    if((expr1) == (expr2)) { PASS_MSG(msg); } else { FAIL_MSG(msg); }

#define CTF_REQUIRE(expr) \
    if((expr)) { PASS(); } else { FAIL(); THROW; }

#define CTF_REQUIRE_MSG(expr, msg) \
    if((expr)) { PASS_MSG(msg); } else { FAIL_MSG(msg); THROW; }

#ifdef CTF_LOG_FORMAT_USE_TIME
#define CTF_GET_CURRENT_TIME_STR(str, size) \
    time_t now = time(NULL); \
    struct tm tm_now;        \
    localtime_r(&now, &tm_now); \
    strftime(s_now, size, "[%d/%m/%y][%H:%M:%S]", &tm_now);
#else
#define CTF_GET_CURRENT_TIME_STR(str, size) \
    memset(str, 0, size);
#endif

#define PASS() \
    CTF_passed++; \
    CTF_total++; \
    if(CTF_log_level == CTF_LOG_LEVEL_VERBOSE) { \
        char s_now[256]; \
        CTF_GET_CURRENT_TIME_STR(s_now, 256); \
        fprintf(stderr, "%s %s(%d): check in \"%s\": check passed\n", \
                s_now, __FILE__, __LINE__, CTF_current_test_case_name); \
    }

#define FAIL() \
    CTF_failed++; \
    CTF_total++; \
    if(CTF_log_level != CTF_LOG_LEVEL_SILENT) { \
        char s_now[256]; \
        CTF_GET_CURRENT_TIME_STR(s_now, 256); \
        fprintf(stderr, "%s %s(%d): error in \"%s\": check failed\n", \
                s_now, __FILE__, __LINE__, CTF_current_test_case_name); \
    }

#define PASS_MSG(msg) \
    CTF_passed++; \
    CTF_total++; \
    if(CTF_log_level == CTF_LOG_LEVEL_VERBOSE) { \
        char s_now[256]; \
        CTF_GET_CURRENT_TIME_STR(s_now, 256); \
        fprintf(stderr, "%s %s(%d): %s\n", \
                s_now, __FILE__, __LINE__, msg); \
    }

#define FAIL_MSG(msg) \
    CTF_failed++; \
    CTF_total++; \
    if(CTF_log_level != CTF_LOG_LEVEL_SILENT) { \
        char s_now[256]; \
        CTF_GET_CURRENT_TIME_STR(s_now, 256); \
        fprintf(stderr, "%s %s(%d): %s\n", \
                s_now, __FILE__, __LINE__, msg); \
    }

#ifdef CTF_THROW_EXIT
#define THROW exit(EXIT_FAILURE)
#else
#define THROW return
#endif

#define CTF_TEST_DATA_FILE(file_name) \
    const char CTF_test_data_file_name[] = file_name

#define CTF_TEST_DATA_FILE_NAME_DECL \
    extern const char CTF_test_data_file_name[]

#define CTF_STAT_FILE(file_name) \
    const char CTF_stat_file_name[] = file_name

#define CTF_STAT_FILE_NAME_DECL \
    extern const char CTF_stat_file_name[]

#define CTF_TEST_DATA(var_name) \
    ctf_read_data(var_name, CTF_test_data_file_name)

#define WRITE_STATS(passed, failed, total) \
    ctf_write_stat(passed, failed, total, CTF_stat_file_name)

#define CTF_RUN(...) \
   int main(int argc, char *argv[]) { \
       if(argc == 2) { \
           if(strcmp(argv[1], "silent") == 0) {          \
               CTF_log_level = CTF_LOG_LEVEL_SILENT;     \
           } else if(strcmp(argv[1], "default") == 0) {  \
               CTF_log_level = CTF_LOG_LEVEL_DEFAULT;    \
           } else if(strcmp(argv[1], "verbose") == 0) {  \
               CTF_log_level = CTF_LOG_LEVEL_VERBOSE;    \
           }                                             \
       }                                                 \
       if(CTF_log_level != CTF_LOG_LEVEL_SILENT) {       \
           fprintf(stderr, "Running tests...\n");        \
       }                                                 \
       __VA_ARGS__;                                      \
       WRITE_STATS(CTF_passed, CTF_failed, CTF_total);   \
       return 0;					 \
   }

#endif /* _CTF_H_ */
