/*
 * profile.c - methods and structs for work with profile data
 * This file is part of MySocials driver library
 *
 * Copyright (C) 2010 - MySocials team
 *
 * libvkontakte library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * libvkontakte library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with libvkontakte library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor,
 * Boston, MA  02110-1301  USA
 */
 
#include <stdlib.h>
#include <string.h>

#include "profile.h"
#include "driver.h"

/**
 * Create empty profile with id.
 *
 * @param name - string for identify current module
 * @return new profile entity
 */
struct ProfileData *profile_create(gchar *name)
{
    struct ProfileData *ret = (struct ProfileData *)malloc(sizeof (struct ProfileData));
    int num_accounts = 0; /* number of exist accounts */
    int id_busy = 0;
 
    if (profile_list == NULL) {
    	profile_list = ret;
    } else {
        struct ProfileData *last = (struct ProfileData *)profile_list;
        while(last->next != NULL) {
            if(g_strcmp0(last->id, name) == 0) {
                id_busy = 1;
            }
            num_accounts++;
            last = last->next;
        }
        last->next = ret;
    }
    
    if(id_busy)
        ret->id = g_strdup_printf("%s%d", name, num_accounts);
    else
        ret->id = g_strdup_printf("%s", name);

    ret->username = NULL;
    ret->token = NULL;
    ret->secret = NULL;
    ret->subdomain = NULL;
    ret->next = NULL;

    return ret;
}

/**
 * remove selected profile
 *
 * @param data - pointer to selected profile
 */
void profile_remove(struct ProfileData *data)
{
    if (data->id != NULL)
	g_free(data->id);
    if (data->username != NULL)
	g_free(data->username);
    if (data->token != NULL)
	g_free(data->token);
    if (data->secret != NULL)
	g_free(data->secret);
    if (data->subdomain != NULL)
	g_free(data->subdomain);

    free(data);
}

/**
 * Find profile in list, contains selected id.
 *
 * @param id - string which profile contains
 * @return found profile entity or NULL
 */
const struct ProfileData *profile_get_by_id(gchar *id)
{
    const struct ProfileData *ret = profile_list;
    while(ret != NULL)
    {
	msa_debug("%s:%d:%s: cur_id=%s, list_id=%s", __FILE__, __LINE__, __FUNCTION__, id, ret->id);
	if (!strcmp(ret->id, id))
	    return ret;
        ret = ret->next;
    }

    return ret;
}

/**
 * update selected profile, using settings from xml
 *
 * @param data - pointer to selected profile
 * @return 0 on success, 1 - otherwise
 */
int profile_update(struct ProfileData *data, xmlDocPtr settings)
{
    msa_debug("profile_update START");
    xmlDocDump(stdout, settings);
    xmlNodePtr node = NULL;

    if (data != NULL){

        node = xpath_get_node("//*/Params/string[@name='username']", settings);
        if (node == NULL)
            return 1;
        if (data->username != NULL)
        {
            g_free(data->username);
            data->username = NULL;
        }
        data->username = (gchar *)xmlNodeGetContent(node);

        node = xpath_get_node("//*/Params/string[@name='token']", settings);
        if (node == NULL)
            return 1;
        if (data->token != NULL)
        {
            g_free(data->token);
            data->token = NULL;
        }
        data->token = (gchar *)xmlNodeGetContent(node);

        node = xpath_get_node("//*/Params/string[@name='secret']", settings);
        if (node == NULL)
            return 1;
        if (data->secret != NULL)
        {
            g_free(data->secret);
            data->secret = NULL;
        }
        data->secret = (gchar *)xmlNodeGetContent(node);

        node = xpath_get_node("//*/Params/string[@name='subdomain']", settings);
        if (node == NULL)
            return 1;
        if (data->subdomain != NULL)
        {
            g_free(data->subdomain);
            data->subdomain = NULL;
        }
        data->subdomain = (gchar *)xmlNodeGetContent(node);

        return 0;
    }

    return 1;
}

/**
 * checks profiles parameters
 *
 * @param data - pointer to selected profile
 * @return 0 on success, 1 - otherwise
 */
int profile_check(const struct ProfileData *data)
{
    if (data == NULL)
        return 1;
    if (data->id == NULL || data->username == NULL || data->token == NULL || data->secret == NULL || data->subdomain == NULL)
        return 1;
    else 
        return 0;
}

/**
 * get profiles parameters
 *
 * @param data - pointer to selected profile 
 */
gchar** profile_get_params(const struct ProfileData *data)
{
    GTimeVal time;
    g_get_current_time(&time);
    gchar *nonce = NULL, *md5_nonce = NULL;

    gchar *tmp[11];
    if(data != NULL){
        nonce = g_strdup_printf("\"%d\"", g_random_int());
        md5_nonce = g_compute_checksum_for_string(G_CHECKSUM_SHA1, nonce, strlen(nonce));
        tmp[0] = g_strdup("USERNAME");
        tmp[1] = g_strdup_printf("\"%s\"", data->username);
        tmp[2] = g_strdup("TOKEN");
        tmp[3] = g_strdup_printf("\"%s\"", data->token);
        tmp[4] = g_strdup("SECRET");
        tmp[5] = g_strdup_printf("\"%s\"", data->secret);
        tmp[6] = g_strdup("TIME");
        tmp[7] = g_strdup_printf("\"%d\"", time.tv_sec - 200);
        tmp[8] = g_strdup("NONCE");
        tmp[9] = g_strdup_printf("\"%s\"", md5_nonce);
        tmp[10] = g_strdup("SUBDOMAIN");
        tmp[11] = g_strdup_printf("\"%s\"", data->subdomain);
        tmp[12] = NULL;
    } else
        return NULL;

    g_free(nonce);
    g_free(md5_nonce);
    gchar **params = g_strdupv(tmp);
    g_free((gpointer*)tmp[0]);
    g_free((gpointer*)tmp[1]);
    g_free((gpointer*)tmp[2]);
    g_free((gpointer*)tmp[3]);
    g_free((gpointer*)tmp[4]);
    g_free((gpointer*)tmp[5]);
    g_free((gpointer*)tmp[6]);
    g_free((gpointer*)tmp[7]);
    g_free((gpointer*)tmp[8]);
    g_free((gpointer*)tmp[9]);
    g_free((gpointer*)tmp[10]);
    g_free((gpointer*)tmp[11]);

    return params;
}

