/*
 *  Microfeed - Backend for accessing feed-based services
 *  Copyright (C) 2009 Henrik Hedberg <henrik.hedberg@innologies.fi>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as published by
 *  the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <string.h>
#include <stdio.h>

#include <microfeed-provider/microfeedprovider.h>
#include <microfeed-common/microfeedmisc.h>
#include <microfeed-common/microfeedstore.h>
#include <microfeed-common/microfeedprotocol.h>

typedef struct _DisconnectedCallback DisconnectedCallback;
typedef struct _Timeout Timeout;

struct _MicrofeedProvider {
	MicrofeedObject object;

	MicrofeedConfiguration* configuration;
	char* bus_name;
	MicrofeedProviderCallbacks callbacks;
	void* user_data;
	DBusConnection* connection;
	
	MicrofeedStore* publishers;
	MicrofeedStore* subscribers;
	MicrofeedThreadPool* thread_pool;
	Timeout* timeouts;
};

struct _DisconnectedCallback {
	DisconnectedCallback* next;
	DisconnectedCallback* previous;
	MicrofeedProviderSubscriberDisconnectedCallback callback;
	void* user_data;
};

typedef struct {
	char* unique_connection_name;
	DisconnectedCallback* disconnected_callbacks;
} Subscriber;

struct _Timeout {
	Timeout* next;
	Timeout* previous;
	MicrofeedProvider* provider;
	unsigned long int interval_seconds;
	MicrofeedProviderTimeoutFunction function;
	MicrofeedWeakReference* weak_reference;
	void* data;
	void* implementation;
};

typedef struct {
	MicrofeedProviderTimeoutFunction function;
	MicrofeedObject* object;
	void* data;
} ThreadData;

static void free_function(MicrofeedProvider* provider);
static const char* subscriber_get_unique_connection_name(Subscriber* subscriber);
static DBusHandlerResult handle_message(DBusConnection* connection, DBusMessage* message, void* user_data);
static void remove_publisher(void* data, void* user_data);
static void start_thread(MicrofeedProvider* provider, MicrofeedProviderTimeoutFunction function, MicrofeedObject* object, void* data);
static void* thread_function(void* data);
static void thread_exited(MicrofeedThread* thread, void* user_data);
static int timeout_handler(void* data);

static MicrofeedClass microfeed_provider_class = {
	"MicrofeedProvider",
	(MicrofeedObjectFreeFunction)free_function
};

/**
 * Instantiates a new provider.
 * 
 * The provider requests the given DBus bus name and adds a DBus message filter to monitor name owner changes.
 * 
 * Usually there is one provider in the provider implementation. If you use #MicrofeedMain as a main loop,
 * you can get the connection with #microfeed_main_get_dbus_connection.
 * 
 * @param bus_name A well-known DBus bus name for the provider.
 * @param configuration Instantiated #MicrofeedConfiguration.
 * @param connection A DBus connection.
 * @param callbacks A callback functions that are needed when a new #MicrofeedPublisher is instantiated.
 * @param user_data A pointer to user data that is needed when a new #MicrofeedPublisher is instantiated.
 * @return Instantiated MicrofeedProvider, or NULL if the registration of the bus name failed.
 */ 
MicrofeedProvider* microfeed_provider_new(const char* bus_name, DBusConnection* connection, MicrofeedProviderCallbacks* callbacks, void* user_data) {
	MicrofeedProvider* provider;
	DBusError error;
	
	provider = microfeed_object_new(&microfeed_provider_class, MicrofeedProvider);

	dbus_error_init(&error);
	if (dbus_bus_request_name(connection, bus_name, 0, &error) == -1) {
		microfeed_memory_free(provider);
		provider = NULL;
	
		return NULL;
	} else if (!dbus_connection_add_filter(connection, handle_message, provider, NULL)) {
		microfeed_memory_free(provider);
		provider = NULL;
		dbus_bus_release_name(connection, bus_name, &error);

		return NULL;
	} else {
		provider->bus_name = strdup(bus_name);
		provider->configuration = microfeed_configuration_new();
		provider->connection = connection;
		provider->callbacks = *callbacks;
		provider->user_data = user_data;	
		provider->publishers = microfeed_store_new_sorted_weak_references((MicrofeedStoreCompareKeysFunction)strcmp,
		                                                                  (MicrofeedStoreGetKeyFunction)microfeed_publisher_get_object_path);
		provider->subscribers = microfeed_store_new_sorted_data((MicrofeedStoreCompareKeysFunction)strcmp,
		                                                        (MicrofeedStoreGetKeyFunction)subscriber_get_unique_connection_name);

		provider->thread_pool = microfeed_thread_pool_new_with_exit_callback(10, NULL, NULL);
	}		

	return provider;
}

/**
 * Frees the resources allocated for the provider.
 * 
 * Removes the DBUs message filter and releases the DBus bus name.
 *
 * @param publisher Instantiated MicrofeedPublisher.
 */
void free_function(MicrofeedProvider* provider) {
	DBusError error;

	dbus_error_init(&error);
	dbus_connection_remove_filter(provider->connection, handle_message, provider);
	dbus_bus_release_name(provider->connection, provider->bus_name, &error);

	free(provider->bus_name);
	microfeed_configuration_free(provider->configuration);
	microfeed_thread_pool_free(provider->thread_pool);
	microfeed_store_free(provider->publishers);
	microfeed_store_free(provider->subscribers);
}

/**
 * Adds a new publisher.
 * 
 * Normally this function is not needed since the provider autimatically instantiates existing
 * publishers when a first message to a non-instantiated publisher is received from a subscriber.
 * 
 * @param provider Instantiated MicrofeedProvider.
 * @param publisher Instantiated #MicrofeedPublisher. 
 */
void microfeed_provider_add_publisher(MicrofeedProvider* provider, MicrofeedPublisher* publisher) {
	microfeed_object_lock(provider, MicrofeedProvider);

	microfeed_store_insert(provider->publishers, publisher);

	microfeed_object_unlock(provider, MicrofeedProvider);
}

/**
 * Removes an existing publisher.
 * 
 * @param provider Instantiated MicrofeedProvider.
 * @param publisher Instantiated #MicrofeedPublisher that has previously added with #microfeed_provider_add_publisher. 
 */
void microfeed_provider_remove_publisher(MicrofeedProvider* provider, MicrofeedPublisher* publisher) {
	microfeed_object_lock(provider, MicrofeedProvider);

	microfeed_store_remove(provider->publishers, publisher);

	microfeed_object_unlock(provider, MicrofeedProvider);
}

/**
 * Returns the DBus connection used by the provider.
 *
 * @param Instantiated MicrofeedProvider.
 * @return DBus connection.
 */
DBusConnection* microfeed_provider_get_dbus_connection(MicrofeedProvider* provider) {

	return provider->connection;
}

/**
 * Return the well-known DBus bus name reserved by the provider.
 * 
 * @param provider Instantiated MicrofeedProvider.
 * @return A DBus bus name.
 */
const char* microfeed_provider_get_bus_name(MicrofeedProvider* provider) {

	return provider->bus_name;
}

void microfeed_provider_add_subscriber_disconnected_callback(MicrofeedProvider* provider, const char* unique_connection_name, MicrofeedProviderSubscriberDisconnectedCallback callback, void* user_data) {
	Subscriber* subscriber;
	char buffer[1024];
	DisconnectedCallback* disconnected_callback;

	microfeed_object_lock(provider, MicrofeedProvider);
	
	if (!(subscriber = microfeed_store_get(provider->subscribers, unique_connection_name, Subscriber))) {
		subscriber = microfeed_memory_allocate(Subscriber);
		subscriber->unique_connection_name = strdup(unique_connection_name);
		microfeed_store_insert(provider->subscribers, subscriber);

		snprintf(buffer, 1024, "type='signal',sender='org.freedesktop.DBus',interface='org.freedesktop.DBus',member='NameOwnerChanged',arg0='%s'", unique_connection_name);
		dbus_bus_add_match(provider->connection, buffer, NULL);
	}
	
	for (disconnected_callback = subscriber->disconnected_callbacks; disconnected_callback; disconnected_callback = disconnected_callback->next) {
		if (disconnected_callback->callback == callback && disconnected_callback->user_data == user_data) {
			break;
		}
	}
	if (!disconnected_callback) {
		disconnected_callback = microfeed_memory_allocate(DisconnectedCallback);
		disconnected_callback->callback = callback;
		disconnected_callback->user_data = user_data;
		
		disconnected_callback->next = subscriber->disconnected_callbacks;
		subscriber->disconnected_callbacks = disconnected_callback;
	}

	microfeed_object_unlock(provider, MicrofeedProvider);
}

void microfeed_provider_remove_subscriber_disconnected_callback(MicrofeedProvider* provider, const char* unique_connection_name, MicrofeedProviderSubscriberDisconnectedCallback callback, void* user_data) {
	Subscriber* subscriber;
	DisconnectedCallback* disconnected_callback;
	char buffer[1024];

	microfeed_object_lock(provider, MicrofeedProvider);
	
	if ((subscriber = microfeed_store_get(provider->subscribers, unique_connection_name, Subscriber))) {
		for (disconnected_callback = subscriber->disconnected_callbacks; disconnected_callback; disconnected_callback = disconnected_callback->next) {
			if (disconnected_callback->callback == callback && disconnected_callback->user_data == user_data) {
				break;
			}
		}
		if (disconnected_callback) {
			if (disconnected_callback->next) {
				disconnected_callback->next->previous = disconnected_callback->previous;
			}
			if (disconnected_callback->previous) {
				disconnected_callback->previous->next = disconnected_callback->next;
			} else {
				subscriber->disconnected_callbacks = disconnected_callback->next;
			}

			microfeed_memory_free(disconnected_callback);
			
			if (!subscriber->disconnected_callbacks) {
				snprintf(buffer, 1024, "type='signal',sender='org.freedesktop.DBus',interface='org.freedesktop.DBus',method='NameOwnerChanged',arg0='%s'", unique_connection_name);
				dbus_bus_remove_match(provider->connection, buffer, NULL);

				microfeed_store_remove(provider->subscribers, subscriber);
				free(subscriber->unique_connection_name);
				microfeed_memory_free(subscriber);
			}
		}
	}

	microfeed_object_unlock(provider, MicrofeedProvider);
}

void microfeed_provider_send_message(MicrofeedProvider* provider, DBusMessage* message) {
	dbus_connection_send(provider->connection, message, NULL);
}

void microfeed_provider_send_item_signal(MicrofeedProvider* provider, const char* destination, const char* signal_name, const char* object_path, const char* uri, MicrofeedItem* item) {
	DBusMessage* message;
	DBusMessageIter iter;
	MicrofeedItemIterator* item_iterator;
	const char* uid;
	dbus_uint64_t timestamp;
	char status;
	const char* key;
	const char* value;

	printf("%s : %s (%s)\n", object_path, signal_name, destination);
	
	message = dbus_message_new_signal(object_path, (destination ? MICROFEED_DBUS_INTERFACE_PUBLISHER_TO_DESTINATION : MICROFEED_DBUS_INTERFACE_PUBLISHER), signal_name);
	if (destination) {
		dbus_message_set_destination(message,destination);
	}
	dbus_message_iter_init_append(message, &iter);
	dbus_message_iter_append_basic(&iter, DBUS_TYPE_STRING, &uri);
	uid = microfeed_item_get_uid(item);
	dbus_message_iter_append_basic(&iter, DBUS_TYPE_STRING, &uid);
	timestamp = (dbus_uint64_t)microfeed_item_get_timestamp(item);
	dbus_message_iter_append_basic(&iter, DBUS_TYPE_UINT64, &timestamp);
	status = (char)microfeed_item_get_status(item);
	dbus_message_iter_append_basic(&iter, DBUS_TYPE_BYTE, &status);
	for (item_iterator = microfeed_item_iterate_properties(item, NULL);
	     microfeed_item_iterator_get(item_iterator, &key, &value);
	     microfeed_item_iterator_next(item_iterator)) {
		dbus_message_iter_append_basic(&iter, DBUS_TYPE_STRING, &key);
		dbus_message_iter_append_basic(&iter, DBUS_TYPE_STRING, &value);
	}
	microfeed_item_iterator_free(item_iterator);
	dbus_connection_send(provider->connection, message, NULL);
	dbus_message_unref(message);	
}

void microfeed_provider_send_item_uid_signal(MicrofeedProvider* provider, const char* destination, const char* signal_name, const char* object_path, const char* uri, const char* uid) {
	DBusMessage* message;

	printf("%s : %s (%s)\n", object_path, signal_name, destination);

	message = dbus_message_new_signal(object_path, (destination ? MICROFEED_DBUS_INTERFACE_PUBLISHER_TO_DESTINATION : MICROFEED_DBUS_INTERFACE_PUBLISHER), signal_name);
	if (destination) {
		dbus_message_set_destination(message, destination);
	}
	dbus_message_append_args(message, DBUS_TYPE_STRING, &uri, DBUS_TYPE_STRING, &uid, DBUS_TYPE_INVALID);
	dbus_connection_send(provider->connection, message, NULL);
	dbus_message_unref(message);
}

void microfeed_provider_send_status_changed_signal(MicrofeedProvider* provider, const char* destination, const char* object_path, const char* uri, const char* uid, const char status) {
	DBusMessage* message;

	printf("%s : %s (%s)\n", object_path, MICROFEED_SIGNAL_NAME_ITEM_STATUS_CHANGED, destination);

	message = dbus_message_new_signal(object_path, (destination ? MICROFEED_DBUS_INTERFACE_PUBLISHER_TO_DESTINATION : MICROFEED_DBUS_INTERFACE_PUBLISHER), MICROFEED_SIGNAL_NAME_ITEM_STATUS_CHANGED);
	if (destination) {
		dbus_message_set_destination(message, destination);
	}
	dbus_message_append_args(message, DBUS_TYPE_STRING, &uri, DBUS_TYPE_STRING, &uid, DBUS_TYPE_BYTE, &status, DBUS_TYPE_INVALID);
	dbus_connection_send(provider->connection, message, NULL);
	dbus_message_unref(message);		
}

void microfeed_provider_send_feed_signal(MicrofeedProvider* provider, const char* destination, const char* signal_name, const char* object_path, const char* uri) {
	DBusMessage* message;

	printf("%s : %s (%s)\n", object_path, signal_name, destination);

	message = dbus_message_new_signal(object_path, (destination ? MICROFEED_DBUS_INTERFACE_PUBLISHER_TO_DESTINATION : MICROFEED_DBUS_INTERFACE_PUBLISHER), signal_name);
	if (destination) {
		dbus_message_set_destination(message, destination);
	}
	dbus_message_append_args(message, DBUS_TYPE_STRING, &uri, DBUS_TYPE_INVALID);
	dbus_connection_send(provider->connection, message, NULL);
	dbus_message_unref(message);
}

void microfeed_provider_send_error_signal(MicrofeedProvider* provider, const char* destination, const char* error_name, const char* object_path, const char* uri, const char* uid, const char* error_message) {
	DBusMessage* message;
	DBusMessageIter iter;
	const char* string;

	printf("%s : %s (%s)\n", object_path, error_name, destination);

	message = dbus_message_new_signal(object_path, (destination ? MICROFEED_DBUS_INTERFACE_ERROR_TO_DESTINATION : MICROFEED_DBUS_INTERFACE_ERROR), error_name);
	if (destination) {
		dbus_message_set_destination(message, destination);
	}
	dbus_message_iter_init_append(message, &iter);
	string = (uri ? uri : "");
	dbus_message_iter_append_basic(&iter, DBUS_TYPE_STRING, &string);
	string = (uid ? uid : "");
	dbus_message_iter_append_basic(&iter, DBUS_TYPE_STRING, &string);
	string = (error_message ? error_message : "");
	dbus_message_iter_append_basic(&iter, DBUS_TYPE_STRING, &string);
	dbus_connection_send(provider->connection, message, NULL);
	dbus_message_unref(message);	
}

void microfeed_provider_send_item_data_signal(MicrofeedProvider* provider, const char* destination, const char* object_path, const char* uri, const char* uid, const void* data, size_t data_size) {
	DBusMessage* message;
	DBusMessageIter iter;
	DBusMessageIter subiter;

	printf("%s : %s (%s)\n", object_path, MICROFEED_SIGNAL_NAME_ITEM_DATA, destination);

	message = dbus_message_new_signal(object_path, (destination ? MICROFEED_DBUS_INTERFACE_PUBLISHER_TO_DESTINATION : MICROFEED_DBUS_INTERFACE_PUBLISHER), MICROFEED_SIGNAL_NAME_ITEM_DATA);
	if (destination) {
		dbus_message_set_destination(message,destination);
	}
	dbus_message_iter_init_append(message, &iter);
	dbus_message_iter_append_basic(&iter, DBUS_TYPE_STRING, &uri);
	dbus_message_iter_append_basic(&iter, DBUS_TYPE_STRING, &uid);
	dbus_message_iter_open_container(&iter, DBUS_TYPE_ARRAY, DBUS_TYPE_BYTE_AS_STRING, &subiter);
	dbus_message_iter_append_fixed_array(&subiter, DBUS_TYPE_BYTE, &data, data_size);
	dbus_message_iter_close_container(&iter, &subiter);
	dbus_connection_send(provider->connection, message, NULL);
	dbus_message_unref(message);
}

void microfeed_provider_add_timeout(MicrofeedProvider* provider, unsigned long int first_seconds, unsigned long int interval_seconds, MicrofeedProviderTimeoutFunction function, MicrofeedWeakReference* weak_reference, void* data) {
	Timeout* timeout;

	microfeed_object_lock(provider, MicrofeedProvider);
	
	timeout = microfeed_memory_allocate(Timeout);
	timeout->provider = provider;
	if (first_seconds != interval_seconds) {
		timeout->interval_seconds = interval_seconds;
	}
	timeout->function = function;
	timeout->weak_reference = weak_reference;
	timeout->data = data;
	timeout->implementation = provider->callbacks.add_timeout(provider, (first_seconds ? first_seconds * 1000 : 1000), timeout_handler, timeout, provider->user_data);
	
	timeout->next = provider->timeouts;
	provider->timeouts = timeout;
	
	microfeed_object_unlock(provider, MicrofeedProvider);
}

void microfeed_provider_remove_timeout(MicrofeedProvider* provider, MicrofeedProviderTimeoutFunction function, MicrofeedWeakReference* weak_reference, void* data) {
	Timeout* timeout;
	
	microfeed_object_lock(provider, MicrofeedProvider);

	for (timeout = provider->timeouts; timeout; timeout = timeout->next) {
		if (timeout->function == function && timeout->weak_reference == weak_reference && timeout->data == data) {
			break;
		}
	}
	if (timeout) {
		provider->callbacks.remove_timeout(provider, timeout->implementation, provider->user_data);
	
		if (timeout->next) {
			timeout->next->previous = timeout->previous;
		}
		if (timeout->previous) {
			timeout->previous->next = timeout->next;
		} else {
			provider->timeouts = timeout->next;
		}

		microfeed_memory_free(timeout);
	}

	microfeed_object_unlock(provider, MicrofeedProvider);
}


static const char* subscriber_get_unique_connection_name(Subscriber* subscriber) {

	return subscriber->unique_connection_name;
}

static DBusHandlerResult handle_message(DBusConnection* connection, DBusMessage* message, void* user_data) {
	DBusHandlerResult result = DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
	MicrofeedProvider* provider;
	const char* name;
	const char* old_owner;
	const char* new_owner;
	Subscriber* subscriber;
	DisconnectedCallback* disconnected_callback;
	DisconnectedCallback dc;
	char buffer[1024];
	MicrofeedWeakReference* weak_reference;
	MicrofeedPublisher* publisher;
	MicrofeedPublisher* new_publisher;
	const char* object_path;
	char* publisher_identifier;
	const char* publisher_directory;
	DBusMessage* reply;
	MicrofeedPublisherCallbacks callbacks;
	
	provider = (MicrofeedProvider*)user_data;

	microfeed_object_ref(provider, MicrofeedProvider);
	microfeed_object_lock(provider, MicrofeedProvider);

	if (dbus_message_is_signal(message, "org.freedesktop.DBus", "NameOwnerChanged")) {
		if (dbus_message_get_args(message, NULL, DBUS_TYPE_STRING, &name, DBUS_TYPE_STRING, &old_owner, DBUS_TYPE_STRING, &new_owner, DBUS_TYPE_INVALID) &&
		    name[0] == ':' && new_owner[0] == 0 && !strcmp(name, old_owner) &&
		    (subscriber = microfeed_store_get(provider->subscribers, name, Subscriber))) {
			for (disconnected_callback = subscriber->disconnected_callbacks; disconnected_callback; disconnected_callback = dc.next) {
	/* TODO: Threading may be an issue. */
				dc = *disconnected_callback;

				microfeed_object_unlock(provider, MicrofeedProvider);

				dc.callback(provider, name, dc.user_data);

				microfeed_object_lock(provider, MicrofeedProvider);

				microfeed_memory_free(disconnected_callback);
			}
			snprintf(buffer, 1024, "type='signal',sender='org.freedesktop.DBus',interface='org.freedesktop.DBus',method='NameOwnerChanged',arg0='%s'", name);
			dbus_bus_remove_match(connection, buffer, NULL);

			microfeed_store_remove(provider->subscribers, subscriber);
			free(subscriber->unique_connection_name);
			microfeed_memory_free(subscriber);

			microfeed_store_clean_weak_references(provider->publishers);
			printf("publishers left: %d (subscriber disconnected)\n", microfeed_store_get_size(provider->publishers));
			if (microfeed_store_get_size(provider->publishers) == 0) {
				if (provider->callbacks.no_more_publishers) {
					provider->callbacks.no_more_publishers(provider, provider->user_data);
				}
			}
		}
	} else if (dbus_message_get_type(message) == DBUS_MESSAGE_TYPE_METHOD_CALL &&
	           (object_path = dbus_message_get_path(message)) &&
		   !strncmp(object_path, MICROFEED_DBUS_OBJECT_PATH_PREFIX_PUBLISHER, strlen(MICROFEED_DBUS_OBJECT_PATH_PREFIX_PUBLISHER))) {
		if ((publisher = microfeed_store_get(provider->publishers, object_path, MicrofeedPublisher))) {
			dbus_message_ref(message);
			start_thread(provider, (MicrofeedProviderTimeoutFunction)microfeed_publisher_handle_message, MICROFEED_OBJECT(publisher), message);
			microfeed_object_unref(publisher, MicrofeedPublisher);
			result = DBUS_HANDLER_RESULT_HANDLED;
		} else {
			publisher_identifier = microfeed_util_string_concatenate(object_path + strlen(MICROFEED_DBUS_OBJECT_PATH_PREFIX_PUBLISHER), MICROFEED_PUBLISHER_IDENTIFIER_SEPARATOR_STRING, provider->bus_name, NULL);
			microfeed_configuration_invalidate(provider->configuration);
			if ((publisher_directory = microfeed_configuration_get_publisher_directory(provider->configuration, publisher_identifier)) ||
			    (dbus_message_is_method_call(message, MICROFEED_DBUS_INTERFACE_PUBLISHER, MICROFEED_METHOD_NAME_CREATE_PUBLISHER) &&
			     provider->callbacks.instantiate_publisher)) {
			     
				microfeed_object_unlock(provider, MicrofeedProvider);

				new_publisher = provider->callbacks.instantiate_publisher(provider, publisher_identifier, publisher_directory, provider->user_data);

				microfeed_object_lock(provider, MicrofeedProvider);

				if (new_publisher) {
					if ((publisher = microfeed_store_get(provider->publishers, object_path, MicrofeedPublisher))) {
						microfeed_object_unref(new_publisher, MicrofeedPublisher);
					} else {
						microfeed_store_insert(provider->publishers, new_publisher);
						publisher = new_publisher;
					}
					dbus_message_ref(message);
					start_thread(provider, (MicrofeedProviderTimeoutFunction)microfeed_publisher_handle_message, MICROFEED_OBJECT(publisher), message);
					microfeed_object_unref(publisher, MicrofeedPublisher);
				} else {
					reply = dbus_message_new_error(message, MICROFEED_DBUS_INTERFACE_ERROR "." MICROFEED_ERROR_UNKNOWN, "Instantiation of a publisher failed.");
					dbus_connection_send(connection, reply, NULL);
					dbus_message_unref(reply);
				}
			} else {
				reply = dbus_message_new_error(message, MICROFEED_DBUS_INTERFACE_ERROR "." MICROFEED_ERROR_NO_SUCH_PUBLISHER, "Publisher does not exist.");
				dbus_connection_send(connection, reply, NULL);
				dbus_message_unref(reply);
			}
			result = DBUS_HANDLER_RESULT_HANDLED;
			free(publisher_identifier);
		}

		microfeed_store_clean_weak_references(provider->publishers);
		printf("publishers left: %d (thread exited)\n", microfeed_store_get_size(provider->publishers));
		if (microfeed_store_get_size(provider->publishers) == 0) {
			if (provider->callbacks.no_more_publishers) {
				provider->callbacks.no_more_publishers(provider, provider->user_data);
			}
		}
	}

	microfeed_object_unlock(provider, MicrofeedProvider);
	microfeed_object_unref(provider, MicrofeedProvider);

	return result;
}

static void start_thread(MicrofeedProvider* provider, MicrofeedProviderTimeoutFunction function, MicrofeedObject* object, void* data) {
	ThreadData* thread_data;

	thread_data = microfeed_memory_allocate(ThreadData);
	thread_data->function = function;
	thread_data->object = microfeed_object_ref_generic(object);
	thread_data->data = data;

	microfeed_thread_unref(microfeed_thread_pool_queue_thread_with_exit_callback(provider->thread_pool, thread_function, thread_data, thread_exited, provider));
}

static void* thread_function(void* data) {
	ThreadData* thread_data;
	
	thread_data = (ThreadData*)data;

	thread_data->function(thread_data->object, thread_data->data);
	microfeed_object_unref_generic(thread_data->object);

	microfeed_memory_free(thread_data);
	
	return NULL;
}

static void thread_exited(MicrofeedThread* thread, void* user_data) {
	MicrofeedProvider* provider;
	
	provider = (MicrofeedProvider*)user_data;

	microfeed_object_ref(provider, MicrofeedProvider);
	microfeed_object_lock(provider, MicrofeedProvider);
	
	microfeed_store_clean_weak_references(provider->publishers);
	printf("publishers left: %d (thread exited)\n", microfeed_store_get_size(provider->publishers));
	if (microfeed_store_get_size(provider->publishers) == 0) {
		if (provider->callbacks.no_more_publishers) {
			provider->callbacks.no_more_publishers(provider, provider->user_data);
		}
	}

	microfeed_object_unlock(provider, MicrofeedProvider);
	microfeed_object_unref(provider, MicrofeedProvider);
}	

static int timeout_handler(void* data) {
	int retvalue = 1;
	Timeout* timeout;
	MicrofeedObject* object;
	ThreadData* thread_data;
	
	timeout = (Timeout*)data;

	microfeed_object_lock(timeout->provider, MicrofeedProvider);

	if ((object = microfeed_weak_reference_get_object_generic(timeout->weak_reference))) {
		if (timeout->interval_seconds) {
			timeout->implementation = timeout->provider->callbacks.add_timeout(timeout->provider, timeout->interval_seconds * 1000, timeout_handler, timeout, timeout->provider->user_data);
			timeout->interval_seconds = 0;
			retvalue = 0;
		}

		printf("Timeout\n");

		start_thread(timeout->provider, timeout->function, object, timeout->data);
		microfeed_object_unref_generic(object);
	} else {
		if (timeout->next) {
			timeout->next->previous = timeout->previous;
		}
		if (timeout->previous) {
			timeout->previous->next = timeout->next;
		} else {
			timeout->provider->timeouts = timeout->next;
		}

		microfeed_memory_free(timeout);

	
		retvalue = 0;
	}

	microfeed_object_unlock(timeout->provider, MicrofeedProvider);

	return retvalue;
}

