/*
 *  Microfeed - Backend for accessing feed-based services
 *  Copyright (C) 2009 Henrik Hedberg <henrik.hedberg@innologies.fi>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as published by
 *  the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifndef MICROFEEDDATABASE_H
#define MICROFEEDDATABASE_H

#include <sys/types.h>

/**
 * @addtogroup provider libmicrofeed-provider - Modules that are used in the provider side
 * @{
 * @addtogroup MicrofeedDatabase
 *
 * A database layer.
 *
 * This module is used internally by @link MicrofeedFeed MicrofeedFeed@endlink. It is not accessed directly
 * when developing a provider. However, it can be also utilized outside of the library implementation.
 *
 * @{
 */

/**
 * Opaque data type representing a database environment which contains actual databases.
 */
typedef struct _MicrofeedDatabaseEnvironment MicrofeedDatabaseEnvironment;

/**
 * Opaque data type representing a database.
 */
typedef struct _MicrofeedDatabase MicrofeedDatabase;

/**
 * Opaque data type representing a secondary index of a database.
 */
typedef struct _MicrofeedDatabaseIndex MicrofeedDatabaseIndex;

/**
 * Opaque data type representing database iterator that iterates over the keys of the database.
 */
typedef struct _MicrofeedDatabaseIterator MicrofeedDatabaseIterator;

/**
 * This function is called when a new item is added into a database.
 * 
 * The function must extract the index key from the given primary key and data and return it using the
 * index_key and index_key_size parameters.
 * 
 * @param key The primary key of the added item.
 * @param key_size The size of the key.
 * @param data The data of the added item.
 * @param data_size The size of the data.
 * @param index_key [return] The index key of the item.
 * @param index_key_size [return] The size of the index key.
 */
typedef void (*MicrofeedDatabaseIndexFunction)(void* key, const size_t key_size, void* data, const size_t data_size, void** index_key, size_t* index_key_size);

/**
 * This function is called when two keys in a database are compared by the database engine.
 * 
 * The function must return negative integer if key1 is less than key2, zero if key1 is equal than key2, and
 * positive integer if key1 is greater than key2.
 * 
 * @param key1 The first key to compare.
 * @param key1_size The size of the first key.
 * @param key2 The second key to compare.
 * @param key2_size The size of the second key.
 * @return Negative integer, zero or positive integer depending of the order of the keys.
 */
typedef int (*MicrofeedDatabaseCompareFunction)(const void* key1, const size_t key1_size, const void* key2, const size_t key2_size);

MicrofeedDatabaseEnvironment* microfeed_database_environment_new(const char* name, const char* directory);
MicrofeedDatabase* microfeed_database_environment_get_database(MicrofeedDatabaseEnvironment* database_environment, const char* name, MicrofeedDatabaseCompareFunction);

MicrofeedDatabaseIndex* microfeed_database_get_index(MicrofeedDatabase* database, const char* name, MicrofeedDatabaseIndexFunction index_function);
const char* microfeed_database_get_name(MicrofeedDatabase* database);
int microfeed_database_get_data(MicrofeedDatabase* database, const void* key, size_t key_size, void** data, size_t* data_size);
MicrofeedDatabaseIterator* microfeed_database_iterate(MicrofeedDatabase* database, const void* start_key, size_t start_key_size, int backwards);
void microfeed_database_replace_data(MicrofeedDatabase* database, const void* key, size_t key_size, const void* data, size_t data_size);
void microfeed_database_replace_data_partial(MicrofeedDatabase* database, const void* key, size_t key_size, const void* data, size_t data_size, size_t offset);
void microfeed_database_remove_data(MicrofeedDatabase* database, const void* key, const size_t key_size);
void microfeed_database_remove_data_range(MicrofeedDatabase* database, const void* start_key, size_t start_key_size, const void* end_key, size_t end_key_size);

MicrofeedDatabase* microfeed_database_index_get_database(MicrofeedDatabaseIndex* database_index);
const char* microfeed_database_index_get_name(MicrofeedDatabaseIndex* database_index);
int microfeed_database_index_get_data(MicrofeedDatabaseIndex* database_index, const void* index_key, size_t index_key_size, void** key, size_t* key_size, void** data, size_t* data_size);
int microfeed_database_get_data_partial(MicrofeedDatabase* database, const void* key, const size_t key_size, void* data, size_t* data_size, size_t offset);
MicrofeedDatabaseIterator* microfeed_database_index_iterate(MicrofeedDatabaseIndex* database_index, const void* start_key, const size_t start_key_size, int backwards);
void microfeed_database_index_remove_data(MicrofeedDatabaseIndex* database_index, const void* key, size_t key_size);
void microfeed_database_index_remove_data_range(MicrofeedDatabaseIndex* database_index, const void* start_key, const size_t start_key_size, const void* end_key, const size_t end_key_size);

void microfeed_database_iterator_free(MicrofeedDatabaseIterator* iterator);
int microfeed_database_iterator_get(MicrofeedDatabaseIterator* iterator, const void** key, size_t* key_size, const void** data, size_t* data_size);
void microfeed_database_iterator_next(MicrofeedDatabaseIterator* iterator);

/**
 * @}
 * @}
 */

#endif
