/*
 *  Microfeed - Backend for accessing feed-based services
 *  Copyright (C) 2009 Henrik Hedberg <henrik.hedberg@innologies.fi>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as published by
 *  the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifndef MICROFEEDPROVIDER_H
#define MICROFEEDPROVIDER_H

#define DBUS_API_SUBJECT_TO_CHANGE
#include <dbus/dbus.h>

/**
 * @addtogroup provider libmicrofeed-provider - Modules that are used in the provider side
 * @{
 * @addtogroup MicrofeedProvider
 * 
 * A providers connection to the DBus system.
 *
 * Provider instantiates existing publishers as requested by subscribers. In addition, a new or existing
 * publisher can also be instantiated using #microfeed_publisher_new function and then added into Provider
 * with the #microfeed_provider_add_publisher function. Provider also monitors name changes in the DBus and
 * informs its publishers by calling #microfeed_publisher_remove_subscriber when a potential subscriber has
 * terminated.
 * 
 * @{
 */

/**
 * An opaque datatype presenting a provider.
 */
typedef struct _MicrofeedProvider MicrofeedProvider;

/** @} */

#include <microfeed-provider/microfeedpublisher.h>
#include <microfeed-common/microfeedconfiguration.h>

/** 
 * @addtogroup MicrofeedProvider
 * @{
 */

typedef void (*MicrofeedProviderNoMorePublishersCallback)(MicrofeedProvider* provider, void* user_data);

typedef struct _MicrofeedProviderCallbacks {
	MicrofeedProviderNoMorePublishersCallback no_more_publishers;
	MicrofeedPublisherCallbacks publisher_callbacks;
} MicrofeedProviderCallbacks;

MicrofeedProvider* microfeed_provider_new(const char* bus_name, DBusConnection* connection, MicrofeedProviderCallbacks* callbacks, void* user_data);
void microfeed_provider_free(MicrofeedProvider* provider);
void microfeed_provider_add_publisher(MicrofeedProvider* provider, MicrofeedPublisher* publisher);
void microfeed_provider_remove_publisher(MicrofeedProvider* provider, MicrofeedPublisher* publisher);
DBusConnection* microfeed_provider_get_dbus_connection(MicrofeedProvider* provider);
const char* microfeed_provider_get_bus_name(MicrofeedProvider* provider);

/**
 * @}
 * @}
 */

#endif
