#include <string.h>

#include <microfeed-provider/microfeedprovider.h>
#include <microfeed-common/microfeedmisc.h>
#include <microfeed-common/microfeedstore.h>
#include <microfeed-common/microfeedprotocol.h>

struct _MicrofeedProvider {
	MicrofeedConfiguration* configuration;
	char* bus_name;
	MicrofeedProviderCallbacks callbacks;
	void* user_data;
	DBusConnection* connection;
	
	MicrofeedStore* publishers;
};

static DBusHandlerResult handle_message(DBusConnection* connection, DBusMessage* message, void* user_data);
static void no_more_subscribers(MicrofeedPublisher* publisher, void* user_data);

/**
 * Instantiates a new provider.
 * 
 * The provider requests the given DBus bus name and adds a DBus message filter to monitor name owner changes.
 * 
 * Usually there is one provider in the provider implementation. If you use #MicrofeedMain as a main loop,
 * you can get the connection with #microfeed_main_get_dbus_connection.
 * 
 * @param bus_name A well-known DBus bus name for the provider.
 * @param configuration Instantiated #MicrofeedConfiguration.
 * @param connection A DBus connection.
 * @param callbacks A callback functions that are needed when a new #MicrofeedPublisher is instantiated.
 * @param user_data A pointer to user data that is needed when a new #MicrofeedPublisher is instantiated.
 * @return Instantiated MicrofeedProvider, or NULL if the registration of the bus name failed.
 */ 
MicrofeedProvider* microfeed_provider_new(const char* bus_name, MicrofeedConfiguration* configuration, DBusConnection* connection, MicrofeedProviderCallbacks* callbacks, void* user_data) {
	MicrofeedProvider* provider;
	DBusError error;
	
	provider = microfeed_memory_allocate(MicrofeedProvider);

	dbus_error_init(&error);
	if (dbus_bus_request_name(connection, bus_name, 0, &error) == -1) {
		microfeed_memory_free(provider);
		provider = NULL;
	
		return NULL;
	} else if (!dbus_connection_add_filter(connection, handle_message, provider, NULL)) {
		microfeed_memory_free(provider);
		provider = NULL;
		dbus_bus_release_name(connection, bus_name, &error);

		return NULL;
	} else {
		provider->bus_name = strdup(bus_name);
		provider->configuration = configuration;
		provider->connection = connection;
		provider->callbacks = *callbacks;
		provider->user_data = user_data;	
		provider->publishers = microfeed_store_new_sorted((MicrofeedStoreCompareKeysFunction)strcmp,
		                                           (MicrofeedStoreGetKeyFunction)microfeed_publisher_get_object_path);
	}		

	return provider;
}

/**
 * Frees the resources allocated for the provider.
 * 
 * Removes the DBUs message filter and releases the DBus bus name.
 *
 * @param publisher Instantiated MicrofeedPublisher.
 */
void microfeed_provider_free(MicrofeedProvider* provider) {
	DBusError error;

	dbus_error_init(&error);
	dbus_connection_remove_filter(provider->connection, handle_message, provider);
	dbus_bus_release_name(provider->connection, provider->bus_name, &error);

	free(provider->bus_name);
	provider->connection = NULL;
	provider->bus_name = NULL;
	microfeed_memory_free(provider);
}

/**
 * Adds a new publisher.
 * 
 * Normally this function is not needed since the provider autimatically instantiates existing
 * publishers when a first message to a non-instantiated publisher is received from a subscriber.
 * 
 * @param provider Instantiated MicrofeedProvider.
 * @param publisher Instantiated #MicrofeedPublisher. 
 */
void microfeed_provider_add_publisher(MicrofeedProvider* provider, MicrofeedPublisher* publisher) {
	microfeed_store_insert(provider->publishers, publisher);
}

/**
 * Removes an existing publisher.
 * 
 * @param provider Instantiated MicrofeedProvider.
 * @param publisher Instantiated #MicrofeedPublisher that has previously added with #microfeed_provider_add_publisher. 
 */
void microfeed_provider_remove_publisher(MicrofeedProvider* provider, MicrofeedPublisher* publisher) {
	microfeed_store_remove(provider->publishers, publisher);
}

/**
 * Returns the DBus connection used by the provider.
 *
 * @param Instantiated MicrofeedProvider.
 * @return DBus connection.
 */
DBusConnection* microfeed_provider_get_dbus_connection(MicrofeedProvider* provider) {

	return provider->connection;
}

/**
 * Return the well-known DBus bus name reserved by the provider.
 * 
 * @param provider Instantiated MicrofeedProvider.
 * @return A DBus bus name.
 */
const char* microfeed_provider_get_bus_name(MicrofeedProvider* provider) {

	return provider->bus_name;
}

static DBusHandlerResult handle_message(DBusConnection* connection, DBusMessage* message, void* user_data) {
	DBusHandlerResult result = DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
	MicrofeedProvider* provider;
	const char* name;
	const char* old_owner;
	const char* new_owner;
	MicrofeedStoreIterator* iterator;
	MicrofeedPublisher* publisher;
	const char* object_path;
	char* publisher_identifier;
	const char* publisher_directory;
	DBusMessage* reply;
	MicrofeedPublisherCallbacks callbacks;
	
	provider = (MicrofeedProvider*)user_data;
	if (dbus_message_is_signal(message, "org.freedesktop.DBus", "NameOwnerChanged") &&
	    dbus_message_get_args(message, NULL, DBUS_TYPE_STRING, &name, DBUS_TYPE_STRING, &old_owner, DBUS_TYPE_STRING, &new_owner, DBUS_TYPE_INVALID) &&
	    name[0] == ':' && new_owner[0] == 0 && !strcmp(name, old_owner)) {
		for (iterator = microfeed_store_iterate(provider->publishers, NULL);
		     (publisher = microfeed_store_iterator_get(iterator, MicrofeedPublisher));
		     microfeed_store_iterator_next(iterator)) {
			microfeed_publisher_remove_subscriber(publisher, name);
		}
	}
	object_path = dbus_message_get_path(message);
	if (object_path && !(publisher = microfeed_store_get(provider->publishers, object_path, MicrofeedPublisher)) &&
	    !strncmp(object_path, MICROFEED_DBUS_OBJECT_PATH_PREFIX_PUBLISHER, strlen(MICROFEED_DBUS_OBJECT_PATH_PREFIX_PUBLISHER))) {
		publisher_identifier = microfeed_util_string_concatenate(object_path + strlen(MICROFEED_DBUS_OBJECT_PATH_PREFIX_PUBLISHER), ":", provider->bus_name, NULL);
		if ((publisher_directory = microfeed_configuration_get_publisher_directory(provider->configuration, publisher_identifier)) ||
		    (dbus_message_is_method_call(message, MICROFEED_DBUS_INTERFACE_PUBLISHER, MICROFEED_METHOD_NAME_CREATE_PUBLISHER) &&
		     provider->callbacks.publisher_callbacks.initialize_settings &&
			 (publisher_directory = microfeed_configuration_get_default_publisher_directory(provider->configuration)))) {
			publisher_identifier = microfeed_util_string_concatenate(object_path + strlen(MICROFEED_DBUS_OBJECT_PATH_PREFIX_PUBLISHER), ":", provider->bus_name, NULL);
			callbacks = provider->callbacks.publisher_callbacks;
			callbacks.no_more_subscribers = no_more_subscribers;
			publisher = microfeed_publisher_new(publisher_identifier, publisher_directory, provider->connection, &callbacks, provider->user_data);
			microfeed_publisher_set_provider(publisher, provider);
			microfeed_provider_add_publisher(provider, publisher);
			free(publisher_identifier);
		} else {
			reply = dbus_message_new_error(message, MICROFEED_DBUS_INTERFACE_ERROR "." MICROFEED_ERROR_NO_SUCH_PUBLISHER, "Publisher does not exist.");
			dbus_connection_send(connection, reply, NULL);
			dbus_message_unref(reply);
			result = DBUS_HANDLER_RESULT_HANDLED;
		}
	}
	
	return result;
}

static void no_more_subscribers(MicrofeedPublisher* publisher, void* user_data) {
	MicrofeedProvider* provider;
	
	provider = microfeed_publisher_get_provider(publisher);
	if (provider->callbacks.publisher_callbacks.no_more_subscribers) {
		provider->callbacks.publisher_callbacks.no_more_subscribers(publisher, user_data);
	}
	microfeed_store_remove(provider->publishers, publisher);
	microfeed_publisher_free(publisher);
	if (microfeed_store_get_size(provider->publishers) == 0) {
		provider->callbacks.no_more_publishers(provider, provider->user_data);
	}
}

