
#define _GNU_SOURCE

#include <microfeed-provider/microfeedjson.h>
#include <microfeed-common/microfeedstore.h>
#include <microfeed-common/microfeedmisc.h>
#include <unistd.h>
#include <string.h>
#include <stdio.h>
#include <stdarg.h>

struct _MicrofeedJson {
	MicrofeedJson* parent;
	MicrofeedStore* members;
};

typedef struct {
	char* name;
	MicrofeedJsonType type;
	union {
		MicrofeedJson* object;
		MicrofeedJson* array;
		char* string;
		int integer;
		double decimal;
		int boolean;
	};
	char* as_string;
} Member;

typedef struct {
	MicrofeedJson* json;
	MicrofeedJsonCompareMembersFunction compare_members;
	void* user_data;
} SortData;

static Member* member_new(char* name, MicrofeedJsonType type);
static void member_free(Member* member);
static const char* member_get_name(Member* member);
static const char* member_as_string(Member* member);
static char* member_to_string(Member* member);
static Member* parse_value(char* name, const char** data, const char* end);
static MicrofeedJson* parse_object(const char** data, const char* end);
static MicrofeedJson* parse_array(const char** data, const char* end);
static char* parse_string(const char** data, const char* end);
static Member* parse_number(char* name, const char** data, const char* end);
static void eat_whitespaces(const char** data, const char* end);
static int are_next_characters(const char* next, size_t next_length, const char** data, const char* end);
static int hex_to_int(char hex, unsigned int* int_return);
int compare_json_members_in_sort(const void* data1, const void* data2, unsigned int index1, unsigned int index2, void* user_data);

MicrofeedJson* microfeed_json_new_object(void) {
	MicrofeedJson* json;
	
	json = microfeed_memory_allocate(MicrofeedJson);
	json->members = microfeed_store_new_sorted((MicrofeedStoreCompareKeysFunction)strcmp, (MicrofeedStoreGetKeyFunction)member_get_name);
	
	return json;
}

MicrofeedJson* microfeed_json_new_array(void) {
	MicrofeedJson* json;
	
	json = microfeed_memory_allocate(MicrofeedJson);
	json->members = microfeed_store_new_unsorted((MicrofeedStoreCompareKeysFunction)strcmp, (MicrofeedStoreGetKeyFunction)member_get_name);
	
	return json;
}

MicrofeedJson* microfeed_json_new_from_data(const char* data, size_t length) {
	MicrofeedJson* json = NULL;
	Member* member;
	
	if ((member = parse_value(NULL, &data, data + length))) {
		if (member->type == MICROFEED_JSON_TYPE_OBJECT) {
			json = member->object;
			member->object = NULL;
		} else if (member->type == MICROFEED_JSON_TYPE_ARRAY) {
			json = member->array;
			member->array = NULL;
		}
		member_free(member);
	} else {
		printf("JSON parsing failed: %s\n", data);
	}

	return json;
}

void microfeed_json_free(MicrofeedJson* json) {
	if (json) {
		if (json->parent) {
			/* TODO */		
		}
		microfeed_store_foreach(json->members, (MicrofeedStoreForeachFunction)member_free, NULL);
		microfeed_store_free(json->members);
		microfeed_memory_free(json);
	}
}

int microfeed_json_is_array(MicrofeedJson* json) {
	
	return !microfeed_store_is_sorted(json->members);
}

MicrofeedJson* microfeed_json_get_parent(MicrofeedJson* json) {
	
	return json->parent;
}

unsigned int microfeed_json_get_size(MicrofeedJson* json) {
	
	return microfeed_store_get_size(json->members);
}

MicrofeedJsonType microfeed_json_get_type(MicrofeedJson* json, const char* name) {
	MicrofeedJsonType type = MICROFEED_JSON_TYPE_NULL;
	Member* member;
	
	if (name) {
		if ((member = microfeed_store_get(json->members, name, Member))) {
			type = member->type;
		}
	} else {
		type = (microfeed_store_is_sorted(json->members) ? MICROFEED_JSON_TYPE_OBJECT : MICROFEED_JSON_TYPE_ARRAY);
	}
	
	return type;
}
	
MicrofeedJsonType microfeed_json_get_type_by_index(MicrofeedJson* json, unsigned int index) {
	MicrofeedJsonType type = MICROFEED_JSON_TYPE_NULL;
	Member* member;
	
	if ((member = microfeed_store_get_index(json->members, index, Member))) {
		type = member->type;
	}
	
	return type;
}

MicrofeedJsonType microfeed_json_get_type_by_path(MicrofeedJson* json, const char* name, ...) {
	MicrofeedJsonType type = MICROFEED_JSON_TYPE_NULL;
	Member* member;
	va_list argp;
	const char* s;
	
	if ((member = microfeed_store_get(json->members, name, Member))) {
		va_start(argp, name);
		while ((s = va_arg(argp, const char*)) && member->type == MICROFEED_JSON_TYPE_OBJECT &&
		       (member = microfeed_store_get(member->object->members, s, Member))) {
		}
		va_end(argp);
		if (!s) {
			type = member->type;
		}
	}
		
	return type;
}

int microfeed_json_is_null(MicrofeedJson* json, const char* name) {
	int retvalue = 1;
	Member* member;
	
	if (name) {
		if ((member = microfeed_store_get(json->members, name, Member))) {
			retvalue = (member->type == MICROFEED_JSON_TYPE_NULL);
		}
	} else {
		retvalue = 0;
	}
	
	return retvalue;
}	
	
const char* microfeed_json_get_name_by_index(MicrofeedJson* json, unsigned int index) {
	const char* name = NULL;
	Member* member;
	
	if ((member = microfeed_store_get_index(json->members, index, Member))) {
		name = member->name;
	}
	
	return name;	
}

int microfeed_json_is_null_by_index(MicrofeedJson* json, unsigned int index) {
	int retvalue = 1;
	Member* member;
	
	if ((member = microfeed_store_get_index(json->members, index, Member))) {
		retvalue = (member->type == MICROFEED_JSON_TYPE_NULL);
	}
	
	return retvalue;	
}

int microfeed_json_is_null_by_path(MicrofeedJson* json,const char* name, ...) {
	int retvalue = 1;
	Member* member;
	va_list argp;
	const char* s;
	
	if ((member = microfeed_store_get(json->members, name, Member))) {
		va_start(argp, name);
		while ((s = va_arg(argp, const char*)) && member->type == MICROFEED_JSON_TYPE_OBJECT &&
		       (member = microfeed_store_get(member->object->members, s, Member))) {
		}
		va_end(argp);
		if (!s) {
			retvalue = (member->type == MICROFEED_JSON_TYPE_NULL);
		}
	}
		
	return retvalue;
}

MicrofeedJson* microfeed_json_get_object(MicrofeedJson* json, const char* name) {
	MicrofeedJson* object = NULL;
	Member* member;
	
	if (name) {
		if ((member = microfeed_store_get(json->members, name, Member)) && member->type == MICROFEED_JSON_TYPE_OBJECT) {
			object = member->object;
		}
	} else {
		if (microfeed_store_is_sorted(json->members)) {
			object = json;
		}
	}
	
	return object;
}

MicrofeedJson* microfeed_json_get_object_by_index(MicrofeedJson* json, unsigned int index) {
	MicrofeedJson* object = NULL;
	Member* member;
	
	if ((member = microfeed_store_get_index(json->members, index, Member)) && member->type == MICROFEED_JSON_TYPE_OBJECT) {
		object = member->object;
	}
	
	return object;
}

MicrofeedJson* microfeed_json_get_object_by_path(MicrofeedJson* json, const char* name, ...) {
	MicrofeedJson* object = NULL;
	Member* member;
	va_list argp;
	const char* s;
	
	if ((member = microfeed_store_get(json->members, name, Member))) {
		va_start(argp, name);
		while ((s = va_arg(argp, const char*)) && member->type == MICROFEED_JSON_TYPE_OBJECT &&
		       (member = microfeed_store_get(member->object->members, s, Member))) {
		}
		va_end(argp);
		if (!s && member->type == MICROFEED_JSON_TYPE_OBJECT) {
			object = member->object;
		}
	}
		
	return object;
}

MicrofeedJson* microfeed_json_get_array(MicrofeedJson* json, const char* name) {
	MicrofeedJson* array = NULL;
	Member* member;
	
	if (name) {
		if ((member = microfeed_store_get(json->members, name, Member)) && member->type == MICROFEED_JSON_TYPE_ARRAY) {
			array = member->array;
		}
	} else {
		if (!microfeed_store_is_sorted(json->members)) {
			array = json;
		}
	}
	
	return array;
}

MicrofeedJson* microfeed_json_get_array_by_index(MicrofeedJson* json, unsigned int index) {
	MicrofeedJson* array = NULL;
	Member* member;
	
	if ((member = microfeed_store_get_index(json->members, index, Member)) && member->type == MICROFEED_JSON_TYPE_ARRAY) {
		array = member->array;
	}
	
	return array;
}

MicrofeedJson* microfeed_json_get_array_by_path(MicrofeedJson* json, const char* name, ...) {
	MicrofeedJson* array = NULL;
	Member* member;
	va_list argp;
	const char* s;
	
	if ((member = microfeed_store_get(json->members, name, Member))) {
		va_start(argp, name);
		while ((s = va_arg(argp, const char*)) && member->type == MICROFEED_JSON_TYPE_OBJECT &&
		       (member = microfeed_store_get(member->object->members, s, Member))) {
		}
		va_end(argp);
		if (!s && member->type == MICROFEED_JSON_TYPE_ARRAY) {
			array = member->array;
		}
	}
		
	return array;
}

const char* microfeed_json_get_string(MicrofeedJson* json, const char* name) {
	const char* string = NULL;
	Member* member;
	
	if (name) {
		if ((member = microfeed_store_get(json->members, name, Member)) && member->type == MICROFEED_JSON_TYPE_STRING) {
			string = member->string;
		}
	}
	
	return string;
}
	
const char* microfeed_json_get_string_by_index(MicrofeedJson* json, unsigned int index) {
	const char* string = NULL;
	Member* member;
	
	if ((member = microfeed_store_get_index(json->members, index, Member)) && member->type == MICROFEED_JSON_TYPE_STRING) {
		string = member->string;
	}
	
	return string;
}

const char* microfeed_json_get_string_by_path(MicrofeedJson* json, const char* name, ...) {
	const char* string = NULL;
	Member* member;
	va_list argp;
	const char* s;
	
	if ((member = microfeed_store_get(json->members, name, Member))) {
		va_start(argp, name);
		while ((s = va_arg(argp, const char*)) && member->type == MICROFEED_JSON_TYPE_OBJECT &&
		       (member = microfeed_store_get(member->object->members, s, Member))) {
		}
		va_end(argp);
		if (!s && member->type == MICROFEED_JSON_TYPE_STRING) {
			string = member->string;
		}
	}
		
	return string;
}

int microfeed_json_get_boolean(MicrofeedJson* json, const char* name) {
	int boolean = 0;
	Member* member;
	
	if ((member = microfeed_store_get(json->members, name, Member)) && member->type == MICROFEED_JSON_TYPE_BOOLEAN) {
		boolean = member->boolean;
	}
	
	return boolean;
}
	
int microfeed_json_get_boolean_by_index(MicrofeedJson* json, unsigned int index) {
	int boolean = 0;
	Member* member;
	
	if ((member = microfeed_store_get_index(json->members, index, Member)) && member->type == MICROFEED_JSON_TYPE_BOOLEAN) {
		boolean = member->boolean;
	}
	
	return boolean;
}

int microfeed_json_get_boolean_by_path(MicrofeedJson* json, const char* name, ...) {
	int boolean = 0;
	Member* member;
	va_list argp;
	const char* s;
	
	if ((member = microfeed_store_get(json->members, name, Member))) {
		va_start(argp, name);
		while ((s = va_arg(argp, const char*)) && member->type == MICROFEED_JSON_TYPE_OBJECT &&
		       (member = microfeed_store_get(member->object->members, s, Member))) {
		}
		va_end(argp);
		if (!s && member->type == MICROFEED_JSON_TYPE_BOOLEAN) {
			boolean = member->boolean;
		}
	}
		
	return boolean;	
}

const char* microfeed_json_get_as_string(MicrofeedJson* json, const char* name) {
	const char* string = NULL;
	Member* member;
	
	if (name) {
		if ((member = microfeed_store_get(json->members, name, Member))) {
			string = member_as_string(member);
		}
	}
	
	return string;
}
	
const char* microfeed_json_get_as_string_by_index(MicrofeedJson* json, unsigned int index) {
	const char* string = NULL;
	Member* member;
	
	if ((member = microfeed_store_get_index(json->members, index, Member))) {
		string = member_as_string(member);
	}
	
	return string;
}

const char* microfeed_json_get_as_string_by_path(MicrofeedJson* json, const char* name, ...) {
	const char* string = NULL;
	Member* member;
	va_list argp;
	const char* s;
	
	if ((member = microfeed_store_get(json->members, name, Member))) {
		va_start(argp, name);
		while ((s = va_arg(argp, const char*)) && member->type == MICROFEED_JSON_TYPE_OBJECT &&
		       (member = microfeed_store_get(member->object->members, s, Member))) {
		}
		va_end(argp);
		if (!s) {
			string = member_as_string(member);
		}
	}
		
	return string;
}

char* microfeed_json_to_string(MicrofeedJson* json, const char* name) {
	char* string = NULL;
	Member* member;
	size_t size;
	unsigned int i;
	MicrofeedStoreIterator* iterator;
	size_t length1;
	size_t length2;
	char* s;
	
	if (name) {
		if ((member = microfeed_store_get(json->members, name, Member))) {
			string = member_to_string(member);
		} else {
			string = strdup("null");
		}
	} else {
		size = 2;
		string = (char*)malloc(size + 1);
		i = 1;
		string[i] = ' ';
		for (iterator = microfeed_store_iterate(json->members, NULL);
		     (member = microfeed_store_iterator_get(iterator, Member));
		     microfeed_store_iterator_next(iterator)) {
			if (member->name) {
				length1 = strlen(member->name);
				size += length1 + 3;
			} else {
				length1 = 0;
			}
			s = member_to_string(member);
			length2 = strlen(s);
			size += length2 + 1;
			string = (char*)realloc(string, size + 1);
			if (length1) {
				string[i++] = '"';
				memcpy(string + i, member->name, length1);
				i += length1;
				string[i++] = '"';
				string[i++] = ':';
			}
			memcpy(string + i, s, length2);
			i += length2;
			string[i++] = ',';
			free(s);
			/* TODO: After last entry, there must not be a comma */
		}
		microfeed_store_iterator_free(iterator);
		if (microfeed_store_is_sorted(json->members)) {
			string[0] = '{';
			string[size - 1] = '}';
		} else {
			string[0] = '[';
			string[size - 1] = ']';			
		}
		string[size] = 0;
	}	

	return string;
}

char* microfeed_json_to_string_by_index(MicrofeedJson* json, unsigned int index) {
	char* string = NULL;
	Member* member;
	
	if ((member = microfeed_store_get_index(json->members, index, Member))) {
		string = member_to_string(member);
	} else {
		string = strdup("null");
	}

	return string;
}
	
char* microfeed_json_to_string_by_path(MicrofeedJson* json, const char* name, ...);

void microfeed_json_set_null(MicrofeedJson* json, const char* name) {
	Member* member;

	member = member_new(strdup(name), MICROFEED_JSON_TYPE_NULL);
	if ((member = microfeed_store_replace(json->members, member, Member))) {
		member_free(member);
	}
}

void microfeed_json_append_null(MicrofeedJson* json, MicrofeedJson* object) {
	Member* member;

	member = member_new(NULL, MICROFEED_JSON_TYPE_NULL);
	microfeed_store_insert(json->members, member);
}

void microfeed_json_set_object(MicrofeedJson* json, const char* name, MicrofeedJson* object) {
	Member* member;

	member = member_new(strdup(name), MICROFEED_JSON_TYPE_OBJECT);
	member->object = object;
	if ((member = microfeed_store_replace(json->members, member, Member))) {
		member_free(member);
	}
}
	
void microfeed_json_append_object(MicrofeedJson* json, MicrofeedJson* object) {
	Member* member;

	member = member_new(NULL, MICROFEED_JSON_TYPE_OBJECT);
	member->object = object;
	microfeed_store_insert(json->members, member);
}

void microfeed_json_set_string(MicrofeedJson* json, const char* name, const char* string) {
	Member* member;

	member = member_new(strdup(name), MICROFEED_JSON_TYPE_STRING);
	member->string = strdup(string);
	if ((member = microfeed_store_replace(json->members, member, Member))) {
		member_free(member);
	}
}

void microfeed_json_append_string(MicrofeedJson* json, const char* string) {
	Member* member;

	member = member_new(NULL, MICROFEED_JSON_TYPE_STRING);
	member->string = strdup(string);
	microfeed_store_insert(json->members, member);
}
	
void microfeed_json_set_boolean(MicrofeedJson* json, const char* name, int boolean) {
	Member* member;

	member = member_new(strdup(name), MICROFEED_JSON_TYPE_BOOLEAN);
	member->boolean = boolean;
	if ((member = microfeed_store_replace(json->members, member, Member))) {
		member_free(member);
	}
}

void microfeed_json_append_boolean(MicrofeedJson* json, int boolean) {
	Member* member;

	member = member_new(NULL, MICROFEED_JSON_TYPE_BOOLEAN);
	member->boolean = boolean;
	microfeed_store_insert(json->members, member);
}


void microfeed_json_sort_array(MicrofeedJson* json, MicrofeedJsonCompareMembersFunction compare_members, void* user_data) {
	SortData* sort_data;
	
	sort_data = microfeed_memory_allocate(SortData);
	sort_data->json = json;
	sort_data->compare_members = compare_members;
	sort_data->user_data = user_data;
	microfeed_store_sort(json->members, compare_json_members_in_sort, sort_data);
	microfeed_memory_free(sort_data);
}

int microfeed_json_compare_members(MicrofeedJson* json, unsigned int index1, unsigned int index2) {
	int retvalue = 0;
	Member* member1;
	Member* member2;
	
	if (json &&
	    (member1 = microfeed_store_get_index(json->members, index1, Member)) &&
	    (member2 = microfeed_store_get_index(json->members, index2, Member)) &&
	    member1->type == member2->type) {
		switch (member1->type) {
			case MICROFEED_JSON_TYPE_INTEGER:
				retvalue = member1->integer - member2->integer;
				break;
			case MICROFEED_JSON_TYPE_DECIMAL:
				retvalue = member1->decimal - member2->decimal;
				break;
			case MICROFEED_JSON_TYPE_STRING:
				retvalue = strcmp(member1->string, member2->string);
				break;
			case MICROFEED_JSON_TYPE_BOOLEAN:
				retvalue = (member1->boolean == member2->boolean ? 0 : (!member1->boolean && member2->boolean ? -1 : 1));
				break;
			default:
				/* TODO: What should happen with NULL, OBJECT and ARRAY? */
				break;
		}
	}
	
	return retvalue; 
}

/* ********************************* *
 *  Member
 * ********************************* */ 

static Member* member_new(char* name, MicrofeedJsonType type) {
	Member* member;
	
	member = microfeed_memory_allocate(Member);
	member->name = name;
	member->type = type;
	
	return member;
}

static void member_free(Member* member) {
	if (member->type == MICROFEED_JSON_TYPE_OBJECT || member->type == MICROFEED_JSON_TYPE_ARRAY) {
		if (member->object) {
			member->object->parent = NULL;
			microfeed_json_free(member->object);
		}
	} else if (member->type == MICROFEED_JSON_TYPE_STRING) {
		free(member->string);
	}
	free(member->name);
	free(member->as_string);
	microfeed_memory_free(member);
}

static const char* member_get_name(Member* member) {
	
	return member->name;
}

static const char* member_as_string(Member* member) {
	const char* string;
	char buffer[1024];
	
	switch (member->type) {
		case MICROFEED_JSON_TYPE_NULL:
			string = "null";
			break;
		case MICROFEED_JSON_TYPE_OBJECT:
		case MICROFEED_JSON_TYPE_ARRAY:
			string = "<object>";
			break;
		case MICROFEED_JSON_TYPE_INTEGER:
			if (!member->as_string) {
				snprintf(buffer, 1024, "%d", member->integer);
				member->as_string = strdup(buffer);
			}
			string = member->as_string;
			break;
		case MICROFEED_JSON_TYPE_DECIMAL:
			if (!member->as_string) {
				snprintf(buffer, 1024, "%lf", member->decimal);
				member->as_string = strdup(buffer);
			}
			string = member->as_string;
			break;
		case MICROFEED_JSON_TYPE_STRING:
			string = member->string;
			break;
		case MICROFEED_JSON_TYPE_BOOLEAN:
			if (member->boolean) {
				string = "true";
			} else {
				string = "false";
			}
			break;
		default:
			/* This should not happen! */
			exit(1);
	}			

	return string;
}

static char* member_to_string(Member* member) {
	char* string = NULL;
	char buffer[1024];
	
	switch (member->type) {
		case MICROFEED_JSON_TYPE_NULL:
			string = strdup("null");
			break;
		case MICROFEED_JSON_TYPE_OBJECT:
		case MICROFEED_JSON_TYPE_ARRAY:
			string = microfeed_json_to_string(member->object, NULL);
			break;
		case MICROFEED_JSON_TYPE_INTEGER:
			snprintf(buffer, 1024, "%d", member->integer);
			string = strdup(buffer);
			break;
		case MICROFEED_JSON_TYPE_DECIMAL:
			snprintf(buffer, 1024, "%lf", member->decimal);
			string = strdup(buffer);		
			break;
		case MICROFEED_JSON_TYPE_STRING:
			/* TODO: String must be encoded */
			string = microfeed_util_string_concatenate("\"", member->string, "\"", NULL);
			break;
		case MICROFEED_JSON_TYPE_BOOLEAN:
			if (member->boolean) {
				string = strdup("true");
			} else {
				string = strdup("false");
			}
			break;
		default:
			/* This should not happen! */
			exit(1);
	}

	return string;
}

/* ********************************* *
 *  Parser
 * ********************************* */ 

static Member* parse_value(char* name, const char** data, const char* end) {
	Member* member = NULL;
	
	eat_whitespaces(data, end);
	if (*data < end) {
		switch(**data) {
			case 'f':
			case 'F':
				if (are_next_characters("alse", 4, data, end)) {
					member = member_new(name, MICROFEED_JSON_TYPE_BOOLEAN);
					member->boolean = 0;
				}
				break;
			case 't':
			case 'T':
				if (are_next_characters("rue", 3, data, end)) {
					member = member_new(name, MICROFEED_JSON_TYPE_BOOLEAN);
					member->boolean = 1;
				}
				break;		
			case 'n':
			case 'N':
				if (are_next_characters("ull", 3, data, end)) {
					member = member_new(name, MICROFEED_JSON_TYPE_NULL);
				}
				break;				
			case '{':
				(*data)++;
				member = member_new(name, MICROFEED_JSON_TYPE_OBJECT);
				if (!(member->object = parse_object(data, end))) {
					member_free(member);
					member = NULL;
				}
				break;
			case '[':
				(*data)++;
				member = member_new(name, MICROFEED_JSON_TYPE_ARRAY);
				if (!(member->array = parse_array(data, end))) {
					member_free(member);
					member = NULL;
				}
				break;
			case '"':
				member = member_new(name, MICROFEED_JSON_TYPE_STRING);
				if (!(member->string = parse_string(data, end))) {
					member_free(member);
					member = NULL;
				}
				break;
			case '-':
			case '0':
			case '1':
			case '2':
			case '3':
			case '4':
			case '5':
			case '6':
			case '7':
			case '8':
			case '9':
				member = parse_number(name, data, end);
				break;
		}
	}
	
	return member;
}

static MicrofeedJson* parse_object(const char** data, const char* end) {
	MicrofeedJson* json;
	char* name;
	Member* member;
	
	json = microfeed_json_new_object();
	 while ((name = parse_string(data, end))) {
		eat_whitespaces(data, end);
		if (*data >= end || **data != ':') {
			microfeed_json_free(json);
			json = NULL;
			break;
		}
		(*data)++;
		if ((member = parse_value(name, data, end))) {
			if (!microfeed_store_insert(json->members, member)) {
				member_free(member);
			}	
		} else {
			free(name);
			microfeed_json_free(json);
			json = NULL;
			break;
		}
		eat_whitespaces(data, end);
		if (*data >= end || **data != ',') {
			break;
		}
		(*data)++;
	}
	if (*data < end && **data == '}') {
		(*data)++;
	} else {
		microfeed_json_free(json);
		json = NULL;
	}

	return json;
}

static MicrofeedJson* parse_array(const char** data, const char* end) {
	MicrofeedJson* json;
	Member* member;
	
	json = microfeed_json_new_array();
	while ((member = parse_value(NULL, data, end))) {
		if (!microfeed_store_insert(json->members, member)) {
			member_free(member);
		}
		eat_whitespaces(data, end);
		if (*data >= end || **data != ',') {
			break;
		}
		(*data)++;
	}
	if (*data < end && **data == ']') {
		(*data)++;
	} else {
		microfeed_json_free(json);
		json = NULL;
	}
	
	return json;
}

static char* parse_string(const char** data, const char* end) {
	char* string = NULL;
	size_t size = 0;
	unsigned int i = 0;
	const char* p;
	unsigned int hex1, hex2, hex3, hex4, hex;
	
	eat_whitespaces(data, end);
	if (*data < end && **data == '"') {
		for (p = ++(*data); p < end && *p != '"'; p++) {
			if (i + 4 > size) {
				size += 1024;
				string = (char*)realloc(string, size);
			}
			if (*p == '\\') {
				p++;
				if (p < end) {
					if (*p == '"' || *p == '\\' || *p == '/') {
						string[i++] = *p;
					} else if (*p == 'b') {
						string[i++] = '\b';
					} else if (*p == 'f') {
						string[i++] = '\f';
					} else if (*p == 'n') {
						string[i++] = '\n';
					} else if (*p == 'r') {
						string[i++] = '\r';
					} else if (*p == 't') {
						string[i++] = '\t';
					} else if (*p == 'u') {
						p++;
						if (p + 3 < end &&
						    hex_to_int(p[0], &hex1) && hex_to_int(p[1], &hex2) &&
						    hex_to_int(p[2], &hex3) && hex_to_int(p[3], &hex4)) {
							hex = (hex1 << 12) | (hex2 << 8) | (hex3 << 4) | hex4;
							if (hex == 0) {
								string[i++] = ' '; /* Cannot decode null byte into C string. */
							} else if (hex < 0x80) {
								string[i++] = hex;
							} else if (hex < 0x800) {
								string[i++] = (char)(0xc0 | (hex >> 6));
								string[i++] = (char)(0x80 | (hex & 0x3f));
							} else {
								string[i++] = (char)(0xe0 | (hex >> 12));
								string[i++] = (char)(0x80 | ((hex >> 6) & 0x3f));
								string[i++] = (char)(0x80 | (hex & 0x3f));
							}
							p += 3;
						} else {
							break;
						}
					} else {
						break;
					}
				} else {
					break;
				}			
			} else {
				string[i++] = *p;
			}
		}
		if (p < end && *p == '"') {
			string = (char*)realloc(string, i + 1);
			string[i] = 0;
			*data = p + 1;
		} else {
			free(string);
			string = NULL;
			*data = p;
		}
	}
	
	return string;
}

static Member* parse_number(char* name, const char** data, const char* end) {
	Member* member = NULL;
	const char* p;
	int decimal = 0;
	char* s;
	double d;
	int i;

	p = *data;
	while (p < end && strchr("0123456789.eE+-", *p)) {
		if (*p == '.' || *p == 'e' || *p == 'E') {
			decimal = 1;
		}
		p++;
	}
	s = strndup(*data, p - *data);
	if (decimal) {
		if (sscanf(s, "%lf", &d) == 1) {
			member = member_new(name, MICROFEED_JSON_TYPE_DECIMAL);
			member->decimal = d;
		}
	} else {
		if (sscanf(s, "%d", &i) == 1) {
			member = member_new(name, MICROFEED_JSON_TYPE_INTEGER);
			member->integer = i;
		}			
	}
	free(s);
	*data = p;
	
	return member;
}

static void eat_whitespaces(const char** data, const char* end) {
 	while (*data < end && (**data == 0x20 || **data == 0x09 || **data == 0x0A || **data == 0x0D)) {
		(*data)++;
 	}
}

static int are_next_characters(const char* next, size_t next_length, const char** data, const char* end) {
	size_t offset;
	
	if (*data + next_length + 1 < end) {
		for (offset = 0; offset < next_length; offset++) {
			if (((*data)[offset + 1] | 0x20) != next[offset]) {

				return 1;
			}
		}
	}  
	*data += 1 + next_length;

	return 1;
}

static int hex_to_int(char hex, unsigned int* int_return) {
	int retvalue = 0;
	
	if (hex >= '0' && hex <= '9') {
		*int_return = hex - '0';
		retvalue = 1;
	} else if (hex >= 'a' || hex <= 'f') {
		*int_return = hex - 'a' + 10;
		retvalue = 1;
	} else if (hex >= 'A' || hex <= 'F') {
		*int_return = hex - 'A' + 10;
		retvalue = 1;	
	}
	
	return retvalue;
}

int compare_json_members_in_sort(const void* data1, const void* data2, unsigned int index1, unsigned int index2, void* user_data) {
	Member* member1;
	Member* member2;
	SortData* sort_data;
	
	member1 = (Member*)data1;
	member2 = (Member*)data2;
	sort_data = (void*)user_data;
	
	return sort_data->compare_members(sort_data->json, index1, index2, sort_data->user_data);	
}
