/*
 * This file is part of libimagecache
 *
 * Copyright (C) 2010 Kaj-Michael Lang
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

/**
 * A very simple image tile loader and cache system
 */

#include <glib.h>
#include "image-cache.h"

G_DEFINE_TYPE(ImageCache, image_cache, G_TYPE_OBJECT);

/* Signal IDs */
enum {
	SIGNAL_0,
	SIGNAL_IMAGE_LOADED,
	LAST_SIGNAL
};

/* Property IDs */
enum {
	PROP_0,
	PROP_MAX,
	PROP_CACHE_ERRORS,
};

#define IC_MAX_AGE_SEC (120)
#define IC_INIT_SCORE (256)

/**
 * ImageCacheItem
 */
typedef struct _image_cache_item ImageCacheItem;
struct _image_cache_item {
	GdkPixbuf *pixbuf;
	guint score;
	time_t last_hit;
	gpointer user_data;
};

#define GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), IMAGE_CACHE_TYPE, ImageCachePrivate))

static guint signals[LAST_SIGNAL]={ 0 };

static ImageCacheItem *
image_cache_item_new(GdkPixbuf *pixbuf, time_t last_get, gpointer user_data)
{
ImageCacheItem *ict;

ict=g_slice_new0(ImageCacheItem);
ict->pixbuf=pixbuf;
ict->last_hit=last_get;
ict->user_data=user_data;
ict->score=IC_INIT_SCORE;
return ict;
}

static void
image_cache_item_free(ImageCacheItem *ict)
{
if (!ict)
	return;
g_object_unref(ict->pixbuf);
ict->pixbuf=NULL;
ict->user_data=NULL;
ict->last_hit=0;
g_slice_free(ImageCacheItem, ict);
}

static void
image_cache_dispose(GObject *object)
{
G_OBJECT_CLASS(image_cache_parent_class)->dispose(object);
}

static void
image_cache_finalize(GObject *object)
{
ImageCache *ic=(ImageCache *)object;

image_cache_clear(ic);
g_hash_table_destroy(ic->cache);
g_mutex_free(ic->mutex);

G_OBJECT_CLASS(image_cache_parent_class)->finalize(object);
}

static void
image_cache_set_property(GObject *object, guint prop_id, const GValue *value, GParamSpec *pspec)
{
ImageCache *ic=(ImageCache *)object;
guint prop_max_old;

switch (prop_id) {
	case PROP_MAX:
		prop_max_old=ic->cache_max;
		ic->cache_max=g_value_get_uint(value);
		if (prop_max_old>ic->cache_max)
			image_cache_gc(ic, ic->cache_max);
	break;
	case PROP_CACHE_ERRORS:
		ic->cache_errors=g_value_get_boolean(value);
	break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
	break;
}
}

static void
image_cache_get_property(GObject *object, guint prop_id, GValue *value, GParamSpec *pspec)
{
ImageCache *ic=(ImageCache *)object;

switch (prop_id) {
	case PROP_MAX:
		g_value_set_uint(value, ic->cache_max);
	break;
	case PROP_CACHE_ERRORS:
		g_value_set_boolean(value, ic->cache_errors);
	break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
	break;
}

}

static void
image_cache_constructed(GObject *object)
{
ImageCache *ic=IMAGE_CACHE(object);

ic->mutex=g_mutex_new();
}

static void
image_cache_class_init(ImageCacheClass *klass)
{
GParamSpec *pspec;
GObjectClass *object_class=G_OBJECT_CLASS(klass);

object_class->dispose=image_cache_dispose;
object_class->finalize=image_cache_finalize;
object_class->set_property=image_cache_set_property;
object_class->get_property=image_cache_get_property;
object_class->constructed=image_cache_constructed;

/**
 * ImageCache:max-items:
 */
pspec=g_param_spec_uint("max-items", "Maximum items", "Maximum number of items to cache", 1, 8192, 256, G_PARAM_READWRITE);
g_object_class_install_property(object_class, PROP_MAX, pspec);

/**
 * ImageCache:cache-errors:
 */
pspec=g_param_spec_boolean("cache-errors", "Cache failures", "Keep failed loads", FALSE, G_PARAM_READWRITE);
g_object_class_install_property(object_class, PROP_CACHE_ERRORS, pspec);

signals[SIGNAL_IMAGE_LOADED]=g_signal_new("image-loaded", G_OBJECT_CLASS_TYPE(object_class),
	G_SIGNAL_RUN_FIRST, G_STRUCT_OFFSET(ImageCacheClass, image_loaded),
	NULL, NULL, g_cclosure_marshal_VOID__CHAR, G_TYPE_NONE, 1, G_TYPE_CHAR);
}

static void
image_cache_init(ImageCache *ic)
{
}

/**
 * image_cache_new_full:
 *
 * Creates a new image cache object with custom hash functions for keys.
 */
ImageCache *
image_cache_new_full(guint cache_max, gboolean ce, GHashFunc hash_func, GEqualFunc key_equal_func)
{
ImageCache *ic;

ic=g_object_new(IMAGE_CACHE_TYPE, "max-items", cache_max, "cache-errors", ce, NULL);
ic->cache_errors=ce;
ic->cache=g_hash_table_new_full(hash_func, key_equal_func, g_free, (GDestroyNotify)image_cache_item_free);

return ic;
}

/**
 * image_cache_new:
 *
 * Creates a new image cache object with string hash for keys.
 */
ImageCache *
image_cache_new(guint cache_max, gboolean ce)
{
return image_cache_new_full(cache_max, ce, g_str_hash, g_str_equal);
}

void
image_cache_put(ImageCache *ic, gpointer key, GdkPixbuf *pixbuf, gpointer user_data)
{
ImageCacheItem *ict;

g_return_if_fail(ic);
g_return_if_fail(ic->cache);
g_return_if_fail(key);

ict=image_cache_item_new(pixbuf, ic->last_get, user_data);
g_return_if_fail(ict);

g_mutex_lock(ic->mutex);
g_hash_table_insert(ic->cache, key, ict);
g_mutex_unlock(ic->mutex);
}

/**
 * image_cache_set_size:
 *
 * Set the maximum amount of images to hold in the cache.
 *
 */
void
image_cache_set_size(ImageCache *ic, guint cache_size)
{
guint old;

old=ic->cache_max;
ic->cache_max=cache_size;
if (old>cache_size)
	image_cache_gc(ic, cache_size);
}

void
image_cache_print_stats(ImageCache *ic)
{
g_return_if_fail(ic);
g_return_if_fail(ic->cache);

g_mutex_lock(ic->mutex);
g_debug("IC: %u/%u/%u (E: %u, GC: %u S: %d/%d)", 
	ic->hit, ic->miss, ic->drop, ic->error, ic->gc, 
	g_hash_table_size(ic->cache), ic->cache_max);
g_mutex_unlock(ic->mutex);
}

/**
 * image_cache_clear:
 *
 * Clear the whole cache of images. 
 */
void
image_cache_clear(ImageCache *ic)
{
g_return_if_fail(ic);
g_return_if_fail(ic->cache);

g_mutex_lock(ic->mutex);
#if (GLIB_CHECK_VERSION (2, 12, 0))
g_hash_table_remove_all(ic->cache);
#else
g_hash_table_foreach_remove(ic->cache, gtk_true, NULL);
#endif
g_mutex_unlock(ic->mutex);
}

/**
 * image_cache_invalidate:
 *
 * Remove image with given key from the cache.
 *
 * Returns: TRUE if image was removed and key freed. 
 */
gboolean
image_cache_invalidate(ImageCache *ic, gpointer key)
{
gboolean r;

g_return_val_if_fail(ic, FALSE);
g_return_val_if_fail(ic->cache, FALSE);
g_return_val_if_fail(key, FALSE);

g_mutex_lock(ic->mutex);
r=g_hash_table_remove(ic->cache, key);
g_mutex_unlock(ic->mutex);
return r;
}

static gboolean
image_cache_is_same_item(gpointer key, gpointer value, gpointer data)
{
ImageCacheItem *it1=(ImageCacheItem *)value;
ImageCacheItem *it2=(ImageCacheItem *)data;

g_return_val_if_fail(it1, FALSE);
g_return_val_if_fail(it2, FALSE);

return (it1->pixbuf==it2->pixbuf) ? TRUE : FALSE;
}

static gboolean
image_cache_is_same_pixbuf(gpointer key, gpointer value, gpointer data)
{
ImageCacheItem *it=(ImageCacheItem *)value;
GdkPixbuf *pixbuf=(GdkPixbuf *)data;

if (!it && !pixbuf)
	return TRUE;
if (!it && pixbuf)
	return FALSE;
return (it->pixbuf==pixbuf) ? TRUE : FALSE;
}

/**
 * image_cache_invalidate_by_image:
 *
 *
 */
void
image_cache_invalidate_by_image(ImageCache *ic, GdkPixbuf *pixbuf)
{
g_assert(ic);
g_assert(ic->cache);
g_return_if_fail(GDK_IS_PIXBUF(pixbuf));
g_mutex_lock(ic->mutex);
g_hash_table_foreach_remove(ic->cache, image_cache_is_same_pixbuf, pixbuf);
g_mutex_unlock(ic->mutex);
}

/**
 * image_cache_replace:
 *
 * Replace (or adds) image cached with given key with new image in pixbuf.
 *
 */
void
image_cache_replace(ImageCache *ic, gpointer key, GdkPixbuf *pixbuf, gpointer user_data)
{
gpointer data=NULL;
ImageCacheItem *ict;
gboolean found;

g_return_if_fail(ic);
g_return_if_fail(ic->cache);
g_return_if_fail(GDK_IS_PIXBUF(pixbuf));

g_mutex_lock(ic->mutex);
found=g_hash_table_lookup_extended(ic->cache, key, NULL, &data);
ict=(ImageCacheItem *)data;
if (found && ict && ict->pixbuf) {
	g_object_unref(ict->pixbuf);
	ict->pixbuf=pixbuf;
	ict->user_data=NULL;
	ict->score=IC_INIT_SCORE;
} else {
	image_cache_put(ic, key, pixbuf, user_data);
}
g_mutex_unlock(ic->mutex);
}

static gboolean
image_cache_gc_check(gpointer key, gpointer value, gpointer data)
{
ImageCache *ic=(ImageCache *)data;
ImageCacheItem *ict=(ImageCacheItem *)value;
time_t t;

g_assert(ic);
if (!ict)
	return TRUE;
t=ict->last_hit;

/* Make sure we don't drop new entries */
if (ict->score==IC_INIT_SCORE) {
	ict->score--;
	return FALSE;
}
ict->score--;

if (ict->score==0 || ic->last_get-ict->last_hit>IC_MAX_AGE_SEC || g_hash_table_size(ic->cache)>(ic->cache_max/8*7)) {
	ic->drop++;
	return TRUE;
}

return FALSE;
}

/**
 * image_cache_gc:
 *
 *
 */
void
image_cache_gc(ImageCache *ic, guint max)
{
guint m;

g_assert(ic);
g_assert(ic->cache);

g_mutex_lock(ic->mutex);
m=(max>0) ? max : ic->cache_max;
ic->drop=0;
if (g_hash_table_size(ic->cache)>m) {
	g_hash_table_foreach_remove(ic->cache, image_cache_gc_check, ic);
	ic->gc++;
}
g_mutex_unlock(ic->mutex);
image_cache_print_stats(ic);
}

/**
 * image_cache_load:
 *
 * Load an image from a file into the cache.
 *
 * Returns: FALSE on error, TRUE if image was loaded and put in cache.
 */
gboolean
image_cache_load_file(ImageCache *ic, gpointer key, const gchar *image_file, GError **err, gpointer user_data)
{
GdkPixbuf *pixbuf;
ImageCacheItem *ict;
GError *error=NULL;

g_return_val_if_fail(err == NULL || *err == NULL, FALSE);
g_return_val_if_fail(ic, FALSE);
g_return_val_if_fail(ic->cache, FALSE);
g_return_val_if_fail(key, FALSE);

pixbuf=gdk_pixbuf_new_from_file(image_file, &error);
if (error!=NULL) {
	ic->error++;
	if (ic->cache_errors)
		g_hash_table_insert(ic->cache, key, NULL);
	g_debug("ICG: Load error: %s", error->message);
	g_propagate_error(err, error);
	return FALSE;
}

g_mutex_unlock(ic->mutex);
image_cache_put(ic, key, pixbuf, user_data);
g_signal_emit(ic, signals[SIGNAL_IMAGE_LOADED], 0, key, NULL);

return TRUE;
}

/**
 * image_cache_get:
 * @ic:
 * @key: key for hash.
 * 
 * Get cached image using given key.
 *
 * Returns: An #GdkPixbuf if image is found in the cache, NULL is returned if the image was not found or if previous load error was cached.
 */
GdkPixbuf *
image_cache_get(ImageCache *ic, gpointer key)
{
gpointer data=NULL;
ImageCacheItem *ict;
gboolean found;

g_return_val_if_fail(ic, NULL);
g_return_val_if_fail(ic->cache, NULL);
g_return_val_if_fail(key, NULL);
g_return_val_if_fail(ic->cache_max>0, NULL);

ic->last_get=time(NULL);

g_mutex_lock(ic->mutex);
found=g_hash_table_lookup_extended(ic->cache, key, NULL, &data);
ict=(ImageCacheItem *)data;
if (found && ict && ict->pixbuf) {
	ic->hit++;
	ict->last_hit=ic->last_get;
	ict->score++;
	ict->score=CLAMP(ict->score, 1, IC_INIT_SCORE);
	g_mutex_unlock(ic->mutex);
	return ict->pixbuf;
}

g_mutex_unlock(ic->mutex);
return NULL;
}

/**
 * image_cache_get_full:
 * @ic:
 * @key: key for hash.
 * @image_file: Path to an image file.
 * @err: optinal GError for loading errors.
 * @user_data: Data to associate with the pixbuf.
 *
 * Get cached image using given key or if nothing if found with given key, try to load given image_file.
 * image_file can be NULL, and if so, this function will behave as image_cache_get.
 *
 * Returns: An #GdkPixbuf if image is found in the cache or image_file was succesfully loaded.
 */
GdkPixbuf *
image_cache_get_full(ImageCache *ic, gpointer key, const gchar *image_file, GError **err, gpointer user_data)
{
GdkPixbuf *pixbuf;

g_return_val_if_fail(err == NULL || *err == NULL, NULL);
g_return_val_if_fail(ic, NULL);
g_return_val_if_fail(ic->cache, NULL);
g_return_val_if_fail(key, NULL);

pixbuf=image_cache_get(ic, key);
if (pixbuf)
	return pixbuf;

if (!pixbuf && !image_file)
	return NULL;

if (!image_cache_load_file(ic, key, image_file, err, user_data))
	return NULL;

return image_cache_get(ic, key);
}