/*
 * This file is a part of hildon-extras
 *
 * Copyright (C) 2010 Gabriel Schulhof <nix@go-nix.ca>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version. or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/**
 * SECTION:he-menu-view
 * @short_description: A widget to display a #GtkMenu as a series of #GtkTreeView columns.
 *
 * #HeMenuView takes a #GtkMenu and displays it as a series of columns of #GtkTreeView widgets. When the user clicks on 
 * a row, the corresponding #GtkMenuItem is activated.
 */

#include <hildon/hildon.h>
#include "he-menu-store.h"
#include "he-menu-view-column.h"
#include "he-menu-view.h"
#include "he-menu-view_priv.h"

struct _HeMenuViewPrivate
{
	GQueue *menu_q;
	GtkMenu *menu;
	GtkWidget *button;

	gboolean head_is_new;
	gboolean resizing;
	gboolean froze_updates;

	GtkWidget *menu_widget;
	gboolean do_activate;
	GQueue *parents_q;
};

enum {
	HE_MENU_VIEW_MENU_PROPERTY = 1,
	HE_MENU_VIEW_MENU_WIDGET_PROPERTY,
	HE_MENU_VIEW_DO_ACTIVATE_PROPERTY,
};

G_DEFINE_TYPE(HeMenuView, he_menu_view, GTK_TYPE_FIXED);

static void hmvc_notify_menu_widget(GtkWidget *hmvc, GParamSpec *pspec, HeMenuView *hmv);
static gboolean my_toggle_menu(HildonWindow *self, guint button, guint32 time);
static void hmvc_destroy_cb(GtkWidget *hmvc, HeMenuView *hmv);

/*
 * Start resizing the widget
 */
static void
start_resizing(HeMenuView *hmv)
{
	if (!(hmv->priv->resizing)) {
		GdkWindow *wnd = GTK_WIDGET(hmv)->window;

		hmv->priv->resizing = TRUE;
		if (wnd && !hmv->priv->froze_updates) {
			gdk_window_freeze_updates(wnd);
	//		g_print("size_allocate: ----- disable updates -----\n");
			hmv->priv->froze_updates = TRUE;
		}
	}
}

static void
redo_parents_q(HeMenuView *hmv)
{
	GtkWidget *parent;
	GList *ll;

	for (ll = hmv->priv->parents_q->head ; ll ; ll = ll->next)
		if (ll->data)
			g_object_remove_weak_pointer(G_OBJECT(ll->data), &(ll->data));
	g_queue_clear(hmv->priv->parents_q);

	if (hmv->priv->menu_q->head)
		for (parent = _hmv_private_menu_get_parent_menu(he_menu_view_column_get_menu(HE_MENU_VIEW_COLUMN(hmv->priv->menu_q->head->data))) ;
		     parent ;
		     parent = _hmv_private_menu_get_parent_menu(parent)) {
			g_queue_push_tail(hmv->priv->parents_q, parent);
			g_object_add_weak_pointer(G_OBJECT(hmv->priv->parents_q->tail->data), (gpointer *)&(hmv->priv->parents_q->tail->data));
		}
}

/*
 * Finish resizing the widget
 */
static void
stop_resizing(HeMenuView *hmv)
{
	if (hmv->priv->resizing) {	
		GdkWindow *wnd = GTK_WIDGET(hmv)->window;
	
		hmv->priv->resizing = FALSE ;
		if (hmv->priv->froze_updates && wnd) {
			gdk_window_thaw_updates(wnd);
			hmv->priv->froze_updates = FALSE;
		}
	}

	redo_parents_q(hmv);
}

/*
 * Add a new column to the widget
 */
static GtkWidget *
add_new_column(HeMenuView *hmv, GtkMenu *menu)
{
	GtkWidget *hmvc;

	start_resizing(hmv);

	hmvc = g_object_new(HE_TYPE_MENU_VIEW_COLUMN, "visible", TRUE, "menu", menu, "do-activate", FALSE, NULL);
	gtk_container_add(GTK_CONTAINER(hmv), hmvc);
	g_signal_connect(G_OBJECT(hmvc), "notify::menu-widget", (GCallback)hmvc_notify_menu_widget, hmv);

	g_object_set_data(G_OBJECT(hmvc), "destroy-signal-id",
		GINT_TO_POINTER(g_signal_connect(G_OBJECT(hmvc), "destroy", (GCallback)hmvc_destroy_cb, hmv)));

	return hmvc;
}

/*
 * Move up menu levels
 */
static void
button_clicked(GtkWidget *button, HeMenuView *hmv)
{
	GList *ll;

	for (ll = hmv->priv->parents_q->head ; ll ; ll = ll->next)
		if (ll->data)
			break;

	if (ll) {
		g_queue_push_head(hmv->priv->menu_q, add_new_column(hmv, GTK_MENU(ll->data)));
		hmv->priv->head_is_new = TRUE;
		redo_parents_q(hmv);
	}

	if (hmv->priv->parents_q->length <= 0)
		gtk_widget_hide(hmv->priv->button);

	if (hmv->priv->menu_q->length <= 0)
		gtk_widget_destroy(GTK_WIDGET(hmv));
}

/*
 * Clean up column queue when column is destroyed
 */
static void
hmvc_destroy_cb(GtkWidget *hmvc, HeMenuView *hmv)
{
	g_queue_remove(hmv->priv->menu_q, hmvc);
}

/*
 * Destroy a column
 */
static void
destroy_hmvc(GObject *hmvc)
{
	guint destroy_signal_id = GPOINTER_TO_INT(g_object_get_data(hmvc, "destroy-signal-id"));

	if (destroy_signal_id)
		g_signal_handler_disconnect(hmvc, destroy_signal_id);
	gtk_widget_destroy(GTK_WIDGET(hmvc));
}

/*
 * Handle notifications about clicks on menu items
 */
static void
hmvc_notify_menu_widget(GtkWidget *hmvc, GParamSpec *pspec, HeMenuView *hmv)
{
	GtkMenuItem *menu_item = NULL;

	g_object_get(G_OBJECT(hmvc), "menu-widget", &menu_item, NULL);

	if (menu_item) {
		GtkWidget *submenu = gtk_menu_item_get_submenu(menu_item);

		if (submenu) {
			while(hmv->priv->menu_q->tail->data != hmvc) {
				destroy_hmvc(G_OBJECT(hmv->priv->menu_q->tail->data));
				g_queue_pop_tail(hmv->priv->menu_q);
			}

			g_queue_push_tail(hmv->priv->menu_q, add_new_column(hmv, GTK_MENU(submenu)));
			hmv->priv->head_is_new = FALSE;
		}
		else {
			hmv->priv->menu_widget = GTK_WIDGET(menu_item);
			g_object_notify(G_OBJECT(hmv), "menu-widget");
		}
	}
}

/*
 * Allocate the space for a single column
 */
static int
do_single_allocate(GtkContainer *me, GtkWidget *widget, gboolean is_last, int x, int y, int max_width, int left_over_part, int height, gboolean *p_is_dirty)
{
	GtkRequisition rq_child;
	int x_child;

	GtkAllocation alloc = {
		.x      = x,
		.y      = y,
		.height = height
	};

	gtk_widget_get_child_requisition(widget, &rq_child);
	gtk_container_child_get(me, widget, "x", &x_child, NULL);
	if (x_child != x) {
		(*p_is_dirty) = TRUE;
		gtk_container_child_set(me, widget, "x", x, NULL);
	}

	alloc.width = is_last ? max_width : MIN(max_width, rq_child.width + left_over_part);

	/* Don't bother allocating if there's gonna be another size-allocate anyway */
	if (!(*p_is_dirty))
		gtk_widget_size_allocate(widget, &alloc);

	return alloc.width;
}

/*
 * Destroy columns starting from the tail or starting from the head
 */
static gboolean
destroy_excess_columns(HeMenuView *hmv, int alloc_width, gboolean tail_is_significant, GList **p_beg, GList **p_end, int advance_offset, gpointer (*pop_fn)(GQueue *))
{
	gboolean keep_looping;
	GList *ll_itr;
	int width_left = alloc_width;
	GtkRequisition rq_child;

	if (GTK_WIDGET_VISIBLE(hmv->priv->button)) {
		gtk_widget_get_child_requisition(GTK_WIDGET(hmv->priv->button), &rq_child);
		width_left -= rq_child.width;
	}

	/* start from head (or tail) and go next (or prev) */
	for (ll_itr = (*p_beg) ; ll_itr ; ll_itr = G_STRUCT_MEMBER(GList *, ll_itr, advance_offset)) {
		gtk_widget_get_child_requisition(GTK_WIDGET(ll_itr->data), &rq_child);
		width_left -= rq_child.width;
		if (width_left < 0)
			break;
	}

	if (ll_itr) {
		if (tail_is_significant && !GTK_WIDGET_VISIBLE(hmv->priv->button)) {
			gtk_widget_show(hmv->priv->button);
			/* Need another size-allocate to account for the button before we can start the actual destruction */
			return FALSE;
		}

		do {
			/* Never empty the queue completely */
			if (hmv->priv->menu_q->length > 1) {
				keep_looping = (ll_itr != (*p_end));
				destroy_hmvc(G_OBJECT((*p_end)->data));
				(*pop_fn)(hmv->priv->menu_q);
			}
			else
				keep_looping = FALSE;
		} while (keep_looping);
	}

	return TRUE;
}

/*
 * distribute the allocation to the various columns
 */
static void
size_allocate(GtkWidget *widget, GtkAllocation *alloc)
{
	gboolean is_dirty = FALSE;
	GtkRequisition hmv_rq, rq_child;
	int x = 0;
	HeMenuView *hmv = HE_MENU_VIEW(widget);
	GtkContainer *hmv_container = GTK_CONTAINER(widget);
	GList *ll_itr;
	int width_left = alloc->width;
	int width_gone = 0;
	int width_accum = 0;
	int left_over_part = 0;

//	g_print("\nHMV: size_allocate (%d,%d)[%dx%d]\n", alloc->x, alloc->y, alloc->width, alloc->height);

	widget->allocation = *alloc;

	gtk_widget_get_child_requisition(widget, &hmv_rq);

	if (hmv_rq.width > alloc->width) {
		if (hmv->priv->head_is_new)
			destroy_excess_columns(hmv, alloc->width, FALSE, &(hmv->priv->menu_q->head), &(hmv->priv->menu_q->tail), G_STRUCT_OFFSET(GList, next), g_queue_pop_tail);
		else
		if (!destroy_excess_columns(hmv, alloc->width, TRUE, &(hmv->priv->menu_q->tail), &(hmv->priv->menu_q->head), G_STRUCT_OFFSET(GList, prev), g_queue_pop_head))
				return;
	}

	if (GTK_WIDGET_VISIBLE(hmv->priv->button)) {
		width_gone = do_single_allocate(hmv_container, hmv->priv->button, FALSE, 0, 0, width_left, 0, alloc->height, &is_dirty);
		x += width_gone;
		width_left -= width_gone;
	}

	for (ll_itr = hmv->priv->menu_q->head ; ll_itr ; ll_itr = ll_itr->next) {
		gtk_widget_get_child_requisition(GTK_WIDGET(ll_itr->data), &rq_child);
		width_accum += rq_child.width;
	}

	if (hmv->priv->menu_q->length > 0)
		left_over_part = (MAX(width_left - width_accum, 0) / hmv->priv->menu_q->length);
//	g_print("HMV: size_allocate: left_over_part = (%d - %d) / %d = %d\n", width_left, width_accum, hmv->priv->menu_q->length, left_over_part);

	for (ll_itr = hmv->priv->menu_q->head ; ll_itr ; ll_itr = ll_itr->next) {
		width_gone = do_single_allocate(hmv_container, GTK_WIDGET(ll_itr->data), ll_itr->next == NULL, x, 0, width_left, left_over_part, alloc->height, &is_dirty);
		x += width_gone;
		width_left -= width_gone;
	}

	if (!is_dirty) {
		if (0 == hmv->priv->menu_q->length)
			button_clicked(hmv->priv->button, hmv);
		else {
			stop_resizing(hmv);
			hmv->priv->head_is_new = TRUE;
		}
	}
}

/*
 * Finalize the widget
 */
static void
finalize(GObject *obj)
{
	g_queue_free(HE_MENU_VIEW(obj)->priv->menu_q);
}

/*
 * Entry point for property value retrievals
 */
static void
get_property(GObject *obj, guint property_id, GValue *val, GParamSpec *pspec)
{
	switch(property_id) {

		case HE_MENU_VIEW_MENU_PROPERTY:
			g_value_set_object(val, G_OBJECT(he_menu_view_get_menu(HE_MENU_VIEW(obj))));
			break;

		case HE_MENU_VIEW_MENU_WIDGET_PROPERTY:
			g_value_set_object(val, HE_MENU_VIEW(obj)->priv->menu_widget);
			break;

		case HE_MENU_VIEW_DO_ACTIVATE_PROPERTY:
			g_value_set_boolean(val, HE_MENU_VIEW(obj)->priv->do_activate);
			break;

		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(obj, property_id, pspec);
			break;
	}
}

/*
 * Entry point for property changes
 */
static void
set_property(GObject *obj, guint property_id, const GValue *val, GParamSpec *pspec)
{
	switch(property_id) {
		case HE_MENU_VIEW_MENU_PROPERTY:
			he_menu_view_set_menu(HE_MENU_VIEW(obj), GTK_MENU(g_value_get_object(val)));
			break;

		case HE_MENU_VIEW_DO_ACTIVATE_PROPERTY:
			he_menu_view_set_do_activate(HE_MENU_VIEW(obj), g_value_get_boolean(val));
			break;

		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(obj, property_id, pspec);
			break;
	}
}

/*
 * Initialize the #HeMenuView class
 */
static void
he_menu_view_class_init(HeMenuViewClass *hmv_class)
{
	GObjectClass *gobject_class = G_OBJECT_CLASS(hmv_class);
	GtkWidgetClass *gtkwidget_class = GTK_WIDGET_CLASS(hmv_class);

	gobject_class->finalize     = finalize;
	gobject_class->set_property = set_property;
	gobject_class->get_property = get_property;

	gtkwidget_class->size_allocate = size_allocate;

	g_object_class_install_property(gobject_class, HE_MENU_VIEW_MENU_PROPERTY,
		g_param_spec_object("menu", "Menu", "The menu to display",
			GTK_TYPE_MENU, G_PARAM_READWRITE));

	g_object_class_install_property(gobject_class, HE_MENU_VIEW_MENU_WIDGET_PROPERTY,
		g_param_spec_object("menu-widget", "Menu Widget", "The last menu widget that was activated",
			GTK_TYPE_WIDGET, G_PARAM_READABLE));

	g_object_class_install_property(gobject_class, HE_MENU_VIEW_DO_ACTIVATE_PROPERTY,
		g_param_spec_boolean("do-activate", "Activate Menu Widget", "Whether to activate the GtkMenuItem widget when the corresponding item is chosen",
			TRUE, G_PARAM_READWRITE));

	g_type_class_add_private(hmv_class, sizeof(HeMenuViewPrivate));
}

/*
 * Initialize the #HeMenuView widget
 */
static void
he_menu_view_init(HeMenuView *hmv)
{
	hmv->priv = G_TYPE_INSTANCE_GET_PRIVATE(hmv, HE_TYPE_MENU_VIEW, HeMenuViewPrivate);
	hmv->priv->do_activate   = TRUE;
	hmv->priv->menu_widget   = NULL;
	hmv->priv->head_is_new   = TRUE;
	hmv->priv->resizing      = FALSE;
	hmv->priv->froze_updates = FALSE;
	hmv->priv->menu_q        = g_queue_new();
	hmv->priv->parents_q     = g_queue_new();
	hmv->priv->button        =
		g_object_new(GTK_TYPE_BUTTON, "image",
			g_object_new(GTK_TYPE_IMAGE, "visible", TRUE, "stock", GTK_STOCK_GO_BACK, "icon-size", GTK_ICON_SIZE_MENU, NULL),
			NULL);

	g_signal_connect(G_OBJECT(hmv->priv->button), "clicked", (GCallback)button_clicked, hmv);
	gtk_container_add(GTK_CONTAINER(hmv), hmv->priv->button);
}

/**
 * he_menu_view_set_menu:
 * @hmv: The #HeMenuView
 * @menu: The #GtkMenu to display
 *
 * Set the menu to be displayed by the widget. The widget takes ownership of the #GtkMenu, so you can release your reference to it afterwards.
 *
 * Since: 0.9.1
 */
void
he_menu_view_set_menu(HeMenuView *hmv, GtkMenu *menu)
{
	g_return_if_fail(HE_IS_MENU_VIEW(hmv));
	g_return_if_fail((NULL == menu) || GTK_IS_MENU(menu));

	if (menu != hmv->priv->menu) {
		if (hmv->priv->menu) {
			g_object_unref(hmv->priv->menu);
			hmv->priv->menu = NULL;
		}

		g_queue_foreach(hmv->priv->menu_q, (GFunc)destroy_hmvc, NULL);
		g_queue_clear(hmv->priv->menu_q);

		if (menu) {
			hmv->priv->menu = g_object_ref_sink(G_OBJECT(menu));
			g_queue_push_head(hmv->priv->menu_q, add_new_column(hmv, GTK_MENU(menu)));
		}

	g_object_notify(G_OBJECT(hmv), "menu");
	}
}

/**
 * he_menu_view_get_menu:
 * @hmv: The #HeMenuView
 *
 * Retrieve the menu being displayed by the widget. The widget owns the #GtkMenu, so you must not destroy it.
 *
 * Returns: The #GtkMenu being displayed.
 *
 * Since: 0.9.1
 */
GtkWidget *
he_menu_view_get_menu(HeMenuView *hmv)
{
	return hmv->priv->menu ? GTK_WIDGET(hmv->priv->menu) : NULL;
}

/*
 * Need to briefly ignore delete events when popping the dialog
 */
static gboolean
dialog_ignore_close(GObject *dlg)
{
	g_object_set_data(dlg, "no-close-timeout-id", GINT_TO_POINTER(0));
	return FALSE;
}

/*
 * Ignore delete during the initial time
 */
static gboolean
dlg_ignore_delete(GObject *dlg, GdkEvent *event, gpointer null)
{
	if ((gboolean)GPOINTER_TO_INT(g_object_get_data(dlg, "no-close-timeout-id")))
		return TRUE;
	else
		gtk_widget_destroy(GTK_WIDGET(dlg));
	return FALSE;
}

/*
 * Unset the borders and spacings of the given widget and all its children
 */
static void
unset_borders(GtkWidget *widget, gpointer null)
{
	if (GTK_IS_CONTAINER(widget)) {
		g_object_set(G_OBJECT(widget), "border-width", 0, NULL);
		if (GTK_IS_BOX(widget))
			g_object_set(G_OBJECT(widget), "spacing", 0, NULL);
		else
		if (GTK_IS_ALIGNMENT(widget))
			g_object_set(G_OBJECT(widget),
				"xscale", 1.0, "yscale", 1.0,
				"left-padding", 0, "right-padding", 0, 
				"top-padding", 0, "bottom-padding", 0, NULL);
		gtk_container_foreach(GTK_CONTAINER(widget), (GtkCallback)unset_borders, NULL);
	}
}

typedef struct {
	GtkWidget *dlg;
	GtkMenuItem *item;
	GtkMenu *menu;
	GtkWidget *hmv;
} DialogHasDiedParams;

/*
 * Only activate the menu item after the dialog has disappeared. This helps prevent the popup dialog from having an effect on
 * dialogs that may be popped up as a result of the action of the menu item.
 */
static void
dialog_has_died(DialogHasDiedParams *params, gpointer dead_dialog)
{
	if (params->item) {
		gtk_menu_item_activate(params->item);
		g_object_unref(G_OBJECT(params->item));
	}
	if (params->menu) {
		g_signal_emit_by_name(G_OBJECT(params->menu), "deactivate");
		g_object_unref(params->menu);
	}
	g_free(params);
}

/*
 * The user has chosen a #GtkMenuItem. Destroy the dialog and set up its activation upon the dialog's destruction.
 */
static void
menu_widget_notify(HeMenuView *hmv, GParamSpec *pspec, DialogHasDiedParams *params)
{
	GtkWidget *menu_widget = he_menu_view_get_menu_widget(hmv);

	if (menu_widget)
		params->item = g_object_ref(menu_widget);
	gtk_widget_destroy(params->dlg);
}

static void
hmv_destroy(GtkWidget *hmv, DialogHasDiedParams *params)
{
	params->menu = NULL;
	gtk_widget_destroy(params->dlg);
}

/*
 * Pop a dialog displaying the given #GtkMenu and return if @run is FALSE.
 */
static void
pop_dialog(GtkWindow *parent, GtkMenu *menu, gboolean run)
{
	GtkWidget *dlg;
	DialogHasDiedParams *params = g_new0(DialogHasDiedParams, 1);

	params->hmv = g_object_new(HE_TYPE_MENU_VIEW, "visible", TRUE, "menu", menu, "do-activate", FALSE, NULL);

	dlg = g_object_new(GTK_TYPE_DIALOG, "visible", TRUE, "transient-for", parent, NULL);
	g_object_set_data_full(G_OBJECT(dlg), "no-close-timeout-id", 
		GINT_TO_POINTER(g_timeout_add_seconds(1, (GSourceFunc)dialog_ignore_close, dlg)),
		(GDestroyNotify)g_source_remove);
	g_signal_connect(G_OBJECT(dlg), "delete-event", (GCallback)dlg_ignore_delete, NULL);
	g_signal_connect(G_OBJECT(params->hmv), "destroy", (GCallback)hmv_destroy, params);

	params->dlg = dlg;
	params->menu = g_object_ref(menu);

	unset_borders(GTK_BIN(dlg)->child, NULL);

	gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dlg)->vbox), params->hmv);

	g_object_weak_ref(G_OBJECT(dlg), (GWeakNotify)dialog_has_died, params);

	g_signal_connect(G_OBJECT(params->hmv), "notify::menu-widget", (GCallback)menu_widget_notify, params);

	if (run)
		gtk_dialog_run(GTK_DIALOG(dlg));
	else
		gtk_widget_show(dlg);
}

/*
 * Replacement for the #HildonWindow toggle_menu class function
 */
static gboolean (*toggle_menu_orig)(HildonWindow *self, guint button, guint32 time) = my_toggle_menu;
static gboolean
my_toggle_menu(HildonWindow *self, guint button, guint32 time)
{
	GtkMenu *menu = hildon_window_get_main_menu(self);

	if (menu) {
		pop_dialog(GTK_WINDOW(self), menu, FALSE);

		return TRUE;
	}

	return toggle_menu_orig(self, button, time);
}

/**
 * he_menu_view_get_menu_widget:
 * @hmv: The #HeMenuView
 *
 * Retrieve the #GtkMenu widget being displayed.
 *
 * Returns: The #GtkMenu being displayed
 *
 * Since: 0.9.1
 */
GtkWidget *
he_menu_view_get_menu_widget(HeMenuView *hmv)
{
	return hmv->priv->menu_widget;
}

/**
 * he_menu_view_set_do_activate:
 * @hmv: The #HeMenuView
 * @do_activate: Whether to activate chosen #GtkMenuItem widgets
 *
 * Set whether clicking a row in one of the #HeMenuViewColumn widgets activates the corresponding #GtkMenuItem.
 *
 * Since: 0.9.1
 */
void 
he_menu_view_set_do_activate(HeMenuView *hmv, gboolean do_activate)
{
	hmv->priv->do_activate = do_activate;
	g_queue_foreach(hmv->priv->menu_q, (GFunc)he_menu_view_column_set_do_activate, GINT_TO_POINTER(do_activate));
	g_object_notify(G_OBJECT(hmv), "do-activate");
}

/**
 * he_menu_view_get_do_activate:
 * @hmv: The #HeMenuView
 *
 * Retrieve whether clicking a row in one of the #HeMenuViewColumn widgets activates the corresponding #GtkMenuItem.
 *
 * Returns: Whether #GtkMenuItem widget activation is on
 *
 * Since: 0.9.1
 */
gboolean
he_menu_view_get_do_activate(HeMenuView *hmv)
{
	return hmv->priv->do_activate;
}

/**
 * he_menu_view_popup:
 * @menu: The #GtkMenu to display
 * @parent: The parent #GtkWindow, or %NULL
 *
 * Pop up a #GtkMenu as a #HeMenuView in a dialog
 *
 * Since: 0.9.1
 */
void
he_menu_view_popup(GtkMenu *menu, GtkWindow *parent)
{
	g_return_if_fail(menu != NULL);
	g_return_if_fail(GTK_IS_MENU(menu));
	if (parent)
		g_return_if_fail(GTK_IS_WINDOW(parent));

	pop_dialog(parent, menu, TRUE);
}

/**
 * he_menu_view_handle_hildon_windows:
 *
 * Add a hook to #HildonWindow that causes all #HildonWindow menus that are #GtkMenu widgets to be displayed as #HeMenuView widgets.
 * This does not affect the display of #HildonAppMenu widgets.
 *
 * Since: 0.9.1
 */
void 
he_menu_view_handle_hildon_windows()
{
	if (G_UNLIKELY(toggle_menu_orig == my_toggle_menu)) {
		toggle_menu_orig = HILDON_WINDOW_CLASS(g_type_class_peek(HILDON_TYPE_WINDOW))->toggle_menu;
		HILDON_WINDOW_CLASS(g_type_class_peek(HILDON_TYPE_WINDOW))->toggle_menu = my_toggle_menu;
	}
}

/**
 * he_menu_view_new:
 *
 * Creates a new #HeMenuView.
 *
 * Returns: A new #HeMenuView.
 *
 * Since: 0.9.1
 *
 */
GtkWidget *
he_menu_view_new()
{
	return g_object_new(HE_TYPE_MENU_VIEW, NULL);
}
