/*
 * This file is a part of hildon-extras
 *
 * Copyright (C) 2009 Andrew Olmsted
 * Copyright (C) 2005, 2008 Nokia Corporation.
 * Copyright (C) 2010 Timur Krist�f
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version. or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/**
 * SECTION:he-color-dialog
 * @short_description: A dialog to allow the user to select colors
 * @see_also: #HeColorSelector, #HeColorButton
 *
 * #HeColorDialog enables the user to select an arbitrary
 * color from a HSV colorspace.
 *
 */

#define _GNU_SOURCE     /* needed for GNU nl_langinfo_l */
#define __USE_GNU       /* needed for locale */

#include <locale.h>

#include <string.h>
#include <stdlib.h>

#include <libintl.h>
#include <langinfo.h>

#include "he-color-selector.h"
#include "he-color-dialog.h"

#define HE_COLOR_DIALOG_GET_PRIVATE(obj)                           \
  (G_TYPE_INSTANCE_GET_PRIVATE ((obj), HE_TYPE_COLOR_DIALOG, HeColorDialogPrivate))

#define _(String) dgettext("hildon-extras", String)
#define _HL(String) dgettext("hildon-libs", String)

struct _HeColorDialogPrivate
{
	HeColorSelector *selector;
};

/* static GObject * he_color_dialog_constructor (GType                  type,
                                                   guint                  n_construct_properties,
                                                   GObjectConstructParam *construct_properties); */

void he_color_dialog_response (GtkDialog *dialog, gint response_id, HeColorDialog *cd);
static void he_color_dialog_class_init (HeColorDialogClass * class);
static void he_color_dialog_init (HeColorDialog * cd);

static gpointer                                 parent_class = NULL;

GType G_GNUC_CONST
he_color_dialog_get_type            (void)
{
    static GType dialog_type = 0;
	dialog_type = g_type_from_name ("HeColorDialog");

    if (!dialog_type) {
        static const GTypeInfo dialog_info =
        {
            sizeof (HeColorDialogClass),
            NULL,
            NULL,
            (GClassInitFunc) he_color_dialog_class_init,
            NULL,
            NULL,
            sizeof (HeColorDialog),
            0,
            (GInstanceInitFunc) he_color_dialog_init,
            NULL
        };

        dialog_type = g_type_register_static (GTK_TYPE_DIALOG,
                "HeColorDialog", &dialog_info, 0);
    }

    return dialog_type;
}

/* private functions */
static void
he_color_dialog_class_init (HeColorDialogClass * class)
{
  //g_warning ("he_color_dialog_class_init");
  GObjectClass *gobject_class;
  GtkObjectClass *object_class;
  GtkWidgetClass *widget_class;

  gobject_class = (GObjectClass *) class;
  object_class = (GtkObjectClass *) class;
  widget_class = (GtkWidgetClass *) class;

  parent_class = g_type_class_peek_parent (class);

  g_type_class_add_private (object_class, sizeof (HeColorDialogPrivate));
}

static void
he_color_dialog_init (HeColorDialog *cd)
{
  //g_warning ("he_color_dialog_init");
  cd->priv = HE_COLOR_DIALOG_GET_PRIVATE (cd);

  gtk_dialog_set_has_separator (GTK_DIALOG (cd), FALSE);
  gtk_window_set_title (GTK_WINDOW (cd), _("Select color"));

  GtkWidget *content_area = gtk_dialog_get_content_area (GTK_DIALOG (cd));
  cd->priv->selector = HE_COLOR_SELECTOR (he_color_selector_new ());
  gtk_container_add (GTK_CONTAINER (content_area), GTK_WIDGET (cd->priv->selector));
  gtk_dialog_add_button (GTK_DIALOG (cd), _HL("wdgt_bd_done"), GTK_RESPONSE_ACCEPT);
  g_signal_connect (G_OBJECT (cd), "response", G_CALLBACK (he_color_dialog_response), cd);
  gtk_widget_show_all (content_area);

  GTK_WIDGET_SET_FLAGS (GTK_WIDGET (cd), GTK_NO_WINDOW);
  gtk_widget_set_redraw_on_allocate (GTK_WIDGET (cd), FALSE);
}

/* static void
he_color_dialog_finalize (GObject * object)
{
  //g_warning ("he_color_dialog_finalize");
  //HeColorDialogPrivate *priv = HE_COLOR_DIALOG_GET_PRIVATE (object);

  if (G_OBJECT_CLASS (parent_class)->finalize)
	G_OBJECT_CLASS (parent_class)->finalize (object);
} */

/* ------------------------------ PRIVATE METHODS ---------------------------- */
void
he_color_dialog_response (GtkDialog *dialog, gint response_id, HeColorDialog *cd)
{
    (void)dialog; // Supress the compiler message about the unused parameter.

    //g_warning ("he_color_dialog_response");
    if (response_id == GTK_RESPONSE_ACCEPT) {
        GdkColor color;
        he_color_selector_get_color (cd->priv->selector, &color);
        he_color_dialog_set_color (cd, &color);
    }
}


/* ------------------------------ PUBLIC METHODS ---------------------------- */

/**
 * he_color_dialog_new:
 *
 * Creates a new #HeColorDialog
 *
 * Returns: a new #HeColorDialog
 *
 * Since: 2.2
 **/
GtkWidget *
he_color_dialog_new ()
{
  //g_warning ("he_color_dialog_new");
  return g_object_new (HE_TYPE_COLOR_DIALOG, NULL);
}

/**
 * he_color_dialog_new_with_color:
 * @dcolor: the color to initialize to
 *
 * Creates a new #HeColorDialog with a specific color.
 *
 * Returns: a new #HeColorDialog
 *
 * Since: 2.2
 **/
GtkWidget *
he_color_dialog_new_with_color (GdkColor *color)
{
  //g_warning ("he_color_dialog_new_with_color");
  GtkWidget *cd = g_object_new (HE_TYPE_COLOR_DIALOG, NULL);
  he_color_dialog_set_color (HE_COLOR_DIALOG (cd), color);
  return cd;
}

/**
 * he_color_dialog_set_color:
 * @cd: the #HeColorDialog
 * @color: the color to set
 *
 * Sets the current color on the #HeColorDialog widget
 *
 * Since: 2.2
 *
 **/
void
he_color_dialog_set_color (HeColorDialog *cd,
                                          GdkColor *color)
{
	//g_warning ("he_color_dialog_set_color");
	g_return_if_fail (HE_IS_COLOR_DIALOG (cd));

	he_color_selector_set_color (cd->priv->selector, color);
}

/**
 * he_color_dialog_get_color:
 * @cd: the #HeColorDialog
 * @color: a #GdkColor to be filled with the current color
 *
 * Since: 2.2
 **/
void
he_color_dialog_get_color (HeColorDialog *cd, GdkColor *color)
{
  //g_warning ("he_color_dialog_get_color");
  g_return_if_fail (HE_IS_COLOR_DIALOG (cd));
  g_return_if_fail (color != NULL);

  he_color_selector_get_color (cd->priv->selector, color);
}

/**
 * he_color_dialog_present:
 * @parent: Optional. The parent window of the dialog, or %NULL.
 * @default_color: Optional. The default #GdkColor to be shown in the dialog, or %NULL.
 * @title: Optional. The title of the dialog window. (If %NULL, the default title is used.)
 *
 * Convenient method that presents the user with a #HeColorDialog.
 *
 * Returns: a #GdkColor* pointing to the selected color. If the dialog is cancelled, a copy of @default_color is returned.
 *          (Or %NULL, if @default_color is %NULL.)
 * The caller is responsible for freeing the result using gdk_color_free().
 **/
GdkColor*
he_color_dialog_present (GtkWindow *parent, GdkColor *default_color, gchar *title)
{
    /* Create the dialog */
    HeColorDialog* dialog = HE_COLOR_DIALOG (he_color_dialog_new());

    /* Set the parent (if not NULL) */
    if (parent != NULL) {
        if (GTK_IS_WINDOW (parent)) {
            gtk_window_set_transient_for (GTK_WINDOW (dialog), parent);
            gtk_window_set_destroy_with_parent(GTK_WINDOW(dialog), TRUE);
        }
        else if (GDK_IS_WINDOW (parent)) {
            gtk_widget_realize (GTK_WIDGET (dialog));
            gdk_window_set_transient_for(gtk_widget_get_window(GTK_WIDGET(dialog)), GDK_WINDOW(parent));
            gtk_window_set_destroy_with_parent(GTK_WINDOW(dialog), TRUE);
        }
        else
        {
            g_printerr ("The parent parameter is invalid.");
        }
    }

    /* Set the title
       (if it is NULL, we don't do anything as HeColorDialog already has a default title. */
    if (title != NULL) {
        gtk_window_set_title (GTK_WINDOW (dialog), title);
    }

    /* Set the selected color of the dialog to the default color (if not NULL) */
    if (default_color != NULL) {
        he_color_dialog_set_color (dialog, default_color);
    }

    /* Run the dialog */
    if (gtk_dialog_run (GTK_DIALOG (dialog)) != GTK_RESPONSE_ACCEPT) {
        /* If it is cancelled, release its memory... */
        gtk_widget_destroy (GTK_WIDGET (dialog));
        /* ... and return a copy of the default color */
        return (default_color == NULL ? NULL : gdk_color_copy (default_color));
    }

    /* Creating the result color */
    GdkColor* result = g_slice_new (GdkColor);
    /* Get the selected color from the dialog */
    he_color_dialog_get_color (dialog, result);
    /* Free up the memory associated with the dialog */
    gtk_widget_destroy (GTK_WIDGET (dialog));
    /* Return the resulting color */
    return result;
}

/**
 * he_color_dialog_present_interop:
 * @parent_winid: Optional. The parent native window of the dialog, or %0.
 * @default_color: Optional. The default #GdkColor to be shown in the dialog, or %NULL.
 * @title: Optional. The title of the dialog window. (If %NULL, the default title is used.)
 *
 * Convenient method that presents the user with a #HeColorDialog.
 *
 * Returns: a #GdkColor* pointing to the selected color. If the dialog is cancelled, a copy of @default_color is returned.
 *          (Or NULL, if @default_color is NULL.)
 * The caller is responsible for freeing the returned pointer's memory.
 *
 **/
GdkColor*
he_color_dialog_present_interop (GdkNativeWindow parent_winid, GdkColor *default_color, gchar *title)
{
    GdkWindow *foreign_window = NULL;

    if (parent_winid != 0)
        foreign_window = gdk_window_foreign_new (parent_winid);

    GdkColor *result = he_color_dialog_present ((GtkWindow*) foreign_window, default_color, title);

    if (foreign_window != NULL)
        g_object_unref(foreign_window);

    return result;
}
