/*
 * This file is a part of hildon-extras
 *
 * Copyright (C) 2010 Gabriel Schulhof <nix@go-nix.ca>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version. or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/**
 * SECTION:he-menu-store
 * @short_description: A #GtkTreeModel based on a #GtkMenu
 *
 * #HeMenuStore implements the #GtkTreeModel interface, allowing you to display the items from a #GtkMenu in a #GtkTreeView.
 */

#include <string.h>
#include "he-menu-store.h"
#include "he-menu-view_priv.h"

struct _HeMenuStorePrivate {
	GtkMenu *menu;
	gint stamp;
	gboolean is_deep;
};

/*
 * Iterator structure:
 * stamp:      menu_store->stamp
 * user_data:  menu_item
 * user_data2: unused
 * user_data3: unused
 */

typedef struct {
	gint stamp;
	GtkMenuItem *menu_item;
	void *user_data2;
	void *user_data3;
} MenuStoreIter;

enum {
	HE_MENU_STORE_MENU_PROPERTY = 1,
	HE_MENU_STORE_DEEP_PROPERTY,
};

#define IS_VALID_MENU_ITEM(widget) \
	(!(GTK_IS_SEPARATOR_MENU_ITEM((widget)) ||  \
	   GTK_IS_TEAROFF_MENU_ITEM((widget))   || \
	  (GTK_IS_MENU_ITEM((widget)) && (NULL == _hmv_private_menu_item_get_label(GTK_WIDGET((widget)), NULL)))))

static void he_menu_store_tree_model_init(GtkTreeModelIface *tree_model_iface);

G_DEFINE_TYPE_EXTENDED(HeMenuStore, he_menu_store, G_TYPE_OBJECT, 0, G_IMPLEMENT_INTERFACE(GTK_TYPE_TREE_MODEL, he_menu_store_tree_model_init));

static void menu_item_label_changed(GObject *label, GParamSpec *null, HeMenuStore *hms);
static void menu_item_changed(GObject *menu_item, GParamSpec *null, HeMenuStore *hms);
static void menu_item_toggled(GObject *menu_item, HeMenuStore *hms);

/*
 * Return the flags for the model
 */
static GtkTreeModelFlags
he_menu_store_get_flags(GtkTreeModel *tm)
{
	return GTK_TREE_MODEL_ITERS_PERSIST;
}

/*
 * Return the number of columns for the model
 */
static gint
he_menu_store_get_n_columns(GtkTreeModel *tm)
{
	return HE_MENU_STORE_N_COLUMNS;
}

/*
 * Get the type of the column at @index
 */
static GType
he_menu_store_get_column_type(GtkTreeModel *tm, gint index)
{
	return 
		HE_MENU_STORE_IMAGE_PIXBUF    == index ? GDK_TYPE_PIXBUF :
//	HE_MENU_STORE_IMAGE_GICON     == index ? G_TYPE_ICON     :
//	HE_MENU_STORE_IMAGE_ICON_NAME == index ? G_TYPE_STRING   :
//	HE_MENU_STORE_IMAGE_STOCK_ID  == index ? G_TYPE_STRING   :
//	HE_MENU_STORE_IMAGE_ICON_SIZE == index ? G_TYPE_UINT     :
	  HE_MENU_STORE_IS_IMAGE_ITEM   == index ? G_TYPE_BOOLEAN  :
	  HE_MENU_STORE_IS_CHECK_ITEM   == index ? G_TYPE_BOOLEAN  :
	  HE_MENU_STORE_IS_RADIO_ITEM   == index ? G_TYPE_BOOLEAN  :
		HE_MENU_STORE_TEXT            == index ? G_TYPE_STRING   :
		HE_MENU_STORE_MARKUP          == index ? G_TYPE_STRING   :
		HE_MENU_STORE_SENSITIVE       == index ? G_TYPE_BOOLEAN  :
		HE_MENU_STORE_ACTIVE          == index ? G_TYPE_BOOLEAN  :
		HE_MENU_STORE_INCONSISTENT    == index ? G_TYPE_BOOLEAN  :
		HE_MENU_STORE_VISIBLE         == index ? G_TYPE_BOOLEAN  :
		HE_MENU_STORE_HAS_CHILDREN    == index ? G_TYPE_BOOLEAN  :
//	HE_MENU_STORE_ACCEL           == index ? G_TYPE_STRING   :
		HE_MENU_STORE_MENU_WIDGET     == index ? G_TYPE_OBJECT   :
			G_TYPE_INVALID;
}

/*
 * Convert the index/indices of a #GtkTreePath into an iterator
 */
static gboolean
assign_iterator(HeMenuStore *hms, MenuStoreIter *itr, int depth, int idx, int *indices, GtkMenu *menu)
{
	GList *ll_children = gtk_container_get_children(GTK_CONTAINER(menu));
	GList *ll_itr;
	int Nix = 0;
	GtkWidget *submenu = NULL;
	gboolean ret = FALSE;

	for (ll_itr = ll_children ; ll_itr ; ll_itr = ll_itr->next)
		if (ll_itr->data)
			if (IS_VALID_MENU_ITEM(ll_itr->data)) {
				if (Nix == indices[idx])
					break;
				Nix++;
			}

	if (ll_itr) {
		if (idx == depth - 1) {
			itr->stamp = hms->priv->stamp;
			itr->menu_item = GTK_MENU_ITEM(ll_itr->data);
			ret = TRUE;
		}
		else
			submenu = gtk_menu_item_get_submenu(GTK_MENU_ITEM(ll_itr->data));
	}

	g_list_free(ll_children);
	return submenu ? assign_iterator(hms, itr, depth, idx + 1, indices, GTK_MENU(submenu)) : ret;
}

/*
 * Get the #GtkTreeIter corresponding to a #GtkTreePath
 */
static gboolean
he_menu_store_get_iter(GtkTreeModel *tm, GtkTreeIter *iter, GtkTreePath *tp)
{
	HeMenuStore *hms = HE_MENU_STORE(tm);
	gint *indices;
	gint depth;

	indices = gtk_tree_path_get_indices(tp);
	depth = gtk_tree_path_get_depth(tp);

	g_return_val_if_fail(depth > 0, FALSE);
	g_return_val_if_fail(depth < 2 || hms->priv->is_deep, FALSE);

	return assign_iterator(hms, (MenuStoreIter *)iter, depth, 0, indices, hms->priv->menu);
}

/*
 * Fill in the indices of a #GtkTreePath from a #GtkMenuItem
 */
static gboolean
fill_indices(GtkTreePath *tp, GtkMenuItem *menu_item, gboolean is_deep)
{
	int Nix;
	gboolean ret = FALSE;
	GtkMenuItem *attach_widget = NULL;
	GtkContainer *parent = GTK_CONTAINER(gtk_widget_get_parent(GTK_WIDGET(menu_item)));
	GList *ll_children = gtk_container_get_children(GTK_CONTAINER(parent)), *ll_itr;

	for (Nix = 0, ll_itr = ll_children ; ll_itr && menu_item != ll_itr->data ; ll_itr = ll_itr->next)
		if (ll_itr->data)
			if (IS_VALID_MENU_ITEM(ll_itr->data))
				Nix++;

	if (ll_itr) {
		attach_widget = _hmv_private_get_attach_widget(GTK_WIDGET(parent));

		gtk_tree_path_prepend_index(tp, Nix);
		ret = TRUE;
	}

	g_list_free(ll_children);
	return (attach_widget && is_deep) ? fill_indices(tp, attach_widget, is_deep) : ret;
}

/*
 * Get a #GtkTreePath from a given #GtkTreeIter
 */
static GtkTreePath *
he_menu_store_get_path(GtkTreeModel *tm, GtkTreeIter *iter)
{
	MenuStoreIter *msi = (MenuStoreIter *)iter;
	GtkMenuItem *menu_item = msi->menu_item;
	HeMenuStore *hms = HE_MENU_STORE(tm);
	GtkTreePath *tp;

	g_return_val_if_fail(iter->stamp == hms->priv->stamp, NULL);

	tp = gtk_tree_path_new();

	if (menu_item)
		if (!fill_indices(tp, menu_item, hms->priv->is_deep)) {
			gtk_tree_path_free(tp);
			tp = NULL;
		}

	return tp;
}

/*
 * Strip the underscores from a string in place
 */
static char *
strip_underscores(char *text)
{
	if (text) {
		char *itr, *end = &text[strlen(text)];

		for (itr = text ; itr < end ; )
			if ('_' == (*itr)) {
				if (itr == end - 1)
					(*itr) = 0;
				else
					memmove(itr, itr + 1, end - itr - 1);
				end--;
				(*end) = 0;
			}
			else
				itr++;
	}

	return text;
}

/*
 * Create a #GdkPixbuf from a #GtkImage
 *
 * We know how to get a #GdkPixbuf from a #GtkImage with the following #GtkImage storage types:
 *
 * [X] GTK_IMAGE_EMPTY,
 * [ ] GTK_IMAGE_PIXMAP,
 * [ ] GTK_IMAGE_IMAGE,
 * [X] GTK_IMAGE_PIXBUF,
 * [X] GTK_IMAGE_STOCK,
 * [ ] GTK_IMAGE_ICON_SET,
 * [ ] GTK_IMAGE_ANIMATION,
 * [ ] GTK_IMAGE_ICON_NAME,
 * [ ] GTK_IMAGE_GICON
 */

static GdkPixbuf *
image_get_pixbuf(GtkImage *image)
{
	GdkPixbuf *pb = NULL;

	switch (gtk_image_get_storage_type(image)) {
		case GTK_IMAGE_PIXBUF:
			pb = gtk_image_get_pixbuf(image);
			if (pb)
				pb = gdk_pixbuf_copy(pb);
			break;

		case GTK_IMAGE_STOCK:
			{
				char *stock_id;
				GtkIconSize size;

				gtk_image_get_stock(image, &stock_id, &size);
				if (stock_id)
					pb = gtk_widget_render_icon(GTK_WIDGET(image), stock_id, size, NULL);
			}
			break;

		default:
			break;
	}

	return pb;
}

static void
set_value_from_menu_item_string(GtkWidget *menu_item, const char *(*text_retr_function)(GtkLabel *), GValue *val)
{
	GtkWidget *label = NULL;

	_hmv_private_get_first_label_found(menu_item, &label);
	if (label) {
		char *text = NULL;

		text = g_strdup(text_retr_function(GTK_LABEL(label)));
		if (gtk_label_get_use_underline(GTK_LABEL(label)))
			text = strip_underscores(text);
		g_value_set_string(val, text);
		g_free(text);
	}
}

/*
 * Get a value for the given column of the given #GtkTreeIter
 */
static void
he_menu_store_get_value(GtkTreeModel *tm, GtkTreeIter *iter, gint column, GValue *val)
{
	HeMenuStore *hms = HE_MENU_STORE(tm);
	MenuStoreIter *msi = (MenuStoreIter *)iter;
	GtkMenuItem *menu_item = msi->menu_item;

	g_return_if_fail(hms->priv->stamp == iter->stamp);

	g_value_init(val, he_menu_store_get_column_type(tm, column));

	switch (column) {
#if (0)
		case HE_MENU_STORE_IMAGE_GICON:
			if (GTK_IS_IMAGE_MENU_ITEM(msi->menu_item)) {
				GtkWidget *image = gtk_image_menu_item_get_image(GTK_IMAGE_MENU_ITEM(msi->menu_item));
				if (image)
					g_object_get_property(G_OBJECT(image), "gicon", val);
			}
			break;

		case HE_MENU_STORE_IMAGE_ICON_NAME:
			if (GTK_IS_IMAGE_MENU_ITEM(msi->menu_item)) {
				GtkWidget *image = gtk_image_menu_item_get_image(GTK_IMAGE_MENU_ITEM(msi->menu_item));
				if (image)
					g_object_get_property(G_OBJECT(image), "icon-name", val);
			}
			break;

		case HE_MENU_STORE_IMAGE_STOCK_ID:
			if (GTK_IS_IMAGE_MENU_ITEM(msi->menu_item)) {
				GtkWidget *image = gtk_image_menu_item_get_image(GTK_IMAGE_MENU_ITEM(msi->menu_item));
				if (image) {
					char *stock;

					g_object_get(G_OBJECT(image), "stock", &stock, NULL);
					g_print("stock: %s\n", stock);
					g_free(stock);
					g_object_get_property(G_OBJECT(image), "stock", val);
				}
			}
			break;

		case HE_MENU_STORE_IMAGE_ICON_SIZE:
			if (GTK_IS_IMAGE_MENU_ITEM(msi->menu_item)) {
				GtkWidget *image = gtk_image_menu_item_get_image(GTK_IMAGE_MENU_ITEM(msi->menu_item));
				if (image)
					g_object_get_property(G_OBJECT(image), "icon-size", val);
			}
			break;
#endif /* (0) */
		case HE_MENU_STORE_IMAGE_PIXBUF:
			if (GTK_IS_IMAGE_MENU_ITEM(menu_item)) {
				GtkWidget *image = gtk_image_menu_item_get_image(GTK_IMAGE_MENU_ITEM(menu_item));
				if (image) {
					GdkPixbuf *pb = image_get_pixbuf(GTK_IMAGE(image));

					g_value_set_object(val, pb);
					g_object_unref(pb);
				}
			}
			break;

		case HE_MENU_STORE_IS_IMAGE_ITEM:
			g_value_set_boolean(val, GTK_IS_IMAGE_MENU_ITEM(menu_item));
			break;

		case HE_MENU_STORE_IS_CHECK_ITEM:
			g_value_set_boolean(val, GTK_IS_CHECK_MENU_ITEM(menu_item));
			break;

		case HE_MENU_STORE_IS_RADIO_ITEM:
			g_value_set_boolean(val, 
				 GTK_IS_RADIO_MENU_ITEM(menu_item) ||
				(GTK_IS_CHECK_MENU_ITEM(menu_item) && gtk_check_menu_item_get_draw_as_radio(GTK_CHECK_MENU_ITEM(menu_item))));
			break;

		case HE_MENU_STORE_SENSITIVE:
			g_object_get_property(G_OBJECT(menu_item), "sensitive", val);
			break;

		case HE_MENU_STORE_ACTIVE:
			if (GTK_IS_CHECK_MENU_ITEM(menu_item))
				g_object_get_property(G_OBJECT(menu_item), "active", val);
			else
				g_value_set_boolean(val, FALSE);
			break;

		case HE_MENU_STORE_INCONSISTENT:
			if (GTK_IS_CHECK_MENU_ITEM(menu_item))
				g_object_get_property(G_OBJECT(menu_item), "inconsistent", val);
			else
				g_value_set_boolean(val, FALSE);
			break;

		case HE_MENU_STORE_VISIBLE:
			g_object_get_property(G_OBJECT(menu_item), "visible", val);
			break;

		case HE_MENU_STORE_HAS_CHILDREN:
			g_value_set_boolean(val, (NULL != gtk_menu_item_get_submenu(menu_item)));
			break;

		case HE_MENU_STORE_TEXT:
			set_value_from_menu_item_string(GTK_WIDGET(menu_item), gtk_label_get_text, val);
			break;

		case HE_MENU_STORE_MARKUP:
			set_value_from_menu_item_string(GTK_WIDGET(menu_item), gtk_label_get_label, val);
			break;

		case HE_MENU_STORE_MENU_WIDGET:
			g_value_set_object(val, menu_item);
			break;
	}
}

/*
 * Advance the #GtkTreeIter to the next item
 */
static gboolean
he_menu_store_iter_next(GtkTreeModel *tm, GtkTreeIter *iter)
{
	HeMenuStore *hms = HE_MENU_STORE(tm);
	gboolean ret = FALSE;
	GList *ll_children, *ll_itr;
	MenuStoreIter *msi = (MenuStoreIter *)iter;
	GtkContainer *parent;

	g_return_val_if_fail(hms->priv->stamp == iter->stamp, FALSE);

	msi->stamp = 0;

	parent = GTK_CONTAINER(gtk_widget_get_parent(GTK_WIDGET(msi->menu_item)));

	ll_children = gtk_container_get_children(parent);
	ll_itr = g_list_find(ll_children, msi->menu_item);

	for (ll_itr = ll_itr->next; ll_itr; ll_itr = ll_itr->next)
		if (ll_itr)
			if (IS_VALID_MENU_ITEM(ll_itr->data)) {
				msi->stamp = hms->priv->stamp;
				msi->menu_item = ll_itr->data;
				ret = TRUE;
				break;
			}

	g_list_free(ll_children);

	return ret;
}

/*
 * Return whether a given #GtkTreeIter has children (a submenu)
 */
static gboolean
he_menu_store_iter_has_child(GtkTreeModel *tm, GtkTreeIter *iter)
{
	HeMenuStore *hms = HE_MENU_STORE(tm);
	g_return_val_if_fail(iter->stamp == hms->priv->stamp, FALSE);

	return hms->priv->is_deep && (gtk_menu_item_get_submenu(GTK_MENU_ITEM(((MenuStoreIter *)iter)->menu_item)) != NULL);
}

/*
 * Retrieve the number of children from a given #GtkTreeIter
 */
static gint
he_menu_store_iter_n_children(GtkTreeModel *tm, GtkTreeIter *iter)
{
	int ret = 0;
	HeMenuStore *hms = HE_MENU_STORE(tm);
	GtkWidget *submenu = NULL;
	MenuStoreIter *msi = (MenuStoreIter *)iter;

	if (iter)
		g_return_val_if_fail(iter->stamp == hms->priv->stamp, 0);

	submenu = msi ? gtk_menu_item_get_submenu(msi->menu_item) : GTK_WIDGET(hms->priv->menu);
	if (submenu) {
		GList *ll_children = gtk_container_get_children(GTK_CONTAINER(submenu));

		ret = g_list_length(ll_children);
		g_list_free(ll_children);
	}

	return ret;
}

/*
 * Get the nth child of a given #GtkTreeIter
 */
static gboolean
he_menu_store_iter_nth_child(GtkTreeModel *tm, GtkTreeIter *iter, GtkTreeIter *parent, gint n)
{
	HeMenuStore *hms = HE_MENU_STORE(tm);
	MenuStoreIter *msi = (MenuStoreIter *)iter, *msi_parent = (MenuStoreIter *)parent;
	gboolean ret = FALSE;
	GtkWidget *submenu;

	if (msi_parent != NULL)
		g_return_val_if_fail(msi_parent->stamp == hms->priv->stamp, FALSE);

	msi->stamp = 0;

	submenu = msi_parent ? gtk_menu_item_get_submenu(msi_parent->menu_item) : GTK_WIDGET(hms->priv->menu);

	if (submenu) {
			GList *ll_children = gtk_container_get_children(GTK_CONTAINER(submenu)), *ll_itr;
			int Nix;

			for (Nix = 0, ll_itr = ll_children; ll_itr && Nix < n ; ll_itr = ll_itr->next)
				if (ll_itr)
					if (IS_VALID_MENU_ITEM(ll_itr->data))
						Nix++;

			if (ll_itr) {
				if (Nix == n) {
					msi->stamp = hms->priv->stamp;
					msi->menu_item = GTK_MENU_ITEM(ll_itr->data);
					ret = TRUE;
				}
			}

			g_list_free(ll_children);
	}

	return ret;
}

/*
 * Move to the first child of the given #GtkTreeIter
 */
static gboolean
he_menu_store_iter_children(GtkTreeModel *tm, GtkTreeIter *iter, GtkTreeIter *parent)
{
	return he_menu_store_iter_nth_child(tm, iter, parent, 0);
}

/*
 * Move to the parent of the given #GtkTreeIter
 */
static gboolean
he_menu_store_iter_parent(GtkTreeModel *tm, GtkTreeIter *iter, GtkTreeIter *child)
{
	HeMenuStore *hms = HE_MENU_STORE(tm);
	gboolean ret = FALSE;

	if (hms->priv->is_deep) {
		GtkWidget *container;
		GtkMenuItem *attach_widget;
		MenuStoreIter *msi = (MenuStoreIter *)iter, *msi_child = (MenuStoreIter *)child;

		g_return_val_if_fail(child->stamp == hms->priv->stamp, FALSE);

		msi->stamp = 0;

		container = gtk_widget_get_parent(GTK_WIDGET(msi_child->menu_item));
		attach_widget = _hmv_private_get_attach_widget(container);
		if (attach_widget) {
			msi->stamp = msi_child->stamp;
			msi->menu_item = attach_widget;
			ret = TRUE;
		}
	}

	return ret;
}

/*
 * Initialize the #GtkTreeModel interface
 */
static void
he_menu_store_tree_model_init(GtkTreeModelIface *tree_model_iface)
{
	tree_model_iface->get_flags       = he_menu_store_get_flags;
	tree_model_iface->get_n_columns   = he_menu_store_get_n_columns;
	tree_model_iface->get_column_type = he_menu_store_get_column_type;
	tree_model_iface->get_iter        = he_menu_store_get_iter;
	tree_model_iface->get_path        = he_menu_store_get_path;
	tree_model_iface->get_value       = he_menu_store_get_value;
	tree_model_iface->iter_next       = he_menu_store_iter_next;
	tree_model_iface->iter_children   = he_menu_store_iter_children;
	tree_model_iface->iter_has_child  = he_menu_store_iter_has_child;
	tree_model_iface->iter_n_children = he_menu_store_iter_n_children;
	tree_model_iface->iter_nth_child  = he_menu_store_iter_nth_child;
	tree_model_iface->iter_parent     = he_menu_store_iter_parent;
}

/*
 * Initialize the #HeMenuStore class
 */
static void
he_menu_store_init(HeMenuStore *hms)
{
	hms->priv = G_TYPE_INSTANCE_GET_PRIVATE(hms, HE_TYPE_MENU_STORE, HeMenuStorePrivate);
	hms->priv->menu = NULL;
	hms->priv->is_deep = FALSE;
	for (hms->priv->stamp = g_random_int(); 0 == hms->priv->stamp; hms->priv->stamp = g_random_int());
}

/*
 * Remove tracking from a given #GtkMenuItem
 */
static void
unhook_signals(GObject *menu_item, HeMenuStore *hms)
{
	if (GPOINTER_TO_INT(g_object_get_data(G_OBJECT(menu_item), "visited"))) {
		GtkWidget *label = NULL;

		_hmv_private_get_first_label_found(GTK_WIDGET(menu_item), &label);
		g_object_set_data(menu_item, "visited", FALSE);
		g_signal_handlers_disconnect_by_func(menu_item, (GCallback)menu_item_changed, hms);
		if (label)
			g_signal_handlers_disconnect_by_func(label, (GCallback)menu_item_label_changed, hms);
		if (GTK_IS_RADIO_MENU_ITEM(menu_item))
			g_signal_handlers_disconnect_by_func(menu_item, (GCallback)menu_item_toggled, hms);
		if (hms->priv->is_deep) {
			GtkWidget *submenu = gtk_menu_item_get_submenu(GTK_MENU_ITEM(menu_item));

			if (submenu)
				gtk_container_foreach(GTK_CONTAINER(submenu), (GtkCallback)unhook_signals, hms);
		}
	}
}

/*
 * Re-index menu items
 */
static void
re_index(HeMenuStore *hms)
{
	int Nix = 0;
	GList *ll = gtk_container_get_children(GTK_CONTAINER(hms->priv->menu));

	for (; ll; ll = g_list_delete_link(ll, ll))
		if (ll->data)
			if (IS_VALID_MENU_ITEM(ll->data))
				g_object_set_data(G_OBJECT(ll->data), "idx", GINT_TO_POINTER(Nix++));
}

/*
 * Notification of removal of a given #GtkMenuItem via #GtkContainer::remove
 */
static void
menu_item_removed(GtkContainer *container, GObject *menu_item, HeMenuStore *hms)
{
	if (IS_VALID_MENU_ITEM(menu_item)) {
		GtkTreePath *tp = gtk_tree_path_new_from_indices(GPOINTER_TO_INT(g_object_get_data(menu_item, "idx")), -1);

		if (tp) {
			gtk_tree_model_row_deleted(GTK_TREE_MODEL(hms), tp);
			gtk_tree_path_free(tp);
		}

		unhook_signals(menu_item, hms);
		re_index(hms);
	}
}

static void
menu_item_label_changed(GObject *label, GParamSpec *null, HeMenuStore *hms)
{
	GObject *menu_item = NULL;
	GtkWidget *parent = NULL;

	do {
		parent = gtk_widget_get_parent(GTK_WIDGET(label));
		if (parent) {
			if (GTK_IS_MENU_ITEM(parent))
				menu_item = G_OBJECT(parent);
		}
		else
			break;
	} while (menu_item == NULL);

	if (menu_item)
		menu_item_changed(menu_item, NULL, hms);
}

/*
 * Notification of change to a given #GtkMenuItem via #GObject::notify
 */
static void
menu_item_changed(GObject *menu_item, GParamSpec *null, HeMenuStore *hms)
{
	GtkTreePath *tp = NULL;
	MenuStoreIter msi = {
		.stamp     = hms->priv->stamp,
		.menu_item = GTK_MENU_ITEM(menu_item)
	};

	tp = he_menu_store_get_path(GTK_TREE_MODEL(hms), (GtkTreeIter *)&msi);

	if (tp) {
		gtk_tree_model_row_changed(GTK_TREE_MODEL(hms), tp, (GtkTreeIter *)&msi);
		gtk_tree_path_free(tp);
	}
}

/*
 * Notification of a #GtkRadioMenuItem having been toggled
 *
 * Unfortunately radio menu items do not emit notify::active when toggled, so we need to do this manually
 */
static void
menu_item_toggled(GObject *menu_item, HeMenuStore *hms)
{
	menu_item_changed(menu_item, NULL, hms);
}

/*
 * Add tracking to the items of a #GtkMenu
 */
static void
hook_up_signals(GtkMenu *menu, HeMenuStore *hms)
{
	GObject *label = NULL;
	GList *ll = gtk_container_get_children(GTK_CONTAINER(menu));
	int Nix = 0;

	for (; ll ; ll = g_list_delete_link(ll, ll))
		if (ll->data)
			if (IS_VALID_MENU_ITEM(ll->data)) {
				if (!GPOINTER_TO_INT(g_object_get_data(G_OBJECT(ll->data), "visited"))) {
					g_object_set_data(G_OBJECT(ll->data), "visited", GINT_TO_POINTER(TRUE));
					g_object_set_data(G_OBJECT(ll->data), "idx", GINT_TO_POINTER(Nix));
					g_signal_connect(ll->data, "notify", (GCallback)menu_item_changed, hms);
					label = NULL;
					_hmv_private_get_first_label_found(GTK_WIDGET(ll->data), (GtkWidget **)&label);
					if (label)
						g_signal_connect(label, "notify", (GCallback)menu_item_label_changed, hms);
					if (GTK_IS_RADIO_MENU_ITEM(ll->data))
						g_signal_connect(ll->data, "toggled", (GCallback)menu_item_toggled, hms);
					if (hms->priv->is_deep) {
						GtkWidget *submenu = gtk_menu_item_get_submenu(GTK_MENU_ITEM(ll->data));

						if (submenu)
							hook_up_signals(GTK_MENU(submenu), hms);
					}
				}
				Nix++;
			}
}

/*
 * Notification of the addition of a #GtkMenuItem via #GtkContainer::add or the #GtkMenuShell insert override
 */
static void
menu_item_added(GtkMenu *menu, GtkWidget *menu_item, HeMenuStore *hms)
{
	if (IS_VALID_MENU_ITEM(menu_item)) {
		MenuStoreIter msi = {
			.stamp = hms->priv->stamp,
			.menu_item = GTK_MENU_ITEM(menu_item),
			.user_data2 = NULL,
			.user_data3 = NULL
		};
		GtkTreePath *tp = he_menu_store_get_path(GTK_TREE_MODEL(hms), (GtkTreeIter *)&msi);

		hook_up_signals(menu, hms);

		if (tp)
			gtk_tree_model_row_inserted(GTK_TREE_MODEL(hms), tp, (GtkTreeIter *)&msi);
		re_index(hms);
	}
}

/*
 * Start serving up the given #GtkMenu as a #GtkTreeModel
 */
static void
set_menu(HeMenuStore *hms, GtkMenu *menu)
{
	if (menu)
		g_return_if_fail(GTK_IS_MENU(menu));

	if (hms->priv->menu != menu) {
		if (hms->priv->menu) {
			GList *ll = gtk_container_get_children(GTK_CONTAINER(hms->priv->menu));
			g_object_set_data(G_OBJECT(hms->priv->menu), "he-menu-store", NULL);
			g_signal_handlers_disconnect_by_func(G_OBJECT(hms->priv->menu), menu_item_removed, hms);
			g_signal_handlers_disconnect_by_func(G_OBJECT(hms->priv->menu), menu_item_added, hms);
			for (; ll; ll = g_list_delete_link(ll, ll))
				if (ll->data)
					if (IS_VALID_MENU_ITEM(ll->data))
						unhook_signals(G_OBJECT(ll->data), hms);
			g_object_unref(G_OBJECT(hms->priv->menu));
			hms->priv->menu = NULL;
		}

		if (menu) {
			hms->priv->menu = g_object_ref_sink(menu);
			g_object_set_data(G_OBJECT(menu), "he-menu-store", hms);
			g_signal_connect(G_OBJECT(menu), "add", (GCallback)menu_item_added, hms);
			g_signal_connect(G_OBJECT(menu), "remove", (GCallback)menu_item_removed, hms);
			hook_up_signals(menu, hms);
		}
		g_object_notify(G_OBJECT(hms), "menu");
	}
}

/*
 * Entry point for property changes
 */
static void
set_property(GObject *obj, guint property_id, const GValue *val, GParamSpec *pspec)
{
	switch (property_id) {
		case HE_MENU_STORE_MENU_PROPERTY:
			set_menu(HE_MENU_STORE(obj), GTK_MENU(g_value_get_object(val)));
			break;

		case HE_MENU_STORE_DEEP_PROPERTY:
			HE_MENU_STORE(obj)->priv->is_deep = g_value_get_boolean(val);
			break;

		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(obj, property_id, pspec);
			break;
	}
}

/*
 * Entry point for property value retrievals
 */
static void
get_property(GObject *obj, guint property_id, GValue *val, GParamSpec *pspec)
{
	switch (property_id) {
		case HE_MENU_STORE_MENU_PROPERTY:
			g_value_set_object(val, G_OBJECT(HE_MENU_STORE(obj)->priv->menu));
			break;

		case HE_MENU_STORE_DEEP_PROPERTY:
			g_value_set_boolean(val, HE_MENU_STORE(obj)->priv->is_deep);
			break;

		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(obj, property_id, pspec);
			break;
	}
}

/*
 * Clean up this instance
 */
static void
finalize(GObject *obj)
{
	set_menu(HE_MENU_STORE(obj), NULL);
}

/*
 * Override function for #GtkMenuShell::insert
 *
 * We need to override #GtkMenuShell::insert, because we get no notification of #GtkMenuItem additions. We only get
 * notifications of removals via #GtkContainer::remove.
 */
static void
he_menu_store_class_gtk_menu_shell_insert(GtkMenuShell *menu_shell, GtkWidget *widget, gint index)
{
	HeMenuStore *hms = g_object_get_data(G_OBJECT(menu_shell), "he-menu-store");
	HeMenuStoreClass *menu_store_class = g_type_class_peek(HE_TYPE_MENU_STORE);

	if (menu_store_class)
		if (menu_store_class->gtk_menu_shell_insert)
			menu_store_class->gtk_menu_shell_insert(menu_shell, widget, index);

	if (hms)
		menu_item_added(GTK_MENU(menu_shell), widget, hms);
}

/*
 * Initialize the #HeMenuStore class
 */
static void
he_menu_store_class_init(HeMenuStoreClass *menu_store_class)
{
	GtkMenuShellClass *gtk_menu_shell_class = GTK_MENU_SHELL_CLASS(g_type_class_ref(GTK_TYPE_MENU_SHELL));
	GObjectClass *gobject_class = G_OBJECT_CLASS(menu_store_class);

	gobject_class->finalize     = finalize;
	gobject_class->set_property = set_property;
	gobject_class->get_property = get_property;

	g_object_class_install_property(gobject_class, HE_MENU_STORE_MENU_PROPERTY,
		g_param_spec_object("menu", "Menu", "The menu widget", GTK_TYPE_MENU,
			G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY));

	g_object_class_install_property(gobject_class, HE_MENU_STORE_DEEP_PROPERTY,
		g_param_spec_boolean("deep", "Deep", "Whether this model reveals submenus as children", FALSE,
			G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY));

	g_type_class_add_private(menu_store_class, sizeof(HeMenuStorePrivate));

	if (gtk_menu_shell_class) {
		menu_store_class->gtk_menu_shell_insert = gtk_menu_shell_class->insert;
		gtk_menu_shell_class->insert = he_menu_store_class_gtk_menu_shell_insert;
	}
}

/**
 * he_menu_store_new:
 * @menu: The #GtkMenu to wrap
 *
 * Creates a new #HeMenuStore from the given #GtkMenu.
 *
 * Returns: A new #HeMenuStore based on the given #GtkMenu
 *
 * Since: 0.9.1
 */
GtkTreeModel *
he_menu_store_new(GtkMenu *menu)
{
	return g_object_new(HE_TYPE_MENU_STORE, "menu", menu, NULL);
}
