/*
 * driver.c -  MySocials driver
 * This file is part of MSA program.
 *
 * Copyright (C) 2009, 2010, 2011 - MySocials team
 *
 * MSA program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * MSA program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with MSA program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor,
 * Boston, MA  02110-1301  USA
 */

#include "driver.h"

#ifdef DEBUG
#	define msa_xmlDocDump(F,D) xmlDocDump(F,D)
#else
#	define msa_xmlDocDump(F,D)
#endif

#define XSL_CONFIG_DIR DATADIR"/"
#define NOT_ESCAPE "!\"#$%&'()*+,-./0123456789:;<=>?@ABCDEFGHIJKLMNOPQRSTUVWXYZ[]^_`abcdefghijklmnopqrstuvwxyz{|}~"


#define REGEXP_PARAM "regexp"
#define TITLE_PARAM "title"
#define CLIENT_ID_PARAM "ClientID"
#define ACCOUNT_ID_PARAM "accountID_RET"

#ifdef TEST
#include "../test/CTF/src/ctf.h"
CTF_TEST_DATA_FILE_NAME_DECL;
#endif

static int global_req = 0;
static int global_get = 0;


/**
 * Allocate memory
 *
 * @param ptr - pointer
 * @param size - size of data
* @return 0 on success or 1 otherwise.
 */
void* myrealloc(void* ptr, size_t size)
{
/* There might be a realloc() out there that doesn't like reallocing
NULL pointers, so we take care of it here */
    if(ptr)
        return realloc(ptr, size);
    else
        return malloc(size);
}

/**
 * Callbak for libcurl for receiving data
 *
 * @param ptr - pointer
 * @param size - size of chunk of data
 * @param nmemb - number of chunks of data
 * @param data - pointer to data
* @return 0 on success or 1 otherwise.
 */
 static size_t
WriteMemoryCallback(void* ptr, size_t size, size_t nmemb, void* data)
{
    size_t realsize = size * nmemb;
    struct MemoryStruct *mem = (struct MemoryStruct *)data;
    mem->memory = myrealloc(mem->memory, mem->size + realsize + 1);
    if (mem->memory) {
        memcpy(&(mem->memory[mem->size]), ptr, realsize);
        mem->size += realsize;
        mem->memory[mem->size] = 0;
    }
    return realsize;
}


/*
 * Process all internal requests in xml request
 *
 * @param pr - profile data
 * @param inner_req - xml request
 * @return 0 on success, 1 otherwise
 */
int process_inner_request(struct ProfileData *pr, xmlDocPtr inner_req)
{   
    int ret_code = 0;
    xmlDocPtr inner_request = NULL, inner_response = NULL;
    xmlNodePtr node = NULL, node_request = NULL, tmp_node = NULL;
    msa_debug("START process_inner_request");
    msa_xmlDocDump(stderr, inner_req);
    while ((node = xpath_get_node(XPATH_TAG_REQUEST, inner_req)) != NULL){
        node_request = xmlCopyNode(node, 1);
        xmlUnlinkNode(node_request);
        inner_request = xmlNewDoc(BAD_CAST "1.0");
        xmlDocSetRootElement(inner_request, node_request);
        
        if(process_request(pr, inner_request, &inner_response) == 1)
            error_message(DRV_REQ_ERROR_CODE, &inner_response, DRV_REQ_ERROR, "can't process inner request");

        msa_debug("INNER REQUEST\n");
        msa_xmlDocDump(stderr, inner_request);
        msa_debug("INNER RESPONSE\n");
        msa_xmlDocDump(stderr, inner_response);
        /*TODO        
          Check if error code = 6 sleep(1) and try request
        */
        tmp_node =  xpath_get_node(XPATH_TAG_ERROR_CODE, inner_response);
        if (tmp_node != NULL){
            gchar* code = NULL;
            code = (gchar*)xmlNodeGetContent(tmp_node);
            if(g_strcmp0(code, DRV_SERVER_BUSY_CODE) == 0){
                msa_debug("Error: Too many requests per second");
                xmlFreeDoc(inner_response);
                inner_response = NULL;
                msa_debug("Sleep");
                sleep(1);
                if(process_request(pr, inner_request, &inner_response) == 1)
                    error_message(DRV_REQ_ERROR_CODE, &inner_response, DRV_REQ_ERROR, "can't process inner request");
                msa_debug("INNER RESPONSE\n");
                msa_xmlDocDump(stderr, inner_response);
            }
            g_free(code);
        }
       xmlFreeDoc(inner_request); 

        if (xpath_get_node(XPATH_TAG_RESPONSE, inner_response) != NULL){
            node = xmlDocGetRootElement(inner_req);
            ret_code = 1;
        }

        tmp_node = xmlCopyNode(xmlDocGetRootElement(inner_response),1);
        xmlFreeDoc(inner_response);
        inner_response = NULL;
        node = xmlReplaceNode(node, tmp_node);
        xmlFreeNode(node);
        if (ret_code)
            break;
    }
    return ret_code;
}

/*
 * Recursively process request 
 *
 * @param pr - profile data
 * @param inner_req - xml request
 * @return 0 on success or 1 otherwise.
 */

int process_request(struct ProfileData *pr, xmlDocPtr request, xmlDocPtr* response)
{   
    msa_debug("\n\nRequest in process");
    msa_xmlDocDump(stdout, request);

    xmlDocPtr inner_req = NULL;
    xmlDocPtr tmp_doc = NULL;
    xmlDocPtr server_response = NULL, resp = NULL, err_resp = NULL;
    xmlNodePtr node = NULL, params_node = NULL, req_node = NULL, resp_node = NULL, tmp_node = NULL;
    gchar* parser = NULL;
    gchar* parser_type = NULL;
    gchar *url = NULL;
    gchar *tmp = NULL;
    gchar *resp_str = NULL;
    gchar* data = NULL;
    gchar* http_type = NULL;
    gchar* params = NULL;
    gchar* encoded_params = NULL;
    gchar** post_data = NULL;
    int http_code = 0;
    int resp_code = 0;
    int data_size = 0;

    msa_debug("START process_request");
    msa_xmlDocDump(stdout, request);
    if (xpath_get_node(XPATH_TAG_REQUEST, request) != NULL) {
        node = xpath_get_node("//Request/parser", request);

        if (node != NULL){
            tmp = (gchar*)xmlNodeGetContent(node);
            parser = g_strconcat(XSL_CONFIG_DIR, tmp, NULL);
            g_free(tmp);
        } else {          
            /*use standart parser*/
            parser = g_strconcat(XSL_CONFIG_DIR, "parser.xsl", NULL);       
        }

        /*create inner request*/
        inner_req = apply_parser(request, parser, pr);
        g_free(parser);
        msa_xmlDocDump(stdout, inner_req);
        
        /*unknown request */
        if (xpath_get_node("//InnerRequest", inner_req) == NULL){
            *response = xmlCopyDoc(inner_req, 1);
            xmlFreeDoc(inner_req);
            return 0;
        }
        if ((resp_code = process_inner_request(pr, inner_req)) == 1){
            *response = xmlCopyDoc(inner_req, 1);
            xmlFreeDoc(inner_req);
            return 0;
        }


        node = xpath_get_node("//InnerRequest", inner_req);

        http_type = (gchar*)xmlGetProp(node, BAD_CAST "type");

        if (g_strcmp0(http_type, "upload") == 0 
            || g_strcmp0(http_type, "get") ==0 
            || g_strcmp0(http_type, "post") ==0 
            || g_strcmp0(http_type, "download") ==0 
            || g_strcmp0(http_type, "webauth") == 0) {

            node = xpath_get_node("//InnerRequest/url", inner_req);
            if (node != NULL) {
                tmp = (gchar*)xmlNodeGetContent(node);
            } else {
                xmlFreeDoc(inner_req);
                g_free(http_type);
                error_message(DRV_REQ_ERROR_CODE, response, DRV_REQ_ERROR, "can't make url for request");
                return 0;
            }

            params_node = xpath_get_node("//InnerRequest/params", inner_req);
            if (params_node != NULL){
                params = (gchar*)xmlNodeGetContent(params_node);
                encoded_params = g_uri_escape_string(params, NOT_ESCAPE, 0); 
                g_free(params);               
                url = g_strconcat(tmp, encoded_params, NULL);
                g_free(encoded_params);
            } else
                url = g_strdup(tmp);
            g_free(tmp);

            process_url(inner_req, pr, url, 0, &tmp);
            g_free(url);
            url = g_strdup(tmp);
            g_free(tmp);

            msa_debug("URL = %s\n", url);

            if(g_strcmp0(http_type, "webauth") == 0){
                data = call_webauth(url, pr);
            } else if(g_strcmp0(http_type, "get") == 0 || g_strcmp0(http_type, "download") == 0){
                http_code = http_get(url, info_proxy, info_port, NULL, &data, &data_size);
            } else if(g_strcmp0(http_type, "post") == 0){
                post_data = g_strsplit(url,"?",2);
                http_code = http_post(post_data[0], info_proxy, info_port, post_data[1], NULL, &data, &data_size);
                g_strfreev(post_data);
            } else if(g_strcmp0(http_type, "upload") == 0){
                msa_debug("UPLOADING...");
                struct UpFile *file = (struct UpFile *)malloc(sizeof (struct UpFile));
                node = xpath_get_node("//InnerRequest/name", inner_req);
                file->name = (gchar*)xmlNodeGetContent(node);
                node = xpath_get_node("//InnerRequest/file", inner_req);
                file->file_path = (gchar*)xmlNodeGetContent(node);
                node = xpath_get_node("//InnerRequest/file_name", inner_req);
                file->file_name = (gchar*)xmlNodeGetContent(node);
                
                post_data = g_strsplit_set(url, "?", 2);
                if(post_data[1] != NULL)
                    file->params = g_strsplit_set(post_data[1], "&=", 100);
                else
                    file->params = NULL;
                http_code = http_upload_file(post_data[0], info_proxy, info_port, NULL, &data, pr, file);
                upfile_remove(file);
                g_strfreev(post_data);
            }
            g_free(url);

            if (http_code >= 400 && http_code < 500){                                        
                tmp = g_strdup_printf("http code %d", http_code);
                error_message(DRV_NETW_ERROR_CODE, &err_resp, DRV_NETW_ERROR, tmp);
                g_free(tmp);
            } 
            if (http_code >= 500){
                tmp = g_strdup_printf("http code %d", http_code);
                error_message(DRV_SERVER_ERROR_CODE, &err_resp, DRV_SERVER_ERROR, tmp);
                g_free(tmp);
            }
        }
        g_free(http_type);

        msa_xmlDocDump(stdout, inner_req);

        /*Parse server response*/
        node = xpath_get_node("//InnerRequest/parser", inner_req);
        if (node != NULL){
            parser_type = (gchar*)xmlGetProp(node, BAD_CAST "type");
            tmp = (gchar*)xmlNodeGetContent(node);

            if(http_code < 300 || (http_code == 400 && g_strcmp0(parser_type, PARSER_JSON) == 0)){
                if (g_strcmp0(parser_type, PARSER_JSON) == 0) {
                    json_parser(inner_req, pr, data, 0, &resp_str);
                    g_free(data);
                    data = g_strdup(resp_str);
                    g_free(resp_str);
                } else if (g_strcmp0(parser_type, PARSER_SETTINGS) == 0) {
                    settings_parser(inner_req, pr, data, 0, &resp_str);
                    g_free(data);
                    data = g_strdup(resp_str);
                    g_free(resp_str);
                } else if (g_strcmp0(parser_type, PARSER_EMPTY) == 0) {
                    empty_parser(inner_req, pr, data, 0, &resp_str);
                    data = g_strdup(resp_str);
                    g_free(resp_str);
                } else if (g_strcmp0(parser_type, PARSER_DOWNLOAD) == 0) {
                    download_parser(inner_req, pr, data, data_size, &resp_str);
                    g_free(data);
                    data = g_strdup(resp_str);
                    g_free(resp_str);
                } else if (g_strcmp0(parser_type, PARSER_TEXT) == 0) {
                    text_parser(inner_req, pr, data, 0, &resp_str);
                    g_free(data);
                    data = g_strdup(resp_str);
                    g_free(resp_str);
                } else if (g_strcmp0(parser_type, PARSER_WEBAUTH) == 0) {
                    webauth_parser(inner_req, pr, data, 0, &resp_str);
                    g_free(data);
                    data = g_strdup(resp_str);
                    g_free(resp_str);
                }

            }
            server_response = xmlParseDoc((xmlChar*)data);
            if (server_response != NULL){
                tmp_doc = xmlNewDoc(BAD_CAST "1.0");
                node = xmlNewNode(NULL, (xmlChar*)"data");
                xmlDocSetRootElement(tmp_doc, node);

                req_node = xmlDocGetRootElement(request);
                tmp_node = xmlCopyNode(req_node, 1);
                xmlUnlinkNode(tmp_node);
                xmlAddChild(node, tmp_node);

                resp_node = xmlDocGetRootElement(server_response); 
                tmp_node = xmlCopyNode(resp_node, 1);
                xmlUnlinkNode(tmp_node);
                xmlAddChild(node, tmp_node);

                parser = g_strconcat(XSL_CONFIG_DIR, tmp, NULL);
                msa_xmlDocDump(stdout, tmp_doc);
                resp = apply_parser(tmp_doc, parser, pr);
                xmlFreeDoc(server_response);
                xmlFreeDoc(tmp_doc);
                g_free(parser);
            } 
            g_free(tmp);
            g_free(parser_type);
            g_free(data);
            xmlFreeDoc(inner_req);

        }
        /*Search for requests*/
        if (resp == NULL && err_resp != NULL)
            resp = xmlCopyDoc(err_resp, 1);
        if (resp != NULL){
            process_inner_request(pr, resp);
            *response = xmlCopyDoc(resp, 1);
        } else
            error_message(DRV_REQ_ERROR_CODE, response, DRV_REQ_ERROR, "can't parse server response");

        if (err_resp != NULL)
            xmlFreeDoc(err_resp);
        if (resp != NULL)
            xmlFreeDoc(resp);
        return 0;
    }

    return 1;
}


xmlDocPtr apply_parser(xmlDocPtr request, gchar* parser, struct ProfileData *pr)
{
    msa_debug("Aply parser: %s", parser);
    xmlDocPtr proc = NULL; 
    xsltStylesheetPtr cur = NULL;
    gchar **params;
    cur = xsltParseStylesheetFile((xmlChar*)parser);

    if(pr != NULL){
        params = profile_get_params(pr);
    } else
        return NULL;
    if(cur != NULL && request != NULL) {
        proc = xsltApplyStylesheet(cur, request, (const char**)params);
        xsltFreeStylesheet(cur);
    }
    g_strfreev(params);
    return proc;
}


/**
 * Perform request
 *
 * @param reqXml - request
 * @param doc - response
 * @return 0 on success or 1 otherwise.
 */
int request(xmlDocPtr reqXml, xmlDocPtr* doc, const struct msa_module* info)
{
    int init_status = 0;
    char* error_code = NULL;
    gchar* func_val = NULL;
    xmlNodePtr node = NULL;
    gchar* no_auth = NULL;

    struct ProfileData *cp = (struct ProfileData *)profile_get_by_id(info->id);
    msa_debug("%s:%d:%s: id=%s", __FILE__, __LINE__, __FUNCTION__, info->id);
    info_proxy = info->proxy;
    info_port = info->port;

    global_req++;
    
    msa_debug("%s:%d:%s(): REQ %d\n", __FILE__, __LINE__, __FUNCTION__, global_req);
    msa_xmlDocDump(stdout, reqXml);
    node = xpath_get_node(XPATH_TAG_REQUEST, reqXml);
    if(node != NULL)
        no_auth = (gchar*)xmlGetProp(node, BAD_CAST "noAuthorize");
    else {
        error_message(DRV_REQ_ERROR_CODE, doc, DRV_REQ_ERROR, "request not found");
        return 0;
    }
    func_val = (gchar*)get_req_function(reqXml);
    if (g_strcmp0(func_val, SET_SETTINGS) != 0
        && g_strcmp0(func_val, GET_SETTINGS) != 0)
        if(no_auth == NULL || g_strcmp0(no_auth, "true") != 0)
            init_status = initialization(cp);
    g_free(func_val);

    if (init_status != 0) {
        msa_debug("init_status = %d", init_status);
        if (init_status == 2){
            error_message(DRV_NETW_ERROR_CODE, doc, DRV_NETW_ERROR, "");
            return 0;
        }
        if(init_status == 1) {
            error_message(DRV_AUTH_ERROR_CODE, doc, DRV_AUTH_ERROR, "");
            return 0;
        }
    }
    if (process_request(cp, reqXml, doc) == 1){
        error_message(DRV_REQ_ERROR_CODE, doc, DRV_REQ_ERROR, "");
        return 0;
    }

    node = xpath_get_node(XPATH_TAG_NOTERROR, *doc);
    if (node != NULL){
        if(init_status == 3) {
            xmlSetProp(node, BAD_CAST "authorized", BAD_CAST "true");
        }
        if(init_status == 0) {
            xmlSetProp(node, BAD_CAST "authorized", BAD_CAST "false");
        }
    }                 

    node = xpath_get_node(XPATH_TAG_ERROR_CODE, *doc);
    if (node != NULL){
        msa_debug("error occured");
        error_code = (gchar *)xmlNodeGetContent(node);
        if(g_strcmp0(error_code, DRV_AUTH_ERROR_CODE) == 0){
            if(no_auth == NULL || g_strcmp0(no_auth, "true") != 0) {
                if (authorize(cp) == 0) {
                    xmlFreeDoc(*doc);
                    process_request(cp, reqXml, doc);
                    node = xpath_get_node(XPATH_TAG_NOTERROR, *doc);
                    if (node != NULL)
                        xmlSetProp(node, BAD_CAST "authorized", BAD_CAST "true");
                }
            }
        } else if(g_strcmp0(error_code, DRV_SERVER_BUSY_CODE) == 0){
            xmlFreeDoc(*doc);
            sleep(1);
            process_request(cp, reqXml, doc);
            if(init_status == 3) {
                xmlSetProp(node, BAD_CAST "authorized", BAD_CAST "true");
            }
        }

        g_free(error_code);
            
    }
    if(no_auth != NULL)
        g_free(no_auth);
    return 0;
}

/**
 * Perform authorization
 *
 * @param cp - current profile settings
 * @return 0 on success or 1 otherwise.
 */
int initialization(struct ProfileData *cp)
{
    msa_debug("init START");

    int auth_status = 0;

    if(profile_check(cp) == 1) {

        g_print("%s: authorization\n", DRIVER_ID);
        auth_status = authorize(cp);
        if (auth_status != 0 ) {
            msa_debug("dr_fb: Could not get remixmid, remixsid, auth_status = %d \n", auth_status); 
            return auth_status;
        }
        return 3;
    }
    msa_debug("init END");
    return 0;
}

/**
 * Authorization (get remixsid and remixmid for future requests)
 *
 * @param cp - current profile
 * @return 0 on success or 1 otherwise.
 */
int authorize(struct ProfileData *cp)
{
    int ret_code = 0;

    if (cp == NULL){
        msa_debug("can't authorize, no profile!");
        return 1;
    }

    xmlDocPtr request = NULL, response = NULL;
    request = xmlParseDoc((xmlChar*)AUTH_REQUEST);    
    if(process_request(cp, request, &response) == 1){
        xmlFreeDoc(request);
        return 1;
    }

    ret_code = profile_update(cp, response);

    xmlFreeDoc(request);
    xmlFreeDoc(response);        
    return ret_code;
}


/**
 * call webauth application
 *
 * @param cp - current profile
 * @return 0 on success or 1 otherwise.
 */
gchar* call_webauth(gchar* url, struct ProfileData *cp)
{
#ifdef TEST
info_proxy = CTF_TEST_DATA("PROXY");
info_port = atoi(CTF_TEST_DATA("PORT"));
#endif

    msa_debug("call_webauth: START");
    gchar* back_url = NULL;
    gchar *proxy = info_proxy;
    gint port = info_port;

    if (proxy == NULL)
    {
	    GConfClient *client = gconf_client_get_default ();
	    gboolean hasProxy = gconf_client_get_bool (client, GCONF_PROXY_USE, NULL);
	    msa_debug("%s:%d:%s: hasProxy=%d", __FILE__, __LINE__, __FUNCTION__, hasProxy);
	    if (hasProxy) {
	        proxy = gconf_client_get_string (client, GCONF_PROXY_HOST, NULL);
	        port = gconf_client_get_int (client, GCONF_PROXY_PORT, NULL);
	        msa_debug("%s:%d:%s: host=%s, port=%d", __FILE__, __LINE__, __FUNCTION__, proxy, port);
	    }
	    g_object_unref(client);
    }
    gchar* sPort = g_strdup_printf("%d", port);

    msa_debug("%s:%d:%s: use proxy=%s:%d", __FILE__, __LINE__, __FUNCTION__, proxy, port);

    if (cp == NULL){
        msa_debug("can't authorize, no profile!");
        return NULL;
    }
    if (url == NULL){
        msa_debug("can't authorize, no url!");
        return NULL;
    }


    msa_debug("Try call webauth with %s = %s", CLIENT_ID_PARAM, cp->id);
    char *patterns[] = { REGEXP_PARAM, LOGIN_SUCCESS, TITLE_PARAM, cp->id, CLIENT_ID_PARAM, DRIVER_ID, ACCOUNT_ID_PARAM, cp->id};
    char **patt_arr = patterns;
    back_url = webauth(url, proxy, sPort, "", "", patt_arr, 8, cp->id);

    g_free(sPort);
   return back_url;
}

/**
 * Get class of request
 *
 * @param reqXml - request
 * @return class of request.
 */
xmlChar* get_req_class(xmlDocPtr reqXml)
{
    xmlNodePtr rootNode = NULL;
    if(reqXml != NULL)
        rootNode = xmlDocGetRootElement(reqXml);
    if (rootNode != NULL)
        return xmlGetProp(rootNode, BAD_CAST CLASS);
    return NULL;
}

/**
 * Get function of request
 *
 * @param reqXml - request
 * @return function of request.
 */
xmlChar* get_req_function(xmlDocPtr reqXml)
{
    xmlNodePtr rootNode = NULL;
    if (reqXml != NULL)
	rootNode = xmlDocGetRootElement(reqXml);
    if (rootNode != NULL)
	return xmlGetProp(rootNode, BAD_CAST FUNCTION);
    return NULL;
}


/**
 * Form error message
 *
 * @param code - error code
 * @param doc - response (buffer for message)
 * @param message - text of message
 * @param comment - comment of message
 */
void error_message(gchar* code, xmlDocPtr *doc, gchar* message, gchar* comment)
{
    gchar *tmp = NULL;
    if (doc != NULL)
        xmlFreeDoc(*doc);

    tmp = g_strdup_printf(ERROR_RESPONSE, DRIVER_NAME, code, message, comment);
    *doc = xmlParseDoc((xmlChar*)tmp);
    g_free(tmp);
}

void curl_set_options(CURL *c, const char* proxy, int port)
{
    /* disable callbacks for working in multithreads */
    curl_easy_setopt(c, CURLOPT_NOSIGNAL, 1);

    gchar *curProxy = (gchar*)proxy;
    gint curPort = port;
    if (curProxy == NULL)
    {
	/* try to get system proxy from gconf */
	GConfClient *client = gconf_client_get_default ();
	gboolean hasProxy = gconf_client_get_bool (client, GCONF_PROXY_USE, NULL);
	msa_debug("%s:%d:%s: hasProxy=%d", __FILE__, __LINE__, __FUNCTION__, hasProxy);
	if (hasProxy) {
	    curProxy = gconf_client_get_string (client, GCONF_PROXY_HOST, NULL);
	    curPort = gconf_client_get_int (client, GCONF_PROXY_PORT, NULL);
	    msa_debug("%s:%d:%s: host=%s, port=%d", __FILE__, __LINE__, __FUNCTION__, curProxy, curPort);
	}
	g_object_unref(client);
    }
    curl_easy_setopt(c, CURLOPT_PROXY, curProxy);
    curl_easy_setopt(c, CURLOPT_PROXYPORT, curPort);
    curl_easy_setopt(c, CURLOPT_TIMEOUT, 120);
    curl_easy_setopt(c, CURLOPT_FOLLOWLOCATION, 1);
}

/**
 * Perform HTTP GET request
 *
 * @param url - request
 * @param proxy - proxy address 
 * @param port - proxy port
 * @param header - buffer for header of response
 * @param body - buffer for body of response
 * @return http server response code 
 */
int http_get(const char* url, const char* proxy, int port, char** header, gchar** body, int* data_size)
{
    global_get++;    
    msa_debug("IN HTTP GET!!! %d\n", global_get);

    CURL* c;
    long code;
    char* type2;
    char* encoded_url = NULL;
    struct MemoryStruct chunk, chunk2;
    
    chunk.memory = NULL; 
    chunk.size = 0;

    chunk2.memory = NULL; 
    chunk2.size = 0;

    c = curl_easy_init();

    msa_debug("%s:%d:%s:url in http_get %s, proxy=%s, port=%d", __FILE__, __LINE__, __FUNCTION__, url, proxy, port);

    curl_easy_setopt(c, CURLOPT_URL, url);

    curl_set_options(c, proxy, port);

    /* disable callbacks for working in multithreads*/
    curl_easy_setopt(c, CURLOPT_NOSIGNAL, 1);
    curl_easy_setopt(c, CURLOPT_SSL_VERIFYPEER, 0);
    

    curl_easy_setopt(c, CURLOPT_WRITEFUNCTION, WriteMemoryCallback);
    curl_easy_setopt(c, CURLOPT_HEADERFUNCTION, WriteMemoryCallback);

    curl_easy_setopt(c, CURLOPT_WRITEDATA, (void *)&chunk);
    curl_easy_setopt(c, CURLOPT_WRITEHEADER, (void *)&chunk2);
    msa_debug("send request");

    CURLcode curlResult = curl_easy_perform(c);
    if ( curlResult != 0) {
        g_warning("Can't perform the request\n");
        g_debug("Curl CODE: %u", curlResult);
        curl_easy_cleanup(c);
        return 400;
    }

    free(encoded_url);
    CURLcode ret = curl_easy_getinfo(c, CURLINFO_RESPONSE_CODE, &code );
    msa_debug("CODE %ld", code);

    /*type2 = g_malloc(sizeof(char)*50);*/
    ret = curl_easy_getinfo(c, CURLINFO_CONTENT_TYPE, &type2 );
    msa_debug("TYPE %s", type2);

    if (header != NULL) {
        *header = g_strdup(chunk2.memory);
    }

    msa_debug("HEADER  %s\n", chunk2.memory);
    if(chunk2.memory)
        free(chunk2.memory);

    if (body != NULL) {
            *data_size = chunk.size;
    	    *body = chunk.memory;
    } else 
        if(chunk.memory)
            free(chunk.memory);

    /*g_print("CONTENT  %s\n", *body);*/

    /*g_free(type2);*/
    curl_easy_cleanup(c);

    return code;
}

/**
 * Perform HTTP POST request
 *
 * @param url - request
 * @param proxy - proxy address
 * @param port - proxy port
 * @param post_data - data to send
 * @param header - buffer for header of response
 * @param body - buffer for body of response
 * @return http server response code
 */

int http_post(const char* url, const char* proxy, int port, char* post_data, char** header, char** body, int* data_size)
{
    CURL* c;
    long code;
    CURLcode res;
    struct MemoryStruct chunk,chunk2;

    chunk.memory = NULL;
    chunk.size = 0;

    chunk2.memory = NULL;
    chunk2.size = 0;


    c = curl_easy_init();
    msa_debug("%s:%d:%s:url in http_post %s, proxy=%s:%d", __FILE__, __LINE__, __FUNCTION__, url, proxy, port);

    curl_easy_setopt(c, CURLOPT_URL, url);

    curl_set_options(c, proxy, port);

    curl_easy_setopt(c, CURLOPT_POST, 1);
    curl_easy_setopt(c, CURLOPT_POSTFIELDS, post_data);
    curl_easy_setopt(c, CURLOPT_POSTFIELDSIZE, strlen(post_data));
    curl_easy_setopt(c, CURLOPT_FOLLOWLOCATION, 1);

    curl_easy_setopt(c, CURLOPT_COOKIEFILE, "");

    curl_easy_setopt(c, CURLOPT_WRITEFUNCTION, WriteMemoryCallback);
    curl_easy_setopt(c, CURLOPT_HEADERFUNCTION, WriteMemoryCallback);

    curl_easy_setopt(c, CURLOPT_WRITEDATA, (void *)&chunk);
    curl_easy_setopt(c, CURLOPT_WRITEHEADER, (void *)&chunk2);
    msa_debug("%s:%d:%s():send request", __FILE__, __LINE__, __FUNCTION__);
    
    CURLcode errornum = curl_easy_perform(c);
    if (errornum != 0) {
        g_warning("%s:%d:%s: Can't perform the request: %s", __FILE__, __LINE__, __FUNCTION__, curl_easy_strerror(errornum));
        curl_easy_cleanup(c);
        return 400;
    }
    
    res = curl_easy_getinfo(c, CURLINFO_RESPONSE_CODE, &code );
    msa_debug("CODE %ld", code);
    msa_debug("HEADER\n  %s", chunk2.memory);    
    
    if (header != NULL) {
        *header = g_strdup(chunk2.memory);
    }
    
    if(chunk2.memory)
        g_free(chunk2.memory);
    
    if (body != NULL) {
        *data_size = chunk.size;
        *body = g_strdup(chunk.memory);
        if(chunk.memory) 
	    g_free(chunk.memory);
    }
    
    curl_easy_cleanup(c);
    return code;
}

int http_upload_file(const char* url, const char* proxy, const int port, char** header, gchar** body, struct ProfileData *cp, struct UpFile *file)
{
    CURL *curl;
    CURLcode res;
    long code;
    struct MemoryStruct chunk,chunk2;
    int i = 0;

    chunk.memory = NULL;
    chunk.size = 0;

    chunk2.memory = NULL;
    chunk2.size = 0;

    struct curl_httppost *formpost=NULL;
    struct curl_httppost *lastptr=NULL;
    struct curl_slist *headerlist=NULL;
    static const char buf[] = "Expect:";

    curl_global_init(CURL_GLOBAL_ALL);

    /* Fill in the file upload field */
    if (file->params != NULL){
        while(file->params[i] != NULL){
            curl_formadd(&formpost,
                          &lastptr,
                 CURLFORM_COPYNAME, file->params[i],
             CURLFORM_COPYCONTENTS, file->params[i+1],
                      CURLFORM_END);
            msa_debug("%s = %s", file->params[i], file->params[i+1]);
            i = i + 2;
        }
    }

    curl_formadd(&formpost,
                  &lastptr,
         CURLFORM_COPYNAME, file->name,
             CURLFORM_FILE, file->file_path,
         CURLFORM_FILENAME, file->file_name,
              CURLFORM_END);

    curl = curl_easy_init();
    /* initalize custom header list (stating that Expect: 100-continue is not
     *      wanted */
    headerlist = curl_slist_append(headerlist, buf);

    if(curl) {
     /* what URL that receives this POST */
        curl_easy_setopt(curl, CURLOPT_URL, url);

     /* only disable 100-continue header if explicitly requested */
        curl_easy_setopt(curl, CURLOPT_HTTPHEADER, headerlist);
        curl_easy_setopt(curl, CURLOPT_HTTPPOST, formpost);
        
        curl_set_options(curl, proxy, port);

        curl_easy_setopt(curl, CURLOPT_WRITEFUNCTION, WriteMemoryCallback);
        curl_easy_setopt(curl, CURLOPT_HEADERFUNCTION, WriteMemoryCallback);

        curl_easy_setopt(curl, CURLOPT_WRITEDATA, (void *)&chunk);
        curl_easy_setopt(curl, CURLOPT_WRITEHEADER, (void *)&chunk2);

        res = curl_easy_perform(curl);

        res = curl_easy_getinfo(curl, CURLINFO_RESPONSE_CODE, &code );
        msa_debug("CODE %ld", code);
        msa_debug("HEADER\n  %s", chunk2.memory);        
        
        if (header != NULL) {
            *header = g_strdup(chunk2.memory);
        }
        
        if(chunk2.memory)
            g_free(chunk2.memory);
        
        if (body != NULL) {
            *body = g_strdup(chunk.memory);
        if(chunk.memory) 
	        g_free(chunk.memory);
        }
        /* always cleanup */
        curl_easy_cleanup(curl);
        /* then cleanup the formpost chain */
        curl_formfree(formpost);
        /* free slist */
        curl_slist_free_all (headerlist);
    }
   return code;
}

xmlNodePtr xpath_get_node(char* path, xmlDocPtr doc)
{
    xmlXPathObject *Obj = NULL;
    xmlNodePtr node = NULL;
    Obj = xpath(path, doc);
    if(Obj == NULL)
        return NULL;
    if(!xmlXPathNodeSetIsEmpty(Obj->nodesetval))
        node = xmlXPathNodeSetItem(Obj->nodesetval, 0);
    xmlXPathFreeObject(Obj);
    return node;
}

/**
 * @param req - xpath req
 * @param doc - xmlDocPtr  
 * @return xmlXPathObject*  
**/
xmlXPathObject* xpath(gchar* req, xmlDocPtr doc)
{
	xmlXPathObject *Obj;
	xmlXPathContextPtr Ctx;
	
	msa_debug("fb_xpath: %s", req);

    if (doc == NULL || req == NULL)
        return NULL;

	Ctx = xmlXPathNewContext(doc);
	
	if (Ctx == NULL) {
		msa_debug("fb_xpath: error wrong request format!");
		return NULL;	
	}

	Obj = xmlXPathEvalExpression(BAD_CAST req, Ctx);


        xmlXPathFreeContext(Ctx);
	msa_debug("fb_xpath: end");

	return Obj;

}

void upfile_remove(struct UpFile *file)
{
    if (file != NULL){
        if (file->name != NULL)
	    g_free(file->name);
        if (file->file_path != NULL)
	    g_free(file->file_path);
        if (file->file_name != NULL)
	    g_free(file->file_name);
        if (file->params != NULL)
            g_strfreev(file->params);

        free(file);
        file = NULL;
    }
}

