/*
 * This file is part of AEGIS
 *
 * Copyright (C) 2009 Nokia Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 * Author: Markku Savela
 */

/*
 * suid-loader.c
 *
 * sudo ./suid-loader suid.txt
 *
 * This program opens and reads the given control file, which is
 * assumed to contain a tab delimited records for setuid/setgid
 * executables. Each line of the control file should follow format
 *
 *	uid-value	gid-value	path-of-executable
 *
 * If executable is setuid, then uid-value should be the uid to
 * set, and otherwise -1.
 *
 * If exectublae is setgid, then gid-value should be the gid to
 * set, and otherwise -1.
 *
 * The command generates and loads the policy for each executable.
 * The resulting policy emulates the original setuid/setgid
 * behaviour (uid == 0 causes all cababilities to be set).
 */
#define _ISOC99_SOURCE /* ..to get isblank from ctypes.h */

#include <stddef.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <unistd.h>
#include <ctype.h>
#include <sys/types.h>
#include <fcntl.h>
#include <err.h>
#include <linux/aegis/creds.h>
#include <linux/aegis/credp.h>

struct stream_buf
	{
	int fd;
	int eol;
	int head;
	int tail;
	char data[200];
	};

static int pull(struct stream_buf *buf, int req_len)
	{
	int len;
	while ((len = buf->tail - buf->head) < req_len)
		{
		int room;
		if (buf->head > 0)
			{
			if (len > 0)
				memmove(buf->data, buf->data + buf->head, len);
			buf->head = 0;
			buf->tail = len;
			}
		if (buf->fd < 0)
			return 0; /* No data */

		
		room = sizeof(buf->data) - buf->tail;
		len = read(buf->fd, buf->data + buf->tail, room);
		if (len <= 0)
			{
			/* No more data available */
			close(buf->fd);
			buf->fd = -1;
			return 0;
			}
		buf->tail += len;
		}
	return len;
	}

static void skip_line(struct stream_buf *buf)
	{
	int len;
	while ((len = pull(buf, 1)) > 0)
		{
		char *s = buf->data + buf->head;
		char *nl = (char *)memchr(s, '\n', len);
		if (nl)
			{
			len = nl - s;
			buf->eol = 0;
			buf->head += len + 1;
			return;
			}
		buf->head += len;
		}
	/* ...missing new line at end of file? */
	buf->eol = 1;
	}

static void skip_blank(struct stream_buf *buf)
	{
	while (pull(buf, 1))
		{
		const int c = buf->data[buf->head];
		if (c == '\n')
			{
			buf->eol = 1;
			break;
			}
		if (!isblank(c))
			break;
		buf->head += 1;
		}
	}

static int get_next_int(struct stream_buf *buf)
	{
	int good = 0;
	int value = 0;
	int negate = 1;

	if (buf->eol)
		return -1;

	skip_blank(buf);

	if (!buf->eol && buf->data[buf->head] == '-')
		{
		++buf->head;
		negate = -1;
		}
	/* Pull decimal digits */
	while (pull(buf, 1))
		{
		const int c = buf->data[buf->head];
		if (!isdigit(c))
			break;
		value = value * 10 + c - '0';
		buf->head += 1;
		good = 1;
		}
	return good ? negate*value : -1;
	}

static int get_next_path(struct stream_buf *buf)
	{
	int len = 0;
	if (buf->eol)
		return -1;
	skip_blank(buf);

	while ((len = pull(buf, len + 1)) != 0)
		{
		char *s = buf->data + buf->head;
		char *nl = (char *)memchr(s, '\n', len);
		if (nl)
			return nl - s;
		if (buf->head == 0)
			break; // Too long string!
		}
	return -1;
	}

static void get_suid_programs(const char *path)
	{
	struct stream_buf buf;

	buf.head = 0;
	buf.tail = 0;
	buf.eol = 0;

	buf.fd = open(path, O_RDONLY);
	if (buf.fd < 0)
		return;

	while (pull(&buf, 1))
		{
		int count = 0;
		int uid = get_next_int(&buf);
		int gid = get_next_int(&buf);
		int len = get_next_path(&buf);
		credp_tlv_t items[20];

		if (uid != -1)
			{
			items[count].T = CREDS_UID;
			items[count++].L = 1;
			items[count++].V = uid;
			}
		if (gid != -1)
			{
			items[count].T = CREDS_GID;
			items[count++].L = 1;
			items[count++].V = gid;
			}
		if (uid == 0)
			{
			/* Emulate old "root has all capabilities */
			items[count].T = CREDS_CAP;
			items[count++].L = 2;
			items[count++].V = ~0; /* caps 0..31 */
			items[count++].V = ~0; /* caps 32..63 */
			}
		printf("Got: uid=%d gid=%d count=%d path=%.*s\n", uid, gid, count, len, buf.data + buf.head);
		if (credp_load(CREDP_TYPE_SET, buf.data + buf.head, len, items, count))
			warn("\tcredp_load failed");

		skip_line(&buf);
		}
	if (buf.fd >= 0)
		close(buf.fd);
	return;
	}


int main (int argc, char **argv)
	{
	while (*++argv)
		get_suid_programs(*argv);
	exit(EXIT_SUCCESS);
	}
