/*
 * This file is part of AEGIS
 *
 * Copyright (C) 2009 Nokia Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 * Author: Markku Savela
 */
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <err.h>
#include <ctype.h>
#include <sys/creds.h>

extern char **environ;
/*
 * Compute a new set of credentials and run the
 * executable under modified credentials.
 *
 * Without special privileges, this can only
 * decrease the current credentials.
 *
 * creds-exec [options] [path [args]]
 *
 * options are:
 *  -c			Clear credentials
 *  -a credential	Add the credential
 *  -d credential	Delete the credential
 */
int main(int argc, char **argv)
{
	int do_creds = 1;
	creds_t creds = creds_init();
	const char *arg;
	while ((arg = *++argv) != NULL && --argc > 0) {
		int option;
		const char *str;
		creds_value_t value;
		creds_type_t type;
		int index;
		char buf[200];

		if (*arg != '-')
			break;
		++arg;
		while ((option = *arg++) != 0) {
			int sub = 0;
			switch (option) {
			case 'n':
				do_creds = 0;
				break;
			case 'c':
				/* Clear current credentials set */
				creds_clear(creds);
				break;
			case 'p':
				/* Reload creds from a task */
				str = arg;
				if (*str == 0 && argc > 1) {
					str = *++argv;
					--argc;
				}
				option = atoi(str);
				creds_free(creds);
				creds = creds_gettask(option);
				goto next_arg;
			case 'd':
				++sub;
				/* FALL THROUGH */
			case 'a':
				/* Add credential to the set */
				str = arg;
				if (*str == 0 && argc > 1) {
					/* The next arg is the creds string */
					str = *++argv;
					--argc;
				}
				type = creds_str2creds(str, &value);

				if (type == CREDS_BAD) {
					warnx("Cannot translate '%s' into credential value\n", str);
					exit(EXIT_FAILURE);
				}

				if (sub)
					creds_sub(creds, type, value);
				else if (creds_add(&creds, type, value)) {
					warnx("Couldn't add credential '%s' (%d,%ld) into credentials\n",
					      str, type, value);
					exit(EXIT_FAILURE);
				}
				goto next_arg;
			case 'D':
				for (index = 0; (type = creds_list(creds, index, &value)) != CREDS_BAD; ++index) {
					(void)creds_creds2str(type, value, buf, sizeof(buf));
					buf[sizeof(buf)-1] = 0;
					printf("%s\n", buf);
				}
				break;
			default:
				warnx("Unkonwn option '%c'\n", option);
				exit(EXIT_FAILURE);
			}
		}
	next_arg:
		;
	}
	if (do_creds && creds_set(creds)) {
		warnx("Couldn't set the credentials for the current process\n");
		exit(EXIT_FAILURE);
	}
	if (arg) {
		execve(arg, argv, environ);
		perror(arg);
		exit(EXIT_FAILURE);
	}
	exit(EXIT_SUCCESS);
}
