/*
 * This file is part of AEGIS
 *
 * Copyright (C) 2009 Nokia Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 * Author: Markku Savela
 */

#if HAVE_LINUX_AEGIS_CREDP_H

#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <unistd.h>
#include <fcntl.h>
#include <string.h>
#include <err.h>
#include <errno.h>
#include <linux/aegis/credp.h>
#include <linux/aegis/credpctl.h>

static const char *const policy_file = "/sys/kernel/security/" CREDP_SECURITY_DIR "/" CREDP_SECURITY_FILE;

static long doit(int op, int flags, const char *path, void *data, size_t dlen)
{
	const size_t plen = path ? strlen(path) : 0;
	struct credp_ioc_arg arg = {
		.flags = flags,
		.path.length = plen,
		.path.name = path,
		.list.length = dlen,
		.list.items = data
	};

	long result;

	int fd = open(policy_file, O_RDONLY);
	if (fd == -1)
		return -errno;

	result = ioctl(fd, op, &arg);
	close(fd);
	return result;
}


long credp_kload(int flags, const char *path, const __u32 *list, size_t list_length)
{
	return doit(SIOCCREDP_LOAD, flags, path, (void *)list, list_length);
}

long credp_kunload(const char *path)
{
	return doit(SIOCCREDP_UNLOAD, 0, path, NULL, 0);
}

long credp_kconfine(const char *path)
{
	return doit(SIOCCREDP_CONFINE, 0, path, NULL, 0);
}

#if defined(HAVE_SIOCCREDP_CONFINE2)
long credp_kconfine2(const char *path, int flags, const __u32 *list, size_t list_length)
{
	return doit(SIOCCREDP_CONFINE2, flags, path, (void *)list, list_length);
}
#endif

long credp_kset(int flags, const __u32 *list, size_t list_length)
{
	return doit(SIOCCREDP_SET, flags, NULL, (__u32 *)list, list_length);
}
#else

/*
 * There is no real fallback, if kernel does not support the
 * functionality -- just return a suitable error code for
 * all attempted requests.
 */

#include <errno.h>
#include <err.h>
#include "credp_fallback.h"

long credp_kload(int flags, const char *path, const __u32 *list, size_t list_length)
{
	warnx("credp_kload: not available -- libcreds compiled without required kernel headers");
	return -EPERM;
}

long credp_kunload(const char *path)
{
	warnx("credp_unload: not available -- libcreds compiled without required kernel headers");
	return -EINVAL;
}

long credp_kconfine(const char *path)
{
	warnx("credp_kconfine: not available -- libcreds compiled without required kernel headers");
	return -EPERM;
}

long credp_kconfine2(const char *path, int flags, const __u32 *list, size_t list_length)
{
	warnx("credp_kconfine2: not available -- libcreds compiled without required kernel headers");
	return -EPERM;
}

long credp_kset(int flags, const __u32 *list, size_t list_length)
{
	warnx("credp_kset: not available -- libcreds compiled without required kernel headers");
	return -EPERM;
}
#endif
