/*
 * This file is part of AEGIS
 *
 * Copyright (C) 2010 Nokia Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 * Author: Markku Savela
 */

#define _ISOC99_SOURCE /* ..to get isblank from ctypes.h */
#define _GNU_SOURCE /* ..to get struct ucred from sys/socket.h */
#include <unistd.h>
#include <syslog.h>
#include <sys/creds.h>
#include <sys/types.h>
#include <sys/wait.h>

#undef _ISOC99_SOURCE
#undef _GNU_SOURCE

#define CREDS_AUDIT_LOG

static void creds_audit_init(creds_t creds, pid_t pid);
static void creds_audit_free(creds_t creds);
static void creds_audit_log(creds_t creds, creds_type_t type, creds_value_t value);

typedef struct
	{
	char *name;
	} creds_audit_t;
	
#include "../creds.c"

static void creds_audit_init(creds_t creds, pid_t pid)
	{
	char buf[200];
	int fd, size;
	char *start, *end;

	if (pid)
		snprintf(buf, sizeof(buf), "/proc/%d/stat", pid);
	else
		strncpy(buf, "/proc/self/stat", sizeof(buf));
	fd = open(buf, O_RDONLY);
	if (fd < 0)
		{
		start = "[not available]";
		goto out;
		}
	size = read(fd, buf, sizeof(buf)-1);
	if (size <= 0)
		size = 0;
	close(fd);
	buf[size] = 0;
	if ((start = strchr(buf, '(')) && (end = strrchr(buf, ')')))
		{
		++start;
		*end = 0;
		}
	else
		start = "[bad format]";
out:
	creds->audit.name = strdup(start);
	}

static void creds_audit_free(creds_t creds)
	{
	if (creds && creds->audit.name)
		{
		free(creds->audit.name);
		creds->audit.name = NULL;
		}
	}

static void creds_audit_log(creds_t creds, creds_type_t type, creds_value_t value)
	{
	/* Static variables used by child process */
	static char str[200];
	static char exe[100];

	ssize_t rc;
	char *name = creds->audit.name ? creds->audit.name : "[null]";
	
	pid_t pid = fork();

	if (pid)
		{
		/* This parent */
		waitpid(pid, NULL, 0);
		return;
		}

	/* This is logger child process. A forked process is used
	 * because this is in a library and syslog settings are global
	 * to the task. Thus, fork and do syslog stuff in child
	 * process which does not affect the main application, which
	 * might be using syslog with other settings.
	 */
	creds_creds2str(type, value, str, sizeof(str));
	rc = readlink("/proc/self/exe", exe, sizeof(exe)-1);
	if (rc < 0)
		strcpy(exe, "[exe not available]");
	else
		exe[rc] = 0;
	openlog("AEGIS", LOG_NDELAY, LOG_AUTH);
	syslog(LOG_WARNING, "SERVER=%s ; CREDENTIAL=%s ; CLIENT=%s", exe, str, name);
	closelog();
	exit(EXIT_SUCCESS);
	}

