#include "user.h"
#include <QDomDocument>

#include <iostream>
using namespace std;

User::User(QObject *parent) :
    QObject(parent)
{
	companiesFollowed = new QHash<QString,QString>;
	peopleConnections = new QHash<QString,QString>;
	jobsBookmarked = new QHash<QString,QString>;
}

User::~User(){
	delete companiesFollowed;
	delete peopleConnections;
}

/**
  Sets the companies that the user is following based on an xml file
  obtained from an OAuth reques to http://api.linkedin.com/v1/people/~/following/companies.
  @param xml The xml file that is returned from LinkedIn
  */
void User::setCompaniesFollowed(QString xml){
	companiesFollowed->clear();

	QDomDocument doc;
	doc.setContent(xml);

	QDomNodeList companies = doc.elementsByTagName("company");

	for(int i = 0; i < companies.length(); i++){
		QString key = companies.at(i).namedItem("id").toElement().text();
		QString value = companies.at(i).namedItem("name").toElement().text();
		companiesFollowed->insert(key, value);
	}


}

/**
  Tests to see if the user is following a company
  @param id The numeric id of the company, not the name
  */
bool User::following(QString id){
	return companiesFollowed->contains(id);
}

/**
  Get the number of companies the user is currently following
  @return The number of companies the user is currently following
  */
int User::numFollowing(){
	return companiesFollowed->size();
}

/**
  Start following a company
  @param id The numeric id of the company
  @param name The name of the company
  */
void User::follow(QString id, QString name){
	companiesFollowed->insert(id, name);
}

/**
  Stop following a company
  @param id The numeric id of the company
  */
void User::stopFollow(QString id){
	companiesFollowed->remove(id);
}

/**
  Get a query string to use when searching Google news.
  It will look similar to this:
	google+OR+linkedin+OR+apple
  @return the query parameter for Google news rss feed
  */
QString User::getCompanyQuery(){
	QList<QString> list = companiesFollowed->values();
	QString query("http://news.google.com/news?pz=1&ned=us&hl=en&output=rss&q=");

	for(int i = 0; i < list.length(); i++){
		query.append(list.at(i));
		if(i != list.length()-1)
			query.append("+OR+");
	}

	return query;
}

/**
  Sets the people that the user is connected to based on an xml file
  obtained from an OAuth reques to http://api.linkedin.com/v1/people/~/connections.
  @param xml The xml file that is returned from LinkedIn
  */
void User::setPeopleConnections(QString xml){
	peopleConnections->clear();

	QDomDocument doc;
	doc.setContent(xml);

	QDomNodeList people = doc.elementsByTagName("person");

	for(int i = 0; i < people.length(); i++){
		QString key = people.at(i).namedItem("id").toElement().text();
		QString value = people.at(i).namedItem("first-name").toElement().text();
		peopleConnections->insert(key, value);
	}
}

/**
  Tests to see if the user is connected to a member
  @param id The id of the member
  */
bool User::connected(QString id){
	if(id == "~")
		return true;
	return peopleConnections->contains(id);
}

/**
  Add a user connection with another member
  @param id The id of the member
  @param firstName The first name of the member
  */
void User::addConnection(QString id, QString firstName){
	peopleConnections->insert(id, firstName);
}

/**
  Remove a connection with another member
  @param id The id of the member
  */
void User::removeConnection(QString id){
	peopleConnections->remove(id);
}

/**
  Sets the jobs that the user has bookmarked based on an xml file
  obtained from an OAuth reques to http://api.linkedin.com/v1/people/~/job-bookmarks.
  @param xml The xml file that is returned from LinkedIn
  */
void User::setJobsBookmarked(QString xml){
	jobsBookmarked->clear();

	QDomDocument doc;
	doc.setContent(xml);

	QDomNodeList jobs = doc.elementsByTagName("job");

	for(int i = 0; i < jobs.length(); i++){
		QString key = jobs.at(i).namedItem("id").toElement().text();
		QString value = jobs.at(i).namedItem("position").toElement().namedItem("title").toElement().text();
		jobsBookmarked->insert(key, value);
	}
}

/**
  Tests to see if the user bookmarked a job
  @param id The id of the job
  */
bool User::bookmarked(QString id){
	return jobsBookmarked->contains(id);
}

/**
  Add a job bookmark
  @param id The id of the job
  @param title The title of the job
  */
void User::bookmark(QString id, QString title){
	jobsBookmarked->insert(id, title);
}

/**
  Remove a job bookmark
  @param id The id of the job
  */
void User::removeBookmark(QString id){
	jobsBookmarked->remove(id);
}

void User::print(){
	QList<QString> tmpList = jobsBookmarked->values();
	for(int i = 0; i < tmpList.length(); i++){
		cout << tmpList.at(i).toStdString() << endl;
	}
}
