/*
    Copyright 2010 Marcello V. Mansueto <marcello.m@yahoo.com>

    This file is part of lightmeter.

    lightmeter is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    lightmeter is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with lightmeter.  If not, see <http://www.gnu.org/licenses/>.
*/

#include <QDebug>
#include <QFile>
#include <QSettings>
#include <math.h>

#include "mainwindow.h"
#include "ui_mainwindow.h"
#include "settingsdialog.h"
#include "about.h"

MainWindow::MainWindow(QWidget *parent)
    : QMainWindow(parent), ui(new Ui::MainWindowClass) {
    ui->setupUi(this);
    statusBar()->hide();
    ui->mainToolBar->hide();
    readSettings();
    mode = APERTURE;
    ui->modeLabel->setText(tr("Mode: APERTURE"));
    connect(ui->lcdNumberAperture, SIGNAL(activated(QString)), this, SLOT(apertureHandler(QString)));
    connect(ui->lcdNumberIso, SIGNAL(activated(QString)), this, SLOT(isoHandler(QString)));
    connect(ui->lcdNumberSpeed, SIGNAL(activated(QString)), this, SLOT(speedHandler(QString)));
    connect(ui->actionSettings, SIGNAL(triggered()), this, SLOT(editSettings()));
    connect(ui->actionAbout, SIGNAL(triggered()), this, SLOT(openAboutDialog()));
    connect(ui->storeButton_1, SIGNAL(clicked()), this, SLOT(storeExp()));
    connect(ui->storeButton_2, SIGNAL(clicked()), this, SLOT(storeExp()));
    connect(ui->storeButton_3, SIGNAL(clicked()), this, SLOT(storeExp()));
    startTimer(1000); // 1-second timer
}

MainWindow::~MainWindow() {
    delete ui;
}

void MainWindow::timerEvent(QTimerEvent *event) {
    QFile luxFile("/sys/class/i2c-adapter/i2c-2/2-0029/lux");
    if (!luxFile.open(QIODevice::ReadOnly | QIODevice::Text)) {
         return;
     }
    QTextStream luxStream(&luxFile);
    QString luxString = luxStream.readLine();
    luxFile.close();
    bool ok;
    int lux = luxString.toInt(&ok);
    double ev = log(lux / 2.5) / log(2);
    double ev100 = ev - (log(100 / iso) / log(2));
    if(mode == APERTURE) {
        double realSpeed = pow(aperture, 2) / pow(2, ev100);
        int index = findNearestListElement(realSpeed, speedList);
        if(index == -1) {
            ui->lcdNumberSpeed->display(INFINITY);
            speedString = QString("INF");
        } else {
            if(speedList.at(index).contains("/")) {
                QStringList list = speedList.at(index).split("/");
                ui->speedLabel->setText(tr("Speed (1/sec)"));
                ui->lcdNumberSpeed->display(list.last());
            } else {
                ui->speedLabel->setText(tr("Speed (sec)"));
                ui->lcdNumberSpeed->display(speedList.at(index));
            }
            speedString = speedList.at(index);
        }
        ui->realSpeedLabel->setText(QString("Real speed: %1").arg(realSpeed));
//        qDebug() << "Aperture mode - lux = " << lux;
//        qDebug() << "Aperture mode - speed = " << realSpeed;
    } else {
        double realAperture = sqrt(speed * pow(2, ev100));
        int index = findNearestListElement(realAperture, apertureList);
        if(index == -1) {
            ui->lcdNumberAperture->display(INFINITY);
            apertureString = QString("INF");
        } else {
            ui->lcdNumberAperture->display(apertureList.at(index));
            apertureString = apertureList.at(index);
        }
        ui->realApertureLabel->setText(QString("Real aperture: %1").arg(realAperture));
//        qDebug() << "Speed mode - lux = " << lux;
//        qDebug() << "Speed mode - aperture = " << realAperture;
    }
}

void MainWindow::apertureHandler(QString value) {
    ui->lcdNumberAperture->display(value);
    bool ok;
    aperture = value.toDouble(&ok);
    apertureString = value;
    ui->realApertureLabel->setText(QString("Real aperture: %1").arg(aperture));
    if(sender()) {
        if(sender()->objectName() != QString("actionSettings")) {
            mode = APERTURE;
            ui->modeLabel->setText(tr("Mode: APERTURE"));
        }
    }
}

void MainWindow::isoHandler(QString value) {
    ui->lcdNumberIso->display(value);
    bool ok;
    iso = value.toDouble(&ok);
    isoString = value;
}

void MainWindow::speedHandler(QString value) {
    if(value.contains("/")) {
        QStringList list = value.split("/");
        ui->speedLabel->setText(tr("Speed (1/sec)"));
        ui->lcdNumberSpeed->display(list.last());
        speed = list.first().toDouble() / list.last().toDouble();
    } else {
        ui->speedLabel->setText(tr("Speed (sec)"));
        ui->lcdNumberSpeed->display(value);
        speed = value.toDouble();
    }
    speedString = value;
    ui->realSpeedLabel->setText(QString("Real speed: %1").arg(speed));
    if(sender()) {
        if(sender()->objectName() != QString("actionSettings")) {
            mode = SPEED;
            ui->modeLabel->setText(tr("Mode: SPEED"));
        }
    }
}

void MainWindow::editSettings() {
    SettingsDialog* s = new SettingsDialog(this);
    if (s->exec()) {
        readSettings();
    }
}

void MainWindow::readSettings() {
    QSettings settings;
    QString camera = settings.value("camera", QString("D80 0.3EV")).toString();
    QString lens = settings.value("lens", QString("Nikkor DX AF-S 16-85mm 1:3.5-5.6G ED VR 0.3EV")).toString();
    ui->confLabel->setText(QString(camera).append(" - ").append(lens));
    QStringList isoList;
    QSettings cameraSettings("/opt/lightmeter/camera.conf", QSettings::NativeFormat);
    QSettings lensSettings("/opt/lightmeter/lens.conf", QSettings::NativeFormat);
    lensSettings.beginGroup(lens);
    apertureList = lensSettings.value("aperture", QStringList()).toStringList();
    lensSettings.endGroup();
    cameraSettings.beginGroup(camera);
    isoList = cameraSettings.value("iso", QStringList()).toStringList();
    speedList = cameraSettings.value("speed", QStringList()).toStringList();
    cameraSettings.endGroup();
    ui->lcdNumberAperture->setType(apertureList, tr("Choose Aperture"), tr("Aperture:"));
    ui->lcdNumberIso->setType(isoList, tr("Choose ISO"), tr("ISO:"));
    ui->lcdNumberSpeed->setType(speedList, tr("Choose Speed"), tr("Speed (sec):"));
    apertureHandler(apertureList.first());
    isoHandler(isoList.first());
    speedHandler(speedList.first());
}

int MainWindow::findNearestListElement(double value, QStringList list) {
    double listValue;
    double delta;
    double deltaP = 0;
    for(int i = 0; i < list.size(); i++) {
        if(list.at(i).contains("/")) {
            QStringList subList = list.at(i).split("/");
            listValue = subList.first().toDouble() / subList.last().toDouble();
        } else {
            listValue = list.at(i).toDouble();
        }
        delta = value - listValue;

//        qDebug() << "value:" << value;
//        qDebug() << "listValue:" << listValue;
//        qDebug() << "delta:" << delta;

        if(delta < 0) {
            if(i == 0) {
                return -1;
            }
            if(fabs(delta) < fabs(deltaP)) {
                return i;
            } else {
                return i-1;
            }
        } else if(delta == 0) {
            return i;
        }
        deltaP = delta;
    }
    return -1;
}
void MainWindow::storeExp() {
    QPushButton* button = (QPushButton*) sender();
    button->setText(QString(speedString).append("s  f").append(apertureString).append(" ISO ").append(isoString));
}

void MainWindow::openAboutDialog() {
    About about;
    about.exec();
}
