/* This file is part of LED Pattern Editor.
 *
 * Copyright (C) 2010 Philipp Zabel
 *
 * LED Pattern Editor is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * LED Pattern Editor is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with LED Pattern Editor. If not, see <http://www.gnu.org/licenses/>.
 */

class LedPatternDialog : Gtk.Dialog {
	public enum Response {
		HELP = 1,
		RESTORE = 2
	}

	private unowned Osso.Context osso_context;
	private unowned List<LedPattern> list;
	private bool close = false;

	public LedPatternDialog (List<LedPattern> _list, Osso.Context osso) {
		list = _list;
		osso_context = osso;
		set_title ("LED Patterns");

		var content = (Gtk.VBox) get_content_area ();
		content.set_size_request (-1, 5*70);

		var pannable = new Hildon.PannableArea ();
		var vbox = new Gtk.VBox (false, 0);

		foreach (LedPattern pattern in list) {
			var button = new LedPatternButton (pattern);
			Hildon.gtk_widget_set_theme_size (button, Hildon.SizeType.FINGER_HEIGHT);
			button.set_data ("pattern", pattern);
			button.clicked.connect (on_pattern_clicked);
			vbox.pack_start (button, true, true, 0);
		}

		pannable.add_with_viewport (vbox);
		content.pack_start (pannable, true, true, 0);
		content.show_all ();

		var button_help = new Gtk.Button.with_label ("Help");
		Hildon.gtk_widget_set_theme_size (button_help, Hildon.SizeType.FINGER_HEIGHT);

		var label = new Gtk.Label ("Restore original\npatterns");
		label.justify = Gtk.Justification.CENTER;
		var button_restore = new Gtk.Button ();
		button_restore.add (label);
		Hildon.helper_set_logical_font (button_restore, "SmallSystemFont");
		Hildon.gtk_widget_set_theme_size (button_restore, Hildon.SizeType.FINGER_HEIGHT);

		add_action_widget (button_help, Response.HELP);
		add_action_widget (button_restore, Response.RESTORE);

		var action_area = (Gtk.ButtonBox) get_action_area ();
		action_area.set_child_secondary (button_help, true);
		action_area.show_all ();

		add_button ("Save", Gtk.ResponseType.OK);

		response.connect (on_response);
	}

	public int run () {
		int response = 0;
		while (response >= 0) {
			response = base.run ();
			if (close)
				return response;
		}
		return response;
	}

	private void on_pattern_clicked (Gtk.Button button) {
		LedPattern pattern = button.get_data ("pattern");
		if (pattern is LedPatternRX51) {
			var dialog = new LedProgramDialog ((LedPatternRX51) pattern);
			dialog.set_transient_for (this);

			int response = 0;
			while (response >= 0)
				response = dialog.run ();
			dialog.destroy ();
		}
	}

	private void on_response (int response_id) {
		if (response_id == Response.HELP) {
			var url = "http://wiki.maemo.org/LED_Pattern_Editor";
			var status = osso_context.rpc_run_with_defaults ("osso_browser", "open_new_window", null, 's', url, 'b', false);
			if (status != Osso.Status.OK)
				Hildon.Banner.show_information (this, null, "Failed to open browser.");
		} else if (response_id == Response.RESTORE) {
			var note = new Hildon.Note.confirmation (this, "Restore original patterns? All user-created patterns will be lost.");
			response_id = note.run ();
			note.destroy ();
			if (response_id == Gtk.ResponseType.OK)
				close = true;
		}
	}
}
