/* This file is part of LED Pattern Editor.
 *
 * Copyright (C) 2010 Philipp Zabel
 *
 * LED Pattern Editor is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * LED Pattern Editor is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with LED Pattern Editor. If not, see <http://www.gnu.org/licenses/>.
 */

public enum LedColor {
	OFF = 0,
	R = 1,
	G = 2,
	B = 4,
	RG = 3,
	RB = 5,
	GB = 6,
	RGB = 7
}

class LedColorWidget : Gtk.DrawingArea {
	public LedColor color;

	public LedColorWidget () {
		color = LedColor.OFF;
	}

	public LedColorWidget.with_color (LedColor _color) {
		color = _color;
	}

	public override bool expose_event (Gdk.EventExpose event) {
		var ctx = Gdk.cairo_create (window);
		int height = allocation.height;
		int width = allocation.width;
		double radius = Math.fmin (width * 0.4, height * 0.4);

		ctx.rectangle (event.area.x, event.area.y, event.area.width, event.area.height);
		ctx.clip ();

		ctx.new_path ();
		ctx.arc (width / 2.0, height / 2.0, Math.fmin (width, height) / 2.0, 0, 2 * Math.PI);

		ctx.clip ();

		ctx.set_operator (Cairo.Operator.ADD);

		if (LedColor.R in color) {
			ctx.set_source_rgb (1.0, 0, 0);
			ctx.new_path ();
			ctx.arc (width / 2.0 + radius / 3.2 * Math.cos (270 * Math.PI / 180.0),
			         height / 2.0 + radius / 3.2 * Math.sin (270 * Math.PI / 180.0),
			         radius, 0, 2 * Math.PI);
			ctx.fill ();
		}

		if (LedColor.G in color) {
			ctx.set_source_rgb (0, 1.0, 0);
			ctx.new_path ();
			ctx.arc (width / 2.0 + radius / 3.2 * Math.cos (150 * Math.PI / 180.0),
			         height / 2.0 + radius / 3.2 * Math.sin (150 * Math.PI / 180.0),
			         radius, 0, 2 * Math.PI);
			ctx.fill ();
		}

		if (LedColor.B in color) {
			ctx.set_source_rgb (0, 0, 1.0);
			ctx.new_path ();
			ctx.arc (width / 2.0 + radius / 3.2 * Math.cos (30 * Math.PI / 180.0),
			         height / 2.0 + radius / 3.2 * Math.sin (30 * Math.PI / 180.0),
			         radius, 0, 2 * Math.PI);
			ctx.fill ();
		}

		return true;
	}

	public void update () {
		unowned Gdk.Region region = window.get_clip_region ();

		// redraw the cairo canvas completely by exposing it
		window.invalidate_region (region, true);
		window.process_updates (true);
	}
}

public class LedColorButton : Gtk.Button {
	LedColorWidget ledcolor;

	public LedColorButton.with_map (string led_map) {
		if (led_map == "r")
			set_color (LedColor.R);
		if (led_map == "g")
			set_color (LedColor.G);
		if (led_map == "b")
			set_color (LedColor.B);
		if (led_map == "rg")
			set_color (LedColor.RG);
		if (led_map == "rb")
			set_color (LedColor.RB);
		if (led_map == "gb")
			set_color (LedColor.GB);
		if (led_map == "rgb")
			set_color (LedColor.RGB);
	}

	public LedColorButton.with_color (LedColor color) {
		ledcolor.color = color;
	}

	public void set_color (LedColor color) {
		ledcolor.color = color;
		ledcolor.update ();
	}

	public LedColor get_color () {
		return ledcolor.color;
	}

	construct {
		Hildon.gtk_widget_set_theme_size (this, Hildon.SizeType.FINGER_HEIGHT);
		ledcolor = new LedColorWidget ();
		ledcolor.set_size_request (55, 55);
		add (ledcolor);
	}
}

class LedColorDialog : Gtk.Dialog {
	public LedColorDialog () {
		set_title ("Pick a LED color");

		var content = (Gtk.VBox) get_content_area ();
		var hbox = new Gtk.HBox (true, 0);

		LedColor[] colors = {
			LedColor.R, LedColor.G, LedColor.B,
			LedColor.RG, LedColor.RB, LedColor.GB,
			LedColor.RGB
		};

		for (int i = 0; i < colors.length; i++) {
			var button = new LedColorButton.with_color (colors[i]);
			Hildon.gtk_widget_set_theme_size (this, Hildon.SizeType.FINGER_HEIGHT);
			button.clicked.connect (on_clicked);
			hbox.pack_start (button, true, true, 0);
		}

		content.pack_start (hbox, true, true, 0);
		content.show_all ();
	}

	void on_clicked (Gtk.Button _button) {
		var button = (LedColorButton) _button;

		response ((int) button.get_color ());
	}
}

