/*
 * This file is part of KDevelop
 * Copyright 2010 Aleix Pol Gonzalez <aleixpol@kde.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include "horarisjobs.h"
#include <QFile>
#include <QDebug>
#include <QNetworkAccessManager>
#include <QNetworkRequest>
#include <QNetworkReply>
#include "xmlrpcparser.h"

HttpPostCall::HttpPostCall(const QUrl& s, const QByteArray& post, QObject* parent)
        : QObject(parent), m_requrl(s), m_post(post)
{}

void HttpPostCall::start()
{
    QNetworkRequest r(m_requrl);

    m_reply=m_manager.post(r, m_post);
    connect(m_reply, SIGNAL(finished()), SLOT(finished()));

    qDebug() << "starting..." << m_requrl << m_post;
}

QVariant HttpPostCall::result() const
{
    Q_ASSERT(m_reply->isFinished());
    return m_result;
}

void HttpPostCall::finished()
{
	XmlRpc::Parser parser;
	QByteArray receivedData = m_reply->readAll();

	bool ok;
	QString error;
	m_result = parser.parse(receivedData, &ok);
	if (!ok) {
		error = "parsing error";
	}
	qDebug() << "parsing..." << ok;

	emit done(m_result, error);
}

XmlRpcCall::XmlRpcCall(const QUrl& s, const QString& methodname, const QList< QVariant >& arguments, QObject* parent)
	: HttpPostCall(s, createXmlRpcCall(methodname, arguments), parent)
{}

QByteArray variantToXmlRpcValue(const QVariant& arg)
{
	QByteArray ret="<value>";
	switch(arg.type()) {
		case QVariant::Double:
			ret += "<double>"+arg.toString()+"</double>";
			break;
		case QVariant::Bool:
			ret += "<boolean>"+QString::number(arg.toInt())+"</boolean>";
			break;
		case QVariant::Int:
			ret += "<int>"+QString::number(arg.toInt())+"</int>";
			break;
		case QVariant::String:
		default:
			ret += "<string>"+arg.toString()+"</string>";
			break;
	}
	ret += "</value>";
	return ret;
}

QByteArray XmlRpcCall::createXmlRpcCall(const QString& name, const QVariantList& args)
{
	QByteArray ret;
	ret += "<?xml version='1.0' ?>";
	ret += "<methodCall>";
	ret += "<methodName>"+name.toLatin1()+"</methodName>";
	if(!args.isEmpty()) {
		ret += "<params>";
		
		foreach(const QVariant& arg, args) {
			ret += "<param>";
			ret += variantToXmlRpcValue(arg);
			ret += "</param>";
		}
		
		ret += "</params>";
	}
	
	ret += "</methodCall>";
// 	qDebug() << "pupupu" << ret;
	return ret;
}

#include "horarisjobs.moc"
