#include "horaris.h"

#include "horarisjobs.h"
#include "ui_horaris.h"
#include "ui_history.h"
#include <QMessageBox>
#include <QStandardItemModel>
#include <QToolBar>
#include <QDebug>

// static const QUrl service("http://www.starcraft.cat/horaris/webservices/");
static const QUrl service("http://horaris.net/webservices/");

Horaris::Horaris(QWidget* parent)
	: QMainWindow(parent)
{
	m_ui = new Ui::Horaris;
	m_ui->setupUi(this);
	
	m_ui->date->setDate(QDate::currentDate());
	m_ui->search->setEnabled(false);
	connect(m_ui->search, SIGNAL(clicked(bool)), SLOT(search()));
	connect(m_ui->restore, SIGNAL(clicked(bool)), SLOT(restoreView()));
	connect(m_ui->options, SIGNAL(currentRowChanged(int)), SLOT(timetableChanged(int)));
	
	XmlRpcCall* p = new XmlRpcCall(service, "get_stations", QVariantList(), this);
	p->start();
	
	connect(p, SIGNAL(done(QVariant, QString)), SLOT(receivedStations(QVariant, QString)));
	
	m_model = new QStandardItemModel(this);
	m_model->setHorizontalHeaderLabels(QStringList() << tr("Departure") << tr("Duration") << tr("Arrival"));
	
	m_lastQueries = new QStandardItemModel(this);
	m_lastQueries->setHorizontalHeaderLabels(QStringList() << tr("Date") << tr("Departure") << tr("Arrival"));
	m_tools = addToolBar(tr("History"));
	m_tools->addAction(QIcon::fromTheme("view-history"), tr("History"), this, SLOT(viewHistory()));
}

Horaris::~Horaris()
{
	delete m_ui;
}

HistoryDialog::HistoryDialog(QStandardItemModel* m, QWidget* parent): QDialog(parent)
{
	m_ui = new Ui::History;
	m_ui->setupUi(this);
	m_ui->view->setModel(m);
	connect(m_ui->view->selectionModel(), SIGNAL(selectionChanged(QItemSelection,QItemSelection)), this, SLOT(acceptableDialog()));
	
	m_ui->buttonBox->button(QDialogButtonBox::Ok)->setEnabled(false);
}

HistoryDialog::~HistoryDialog()
{ delete m_ui; }

void HistoryDialog::acceptableDialog()
{
	m_ui->buttonBox->button(QDialogButtonBox::Ok)->setEnabled(true);
}

int HistoryDialog::selectedRow() const
{
	return m_ui->view->selectionModel()->selectedRows().first().row();
}

void Horaris::viewHistory()
{
	HistoryDialog d(m_lastQueries);
	
	if(d.exec()==QDialog::Accepted) {
		int row = d.selectedRow();
		
		m_ui->departure->setCurrentIndex(m_lastQueries->index(row,1).data(Qt::UserRole+1).toInt());
		m_ui->arrival->setCurrentIndex(m_lastQueries->index(row,2).data(Qt::UserRole+1).toInt());
	}
}

void Horaris::receivedStations(const QVariant& result, const QString& error)
{
	if(!error.isEmpty())
		QMessageBox::warning(this, tr("Horaris Error"), error);
	
	QList< QVariant > opt = result.toList().first().toList();
	
	foreach(const QVariant& val, opt.first().toList()) {
		QVariantList item = val.toList();
		
		m_ui->departure->addItem(item.first().toString(), item[1]);
		m_ui->arrival->addItem(item.first().toString(), item[1]);
	}
	
	sender()->deleteLater();
	m_ui->search->setEnabled(true);
	m_ui->departure->setEnabled(true);
	m_ui->arrival->setEnabled(true);
}

void Horaris::search()
{
	int dept = m_ui->departure->currentIndex();
	int arriv = m_ui->arrival->currentIndex();
	QDate date = m_ui->date->date();
	
	XmlRpcCall* p = new XmlRpcCall(service, "get_timetables",
							QVariantList()	<< m_ui->departure->itemData(dept) << m_ui->arrival->itemData(arriv)
											<< QString::number(date.day()) << QString::number(date.month()) << QString::number(date.year()) << -1
							, this);
	connect(p, SIGNAL(done(QVariant,QString)), SLOT(searchDone(QVariant,QString)));
	
	p->start();
	m_ui->stackedWidget->setCurrentIndex(1);
	m_tools->setVisible(false);
	m_ui->options->clear();
	m_ui->resultsView->setModel(0);
	
	addHistory(dept, arriv, date);
	setBusy(true);
}

void Horaris::setBusy(bool busy)
{
#ifdef Q_WS_MAEMO_5
	setAttribute(Qt::WA_Maemo5ShowProgressIndicator, busy);
#else
	setCursor(QCursor(busy ? Qt::BusyCursor : Qt::ArrowCursor));
#endif
}

void Horaris::addHistory(int dept, int arriv, const QDate& date)
{
	QList<QStandardItem*> hist = QList<QStandardItem*>()
								<< new QStandardItem(date.toString())
								<< new QStandardItem(m_ui->departure->itemText(dept))
								<< new QStandardItem(m_ui->arrival->itemText(arriv));
	hist[0]->setData(date);
	hist[1]->setData(dept);
	hist[2]->setData(arriv);
	m_lastQueries->appendRow(hist);
}

void Horaris::searchDone(const QVariant& values, const QString& error)
{
	setBusy(false);
	if(!error.isEmpty()) {
		QMessageBox::warning(this, tr("Horaris Error"), error);
		return;
	}
	
	QVariantList vals = values.toList();
	if(vals.isEmpty()) {
		QMessageBox::warning(this, tr("Horaris Error"), tr("No results found"));
		return;
	}
	m_ui->options->setEnabled(true);
	m_ui->resultsView->setEnabled(true);
	m_ui->search->setEnabled(true);
	
	vals = vals.first().toList().first().toList();
	
	m_ui->options->clear();
	foreach(const QVariant& entry, vals) {
		QVariantList elem = entry.toList();
		QVariantList title = elem.first().toList();
		Q_ASSERT(title.size()==2);
		
		if(m_ui->options->count()==0) {
			m_ui->originLabel->setText(title.first().toString());
		}
		
		QListWidgetItem* wit = new QListWidgetItem(title.last().toString());
		wit->setData(Qt::UserRole, elem[1]);
		wit->setTextAlignment(Qt::AlignCenter);
		m_ui->options->addItem(wit);
	}
	m_ui->resultsView->setModel(m_model);
	
	if(m_ui->options->count()>0) {
		QFontMetrics f(m_ui->options->font());
		int width = 30;
		m_ui->resultsSplit->setSizes(QList<int>() << width << 300);
	}
	
	m_ui->options->setCurrentRow(0);
	sender()->deleteLater();
	
	int col = m_model->columnCount();
	int w = m_ui->resultsView->viewport()->width()/col-10;
	
	for(int i=0; i<col; i++)
		m_ui->resultsView->setColumnWidth(i, w);
}

void Horaris::restoreView()
{
	m_ui->stackedWidget->setCurrentIndex(0);
	m_tools->setVisible(true);
}

static QString serviceToName(const QString& servei)
{
	if(servei == "Rodalia") return "Rodalia";
	
	return "renfe";
}

void Horaris::timetableChanged(int pos)
{
	if(pos<0)
		return;
	
	QVariantList timetableData = m_ui->options->item(pos)->data(Qt::UserRole).toList();
	
	m_model->removeRows(0, m_model->rowCount());
	
	bool today = QDate::currentDate()==m_ui->date->dateTime().date();
	QTime time = QTime::currentTime();
	int latestbefore = 0, now=time.hour()*100+time.minute();
	foreach(const QVariant& row, timetableData) {
		QVariantList rowdata = row.toList();
		QList<QStandardItem*> rowitems;
		QString departure = rowdata[0].toString();
		
		rowitems += new QStandardItem(departure);
		rowitems += new QStandardItem(rowdata[2].toString());
		rowitems += new QStandardItem(rowdata[1].toString());
		
		rowitems.first()->setIcon(QIcon::fromTheme(serviceToName(rowdata.last().toString())));
		m_model->appendRow(rowitems);
		
		if(now>departure.remove(':').toInt() && today) {
			latestbefore++;
			foreach(QStandardItem* it, rowitems)
				it->setForeground(palette().brush(QPalette::Disabled, QPalette::ButtonText));
		}
		
		foreach(QStandardItem* it, rowitems)
			it->setTextAlignment(Qt::AlignCenter);
	}
	m_ui->resultsView->setModel(m_model);
	
	QModelIndex idx = m_model->index(qMin(latestbefore, m_model->rowCount()-1), 0);
	m_ui->resultsView->scrollTo(idx, QAbstractItemView::PositionAtTop);
}

#include "horaris.moc"
