/* Copyright: (c) 2005-2009 Nokia Corporation
 *
 * The code examples copyrighted by Nokia Corporation that are included to
 * this material are licensed to you under following MIT-style License:
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
 * CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
 * TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 * SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#include "example_desktop_widget.h"
#include <libhildondesktopmm/init.h>
#include <hildonmm/init.h>
#include <hildonmm/banner.h>
#include <gtkmm/button.h>
#include <iostream>

void hd_plugin_module_load(HDPluginModule* plugin)
{									
  HildonDesktop::init();
  Hildon::init();
  const GType gtype = ExampleDesktopWidget::get_type(G_TYPE_MODULE(plugin));
  hd_plugin_module_add_type(plugin, gtype);
}					

void hd_plugin_module_unload (HDPluginModule *plugin)
{
}


ExampleDesktopWidget::ExampleDesktopWidget()
: Glib::ObjectBase("libhildondesktopmmExampleDesktopWidget") //A custom GType name so our desktop widget can be identified and instatiated by this name.
{
  GType gtype = G_OBJECT_TYPE(gobj());  //The custom GType created in the Object constructor, from the typeid.
  
  //Commenting this out will crash hildon-home:
  HildonDesktop::HomePluginItem::add_interface(gtype);

  /* Instead of drawing in expose_event, we could add a widget:    
  Gtk::Button* button = Gtk::manage( new Gtk::Button("Test Button") );
  button->show();
  add(*button);
  */

  // Update the clock once a second:
  m_timeout_handler = Glib::signal_timeout().connect(
    sigc::mem_fun(*this, &ExampleDesktopWidget::on_timeout), 1000);
  
  // Allow this widget to handle button-press events:
  // Note that add_events would only work after the widget is realized:
  const Gdk::EventMask mask = get_events() | Gdk::BUTTON_PRESS_MASK;
  set_events(mask);
  
    
  //gtkmm on Maemo does not have default signal handlers:
  signal_realize().connect(
    sigc::mem_fun(*this, &ExampleDesktopWidget::on_realize));
  signal_expose_event().connect(
    sigc::mem_fun(*this, &ExampleDesktopWidget::on_expose_event), false);
  signal_button_press_event().connect(
    sigc::mem_fun(*this, &ExampleDesktopWidget::on_button_press_event), false);
  
  /* Specify that a settings button should be shown in layout mode, 
   * and handle a request to configure the settings: */
  set_settings();
  
  set_property("show-settings", true);    
}

ExampleDesktopWidget::~ExampleDesktopWidget()
{
}



bool ExampleDesktopWidget::on_timeout()
{ 
  // Update the time:
  time_t timet;
  time (&timet);
  localtime_r (&timet, &m_time);

  // Invalidate the drawing:
  Glib::RefPtr<Gdk::Window> gdkwindow = get_window();
  if (!gdkwindow)
    return true;

  // Redraw the cairo canvas completely by exposing it:
  Gdk::Region region = gdkwindow->get_clip_region();
    
  gdkwindow->invalidate_region(region);
  gdkwindow->process_updates(true);

  return true; // Keep running this event.
}

void ExampleDesktopWidget::on_realize()
{
  // Use An RGBA colormap rather than RGB, 
  // so we can use transparency in our expose_event() implementation:
  Glib::RefPtr<const Gdk::Screen> screen = get_screen();
  if(!screen)
    return;
    
  set_colormap(screen->get_rgba_colormap());

  set_app_paintable();


  //HildonDesktop::HomePluginItem::on_realize();
}


/* This use of cairo is based on Dave Madeley's example code from 
 * http://gnomejournal.org/article/36/writing-a-widget-using-cairo-and-gtk28-part-2
 */
void ExampleDesktopWidget::draw(const Cairo::RefPtr<Cairo::Context>& context)
{
  cairo_t *cr = context->cobj();
  
  double x, y = 0;
  double radius = 0;
  int i = 0;
  int hours, minutes, seconds = 0;

  GtkWidget *widget = GTK_WIDGET(gobj());
  
  /** Make the background transparent:
   * Note that this works because we used gtk_widget_set_colormap() in 
   * our realize() implementation.
   */
  cairo_set_source_rgba(cr, 1.0, 1.0, 1.0, 0.0);
  cairo_set_operator (cr, CAIRO_OPERATOR_SOURCE);
	cairo_paint(cr);
    
  x = widget->allocation.width / 2;
  y = widget->allocation.height / 2;
  radius = 
    MIN (widget->allocation.width / 2, widget->allocation.height / 2) - 5;

  /* The clock's back surface:  */
  cairo_arc (cr, x, y, radius, 0, 2 * M_PI);
  /* TODO: Clutter seems to complain about some colors.
   * How can we restrict the choice of colors? */
  cairo_set_source_rgb (cr, 
    1, 1, 1);
  cairo_fill_preserve (cr);
  cairo_set_source_rgb (cr, 0, 0, 0);
  cairo_stroke (cr);

  /* The clock's tick marks: */
  for (i = 0; i < 12; i++)
  {
    int inset = 0;
  
    cairo_save (cr); /* stack-pen-size */
    
    if (i % 3 == 0)
    {
      inset = 0.2 * radius;
    }
    else
    {
      inset = 0.1 * radius;
      cairo_set_line_width (cr, 0.5 *
          cairo_get_line_width (cr));
    }
    
    cairo_move_to (cr,
      x + (radius - inset) * cos (i * M_PI / 6),
      y + (radius - inset) * sin (i * M_PI / 6));
    cairo_line_to (cr,
      x + radius * cos (i * M_PI / 6),
      y + radius * sin (i * M_PI / 6));
    cairo_stroke (cr);
    cairo_restore (cr); /* stack-pen-size */
  }

  /* The clock hands: */
  hours = m_time.tm_hour;
  minutes = m_time.tm_min + 0; /* TODO: clock->minute_offset; */
  seconds = m_time.tm_sec;
  
  /* Hour hand:
   * The hour hand is rotated 30 degrees (pi/6 r) per hour +
   * 1/2 a degree (pi/360 r) per minute.
   */
  cairo_save (cr);
  cairo_set_line_width (cr, 2.5 * cairo_get_line_width (cr));
  cairo_move_to (cr, x, y);
  cairo_line_to (cr, 
    x + radius / 2 * sin (M_PI / 6 * hours + M_PI / 360 * minutes),
    y + radius / 2 * -cos (M_PI / 6 * hours + M_PI / 360 * minutes));
  cairo_stroke (cr);
  cairo_restore (cr);
  
  /* Minute hand:
   * The minute hand is rotated 6 degrees (pi/30 r) per minute.
   */
  cairo_move_to (cr, x, y);
  cairo_line_to (cr, 
    x + radius * 0.75 * sin (M_PI / 30 * minutes),
    y + radius * 0.75 * -cos (M_PI / 30 * minutes));
  cairo_stroke (cr);
  
  /* Seconds hand:
   * Operates identically to the minute hand.
   */
  cairo_save (cr);
  cairo_set_source_rgb (cr, 1, 0, 0); /* red */
  cairo_move_to (cr, x, y);
  cairo_line_to (cr, 
    x + radius * 0.7 * sin (M_PI / 30 * seconds),
    y + radius * 0.7 * -cos (M_PI / 30 * seconds));
  cairo_stroke (cr);
  cairo_restore (cr);
}

bool ExampleDesktopWidget::on_expose_event(GdkEventExpose* event)
{
  Glib::RefPtr<Gdk::Window> gdkwindow = get_window();
  if (!gdkwindow)
    return true;
    
  Cairo::RefPtr<Cairo::Context> context = gdkwindow->create_cairo_context();
  
  /* Clip only the exposed area of the cairo context,  
   * to potentially avoid unnecessary drawing:
   */
  context->rectangle(
    event->area.x, event->area.y,
    event->area.width, event->area.height);
  context->clip();
  
  draw(context);

  
  //return HildonDesktop::HomePluginItem::on_expose_event();
  
  return true;
}

bool ExampleDesktopWidget::on_button_press_event(GdkEventButton* event)
{
  /*
  gchar* text = g_strdup_printf("Current Time: %s", asctime(&(m_time))); 
  Hildon::Banner::show_information(*this, text);
  g_free (text);
  */
  
  return true; /* Prevent further handling of this signal. */
}



