/*
 * vim:ts=8:sw=3:sts=8:noexpandtab:cino=>5n-3f0^-2{2
 */

/*
 * Copyright (C) 2005-2007 Nokia Corporation.
 * Copyright (C) 2007-2009 INdT.
 *
 * Contact: Mohammad Anwari <Mohammad.Anwari@nokia.com>
 * Contact: Eduardo Lima (Etrunko) <eduardo.lima@openbossa.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 */

#ifndef _HILDON_IM_PROTOCOL_H
#define _HILDON_IM_PROTOCOL_H

#include <X11/X.h>

#ifdef __cplusplus
extern "C" {
#endif

/* IM atoms for each message type */
typedef enum
{
  HILDON_IM_WINDOW,
  HILDON_IM_ACTIVATE,
  HILDON_IM_INSERT_UTF8,
  HILDON_IM_SURROUNDING,
  HILDON_IM_SURROUNDING_CONTENT,
  HILDON_IM_KEY_EVENT,
  HILDON_IM_COM,
  HILDON_IM_CLIPBOARD_COPIED,
  HILDON_IM_CLIPBOARD_SELECTION_QUERY,
  HILDON_IM_CLIPBOARD_SELECTION_REPLY,
  HILDON_IM_PREEDIT_COMMITTED,
  HILDON_IM_PREEDIT_COMMITTED_CONTENT,

  /* always last */
  HILDON_IM_NUM_ATOMS
} Hildon_IM_Atom;

/* IM atom names */
#define HILDON_IM_WINDOW_NAME                    "_HILDON_IM_WINDOW"
#define HILDON_IM_ACTIVATE_NAME                  "_HILDON_IM_ACTIVATE"
#define HILDON_IM_SURROUNDING_NAME               "_HILDON_IM_SURROUNDING"
#define HILDON_IM_SURROUNDING_CONTENT_NAME       "_HILDON_IM_SURROUNDING_CONTENT"
#define HILDON_IM_KEY_EVENT_NAME                 "_HILDON_IM_KEY_EVENT"
#define HILDON_IM_INSERT_UTF8_NAME               "_HILDON_IM_INSERT_UTF8"
#define HILDON_IM_COM_NAME                       "_HILDON_IM_COM"
#define HILDON_IM_CLIPBOARD_COPIED_NAME          "_HILDON_IM_CLIPBOARD_COPIED"
#define HILDON_IM_CLIPBOARD_SELECTION_QUERY_NAME "_HILDON_IM_CLIPBOARD_SELECTION_QUERY"
#define HILDON_IM_CLIPBOARD_SELECTION_REPLY_NAME "_HILDON_IM_CLIPBOARD_SELECTION_REPLY"
#define HILDON_IM_PREEDIT_COMMITTED_NAME         "_HILDON_IM_PREEDIT_COMMITTED"
#define HILDON_IM_PREEDIT_COMMITTED_CONTENT_NAME "_HILDON_IM_PREEDIT_COMMITTED_CONTENT"

/* IM ClientMessage formats */
#define HILDON_IM_WINDOW_ID_FORMAT 32
#define HILDON_IM_ACTIVATE_FORMAT 8
#define HILDON_IM_SURROUNDING_FORMAT 8
#define HILDON_IM_SURROUNDING_CONTENT_FORMAT 8
#define HILDON_IM_KEY_EVENT_FORMAT 8
#define HILDON_IM_INSERT_UTF8_FORMAT 8
#define HILDON_IM_COM_FORMAT 8
#define HILDON_IM_CLIPBOARD_FORMAT 32
#define HILDON_IM_CLIPBOARD_SELECTION_REPLY_FORMAT 32
#define HILDON_IM_PREEDIT_COMMITTED_FORMAT 8
#define HILDON_IM_PREEDIT_COMMITTED_CONTENT_FORMAT 8

/* IM commands and notifications, from context to IM process */
typedef enum
{
  HILDON_IM_MODE,        /* Update the hildon-input-mode property */
  HILDON_IM_SHOW,        /* Show the IM UI */
  HILDON_IM_HIDE,        /* Hide the IM UI */
  HILDON_IM_UPP,         /* Uppercase autocap state at cursor */
  HILDON_IM_LOW,         /* Lowercase autocap state at cursor */
  HILDON_IM_DESTROY,     /* DEPRECATED */
  HILDON_IM_CLEAR,       /* Clear the IM UI state */
  HILDON_IM_SETCLIENT,   /* Set the client window */
  HILDON_IM_SETNSHOW,    /* Set the client and show the IM window */
  HILDON_IM_SELECT_ALL,  /* Select the text in the plugin */

  HILDON_IM_SHIFT_LOCKED,
  HILDON_IM_SHIFT_UNLOCKED,
  HILDON_IM_MOD_LOCKED,
  HILDON_IM_MOD_UNLOCKED,

  /* always last */
  HILDON_IM_NUM_COMMANDS
} Hildon_IM_Command;

/* IM communications, from IM process to context */
typedef enum
{
  HILDON_IM_CONTEXT_HANDLE_ENTER,           /* Virtual enter activated */
  HILDON_IM_CONTEXT_HANDLE_TAB,             /* Virtual tab activated */
  HILDON_IM_CONTEXT_HANDLE_BACKSPACE,       /* Virtual backspace activated */
  HILDON_IM_CONTEXT_HANDLE_SPACE,           /* Virtual space activated */
  HILDON_IM_CONTEXT_CONFIRM_SENTENCE_START, /* Query the autocap state at cursor */
  HILDON_IM_CONTEXT_FLUSH_PREEDIT,          /* Finalize the preedit to the client widget */
  HILDON_IM_CONTEXT_CANCEL_PREEDIT,          /* Clean the preedit buffer */

  /* See Hildon_IM_Commit_Mode for a description of the commit modes */
  HILDON_IM_CONTEXT_BUFFERED_MODE,
  HILDON_IM_CONTEXT_DIRECT_MODE,
  HILDON_IM_CONTEXT_REDIRECT_MODE,
  HILDON_IM_CONTEXT_SURROUNDING_MODE,
  HILDON_IM_CONTEXT_PREEDIT_MODE,

  HILDON_IM_CONTEXT_CLIPBOARD_COPY,            /* Copy client selection to clipboard */
  HILDON_IM_CONTEXT_CLIPBOARD_CUT,             /* Cut client selection to clipboard */
  HILDON_IM_CONTEXT_CLIPBOARD_PASTE,           /* Paste clipboard selection to client */
  HILDON_IM_CONTEXT_CLIPBOARD_SELECTION_QUERY, /* Query if the client has an active selection */
  HILDON_IM_CONTEXT_REQUEST_SURROUNDING,       /* Request the content surrounding the cursor */
  HILDON_IM_CONTEXT_REQUEST_SURROUNDING_FULL,  /* Request the contents of the text widget */
  HILDON_IM_CONTEXT_WIDGET_CHANGED,            /* IM detected that the client widget changed */
  HILDON_IM_CONTEXT_OPTION_CHANGED,            /* The OptionMask for the active context is updated */
  HILDON_IM_CONTEXT_CLEAR_STICKY,              /* Clear the sticky key state */
  HILDON_IM_CONTEXT_ENTER_ON_FOCUS,            /* Generate a virtual enter key event on focus in */

  HILDON_IM_CONTEXT_SPACE_AFTER_COMMIT,        /* Append a space when the preedit text is committed */
  HILDON_IM_CONTEXT_NO_SPACE_AFTER_COMMIT,     /* Do not append said space */

  /* always last */
  HILDON_IM_CONTEXT_NUM_COM
} Hildon_IM_Communication;

/* IM context toggle options */
typedef enum {
  HILDON_IM_AUTOCASE          = 1 << 0, /* Suggest case based on cursor's position in sentence */
  HILDON_IM_AUTOCORRECT       = 1 << 1, /* Limited automatic error correction of commits */
  HILDON_IM_AUTOLEVEL_NUMERIC = 1 << 2, /* Default to appropriate key-level in numeric-only clients */
  HILDON_IM_LOCK_LEVEL        = 1 << 3, /* Lock the effective key-level at pre-determined value */
} Hildon_IM_Option_Mask;

/* IM trigger types, i.e. what is causing the IM plugin to active */
typedef enum
{
  HILDON_IM_TRIGGER_NONE = -1, /* Reserved for non-UI plugins not requiring activation */
  HILDON_IM_TRIGGER_STYLUS,    /* The user poked the screen with the stylus; not used in Fremantle */
  HILDON_IM_TRIGGER_FINGER,    /* The user poked the screen with his finger */
  HILDON_IM_TRIGGER_KEYBOARD,   /* The user pressed a key */
  HILDON_IM_TRIGGER_UNKNOWN
} Hildon_IM_Trigger;

/* Commit modes
   Determines how text is inserted into the client widget

   Buffered mode:  Each new commit replaces any previous commit to the
   client widget until FLUSH_PREEDIT is called.

   Direct mode (default): Each commit is immediately appended to the
   client widget at the cursor position.

   Redirect mode: Proxies input and cursor movement from one text widget
   into another (potentially off-screen) widget. Used when implementing
   fullscreen IM plugins for widgets that contain text formatting.

   Surrounding mode: Each commit replaces the current text surrounding
   the cursor position (see gtk_im_context_get_surrounding).
*/
typedef enum
{
  HILDON_IM_COMMIT_DIRECT,
  HILDON_IM_COMMIT_REDIRECT,
  HILDON_IM_COMMIT_SURROUNDING,
  HILDON_IM_COMMIT_BUFFERED,
  HILDON_IM_COMMIT_PREEDIT,
} Hildon_IM_Commit_Mode;

/* Command activation message, from context to IM (see Hildon_IM_Command) */
typedef struct
{
  Window input_window;
  Window app_window;
  Hildon_IM_Command cmd;
  int input_mode;
  Hildon_IM_Trigger trigger;
} Hildon_IM_Activate_Message;

/* event.xclient.data may not exceed 20 bytes -> Hildon_IM_Insert_Utf8_Message
   may not exceed 20 bytes. So the maximum size of message buffer is
   20 - sizeof(msg_flags) = 16. */
#define HILDON_IM_CLIENT_MESSAGE_BUFFER_SIZE (20 - sizeof(int))

/* Text insertion message, from IM to context */
typedef struct
{
  int msg_flag;
  char utf8_str[HILDON_IM_CLIENT_MESSAGE_BUFFER_SIZE];
} Hildon_IM_Insert_Utf8_Message;

/* Message carrying surrounding interpretation info, sent by both IM and context */
typedef struct
{
  Hildon_IM_Commit_Mode commit_mode;
  int offset_is_relative;
  int cursor_offset;
} Hildon_IM_Surrounding_Message;

/* The surrounding text, sent by both IM and context */
typedef struct
{
  int msg_flag;
  char surrounding[HILDON_IM_CLIENT_MESSAGE_BUFFER_SIZE];
} Hildon_IM_Surrounding_Content_Message;

/* Message carrying information about the committed preedit */
typedef struct
{
  int msg_flag;
  Hildon_IM_Commit_Mode commit_mode;
} Hildon_IM_Preedit_Committed_Message;

/* The committed preedit text, sent by context */
typedef struct
{
  int msg_flag;
  char committed_preedit[HILDON_IM_CLIENT_MESSAGE_BUFFER_SIZE];
} Hildon_IM_Preedit_Committed_Content_Message;

/* Key event message, from context to IM */
typedef struct
{
  Window input_window;
  int type;
  unsigned int state;
  unsigned int keyval;
  unsigned int hardware_keycode;
} Hildon_IM_Key_Event_Message;

/* Type markers for IM messages that span several ClientMessages */
enum
{
  HILDON_IM_MSG_START,
  HILDON_IM_MSG_CONTINUE,
  HILDON_IM_MSG_END
};

/* Communication message from IM to context */
typedef struct
{
  Window input_window;
  Hildon_IM_Communication type;
  Hildon_IM_Option_Mask options;

} Hildon_IM_Com_Message;

#ifdef __cplusplus
}
#endif

#endif
