/*
 * vim:ts=8:sw=3:sts=8:noexpandtab:cino=>5n-3f0^-2{2
 */

/* Operations on UTF-8 strings.
 *
 * Copyright (C) 1999 Tom Tromey
 * Copyright (C) 2000 Red Hat, Inc.
 * Copyright (C) 2007 INdT
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "config.h"

#include "hildon_imf_utils.h"

#define UTF8_COMPUTE(Char, Mask, Len)					      \
  if (Char < 128)							      \
    {									      \
      Len = 1;								      \
      Mask = 0x7f;							      \
    }									      \
  else if ((Char & 0xe0) == 0xc0)					      \
    {									      \
      Len = 2;								      \
      Mask = 0x1f;							      \
    }									      \
  else if ((Char & 0xf0) == 0xe0)					      \
    {									      \
      Len = 3;								      \
      Mask = 0x0f;							      \
    }									      \
  else if ((Char & 0xf8) == 0xf0)					      \
    {									      \
      Len = 4;								      \
      Mask = 0x07;							      \
    }									      \
  else if ((Char & 0xfc) == 0xf8)					      \
    {									      \
      Len = 5;								      \
      Mask = 0x03;							      \
    }									      \
  else if ((Char & 0xfe) == 0xfc)					      \
    {									      \
      Len = 6;								      \
      Mask = 0x01;							      \
    }									      \
  else									      \
    Len = -1;

#define UTF8_GET(Result, Chars, Count, Mask, Len)			      \
  (Result) = (Chars)[0] & (Mask);					      \
  for ((Count) = 1; (Count) < (Len); ++(Count))				      \
    {									      \
      if (((Chars)[(Count)] & 0xc0) != 0x80)				      \
	{								      \
	  (Result) = -1;						      \
	  break;							      \
	}								      \
      (Result) <<= 6;							      \
      (Result) |= ((Chars)[(Count)] & 0x3f);				      \
    }

#define IS(Type, Class) (((unsigned int)1 << (Type)) & (Class))
#define OR(Type, Rest)  (((unsigned int)1 << (Type)) | (Rest))

#define TTYPE_PART1(Page, Char) \
  ((type_table_part1[Page] >= HILDON_IMF_UTILS_UNICODE_MAX_TABLE_INDEX) \
   ? (type_table_part1[Page] - HILDON_IMF_UTILS_UNICODE_MAX_TABLE_INDEX) \
   : (type_data[type_table_part1[Page]][Char]))

#define TTYPE_PART2(Page, Char) \
  ((type_table_part2[Page] >= HILDON_IMF_UTILS_UNICODE_MAX_TABLE_INDEX) \
   ? (type_table_part2[Page] - HILDON_IMF_UTILS_UNICODE_MAX_TABLE_INDEX) \
   : (type_data[type_table_part2[Page]][Char]))

#define TYPE(Char) \
  (((Char) <= HILDON_IMF_UTILS_UNICODE_LAST_CHAR_PART1) \
   ? TTYPE_PART1 ((Char) >> 8, (Char) & 0xff) \
   : (((Char) >= 0xe0000 && (Char) <= HILDON_IMF_UTILS_UNICODE_LAST_CHAR) \
      ? TTYPE_PART2 (((Char) - 0xe0000) >> 8, (Char) & 0xff) \
      : HILDON_IMF_UTILS_UNICODE_UNASSIGNED))

#define hildon_imf_utils_utf8_next_char(p) (char *)((p) + hildon_imf_utils_utf8_skip[*(const unsigned char *)(p)])

static const char hildon_imf_utils_utf8_skip_data[256] = {
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,
  3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,4,4,4,4,4,4,4,4,5,5,5,5,6,6,1,1
};

const char *const hildon_imf_utils_utf8_skip = hildon_imf_utils_utf8_skip_data;

uint32_t
hildon_imf_utils_utf8_char_get(const char *p)
{
   int i, mask = 0, len;
   uint32_t result;
   unsigned char c = (unsigned char) *p;

   UTF8_COMPUTE(c, mask, len);
   if (len == -1)
     return (uint32_t) -1;
   UTF8_GET(result, p, i, mask, len);

   return result;
}

long
hildon_imf_utils_utf8_strlen(const char *p, unsigned int max)
{
   long len = 0;
   const char *start = p;

   if (max < 0)
     {
	while (*p)
	  {
	     p = hildon_imf_utils_utf8_next_char(p);
	     ++len;
	  }
     }
   else
     {
	if (max == 0 || !*p)
	  return 0;

	p = hildon_imf_utils_utf8_next_char(p);

	while (p - start < max && *p)
	  {
	     ++len;
	     p = hildon_imf_utils_utf8_next_char(p);
	  }

	/* only do the last len increment if we got a complete
	 * char (don't count partial chars)
	 */
	if (p - start <= max)
	  ++len;
     }

   return len;
}

char *
hildon_imf_utils_utf8_find_prev_char(const char *str,
				     const char *p)
{
   for (--p; p >= str; --p)
     {
	if ((*p & 0xc0) != 0x80)
	  return (char *)p;
     }
   return NULL;
}

int
hildon_imf_utils_unichar_isspace(uint32_t c)
{
   switch (c)
     {
      /* special-case these since Unicode thinks they are not spaces */
      case '\t':
      case '\n':
      case '\r':
      case '\f':
	 return 1;
	 break;
      default:
	  {
	     return IS(TYPE(c),
		    OR(HILDON_IMF_UTILS_UNICODE_SPACE_SEPARATOR,
		    OR(HILDON_IMF_UTILS_UNICODE_LINE_SEPARATOR,
		    OR(HILDON_IMF_UTILS_UNICODE_PARAGRAPH_SEPARATOR,
		        0)))) ? 1 : 0;
	  }
      break;
    }
}

int
hildon_imf_utils_unichar_ispunct(uint32_t c)
{
   return IS(TYPE(c),
	     OR(HILDON_IMF_UTILS_UNICODE_CONNECT_PUNCTUATION,
	     OR(HILDON_IMF_UTILS_UNICODE_DASH_PUNCTUATION,
	     OR(HILDON_IMF_UTILS_UNICODE_CLOSE_PUNCTUATION,
	     OR(HILDON_IMF_UTILS_UNICODE_FINAL_PUNCTUATION,
	     OR(HILDON_IMF_UTILS_UNICODE_INITIAL_PUNCTUATION,
	     OR(HILDON_IMF_UTILS_UNICODE_OTHER_PUNCTUATION,
	     OR(HILDON_IMF_UTILS_UNICODE_OPEN_PUNCTUATION,
	     OR(HILDON_IMF_UTILS_UNICODE_CURRENCY_SYMBOL,
	     OR(HILDON_IMF_UTILS_UNICODE_MODIFIER_SYMBOL,
	     OR(HILDON_IMF_UTILS_UNICODE_MATH_SYMBOL,
	     OR(HILDON_IMF_UTILS_UNICODE_OTHER_SYMBOL,
	     0)))))))))))) ? 1 : 0;
}

