/*
 * vim:ts=8:sw=3:sts=8:noexpandtab:cino=>5n-3f0^-2{2
 */

/*
 * Copyright (C) 2005-2007 Nokia Corporation.
 * Copyright (C) 2007 INdT.
 *
 * Contact: Mohammad Anwari <Mohammad.Anwari@nokia.com>
 * Contact: Andre Moreira Magalhaes <andrunko@gmail.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 */

#include "config.h"

#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <X11/Xatom.h>
#include <X11/Xlib.h>
#include <X11/keysym.h>
#include <X11/extensions/XTest.h>

#include <Ecore.h>
#include <Ecore_X.h>
#include <Ecore_IMF.h>

#include "hildon_im_protocol.h"
#include "hildon_imf_utils.h"

#define HILDON_IM_DEFAULT_LAUNCH_DELAY 0.07

typedef struct _Hildon_IMF_Context Hildon_IMF_Context;

struct _Hildon_IMF_Context {
   Ecore_X_Display *display;
   Ecore_X_Window window;
   Ecore_X_Window input_window;
   Ecore_Event_Handler *client_message_event_handler;

   Ecore_IMF_Input_Mode input_mode;
   Hildon_IM_Trigger trigger;
   Hildon_IM_Commit_Mode commit_mode;

   char *pre_edit_buffer;

   unsigned int launch_delay;
   Ecore_Timer *launch_delay_timer;

   /* State */
   int last_internal_change;
   int has_focus;
   int auto_correction;

   /* Keep track on cursor position to prevent unnecessary calls */
   int prev_cursor_pos;

   char *surrounding;
};

static void _hildon_imf_context_del(Ecore_IMF_Context *ctx);
static void _hildon_imf_context_client_window_set(Ecore_IMF_Context *ctx, void *window);
static void _hildon_imf_context_show(Ecore_IMF_Context *ctx);
static void _hildon_imf_context_hide(Ecore_IMF_Context *ctx);
static void _hildon_imf_context_focus_in(Ecore_IMF_Context *ctx);
static void _hildon_imf_context_focus_out(Ecore_IMF_Context *ctx);
static void _hildon_imf_context_reset(Ecore_IMF_Context *ctx);
static void _hildon_imf_context_cursor_position_set(Ecore_IMF_Context *ctx, int cursor_pos);
static void _hildon_imf_context_input_mode_set(Ecore_IMF_Context *ctx, Ecore_IMF_Input_Mode input_mode);
static int _hildon_imf_context_filter_event(Ecore_IMF_Context *ctx, Ecore_IMF_Event_Type type, Ecore_IMF_Event *event);

static Ecore_X_Window
_window_id_get(Ecore_X_Display *display, Ecore_X_Atom window_atom)
{
   Ecore_X_Window window = None;
   ecore_x_window_prop_window_get(DefaultRootWindow(display),
				  window_atom, &window, 1);
   return window;
}

static void
_send_command(Hildon_IMF_Context *ctxd, Hildon_IM_Command cmd)
{
   Ecore_X_Window keyboard;
   Hildon_IM_Activate_Message msg;

   keyboard = _window_id_get(ctxd->display,
			     ecore_x_atom_get(HILDON_IM_WINDOW_NAME));

   msg.input_window = ctxd->input_window;
   if (cmd != HILDON_IM_HIDE)
     msg.app_window = ctxd->window;
   msg.cmd = cmd;
   msg.input_mode = ctxd->input_mode;
   msg.trigger = ctxd->trigger;

   ecore_x_client_message8_send(keyboard,
				ecore_x_atom_get(HILDON_IM_ACTIVATE_NAME),
				&msg, sizeof(Hildon_IM_Activate_Message));
   ecore_x_sync();
}

static void
_send_event(Hildon_IMF_Context *ctxd, Ecore_X_Window window, XEvent *event)
{
   if (window == None)
     return;

   event->xclient.type = ClientMessage;
   event->xclient.window = window;

   XSendEvent(ctxd->display, window, False, 0, event);
   XSync(ctxd->display, False );
}

static char *
_get_next_packet_start(char *str)
{
  char *candidate, *good;

  candidate = good = str;

  while (*candidate)
  {
    candidate++;
    if (candidate - str >= HILDON_IM_CLIENT_MESSAGE_BUFFER_SIZE)
    {
      return good;
    }
    good = candidate;
  }

  /* The whole string is small enough */
  return candidate;
}

static void
_send_surrounding_header(Hildon_IMF_Context *ctxd, int offset)
{
   Ecore_X_Window keyboard;
   Hildon_IM_Surrounding_Message *surrounding_msg = NULL;
   XEvent event;

   keyboard = _window_id_get(ctxd->display,
			     ecore_x_atom_get(HILDON_IM_WINDOW_NAME));

   /* Send the cursor offset in the surrounding */
   memset(&event, 0, sizeof(XEvent));
   event.xclient.message_type = ecore_x_atom_get(HILDON_IM_SURROUNDING_NAME);
   event.xclient.format = HILDON_IM_SURROUNDING_FORMAT;

   surrounding_msg = (Hildon_IM_Surrounding_Message *) &event.xclient.data;
   surrounding_msg->commit_mode = ctxd->commit_mode;
   surrounding_msg->cursor_offset = offset;

   _send_event(ctxd, keyboard, &event);
}

/* Send the text of the client widget surrounding the active cursor position,
   as well as the cursor position in the surrounding, to the IM */
static void
_send_surrounding(Ecore_IMF_Context *ctx)
{
   Hildon_IMF_Context *ctxd = ecore_imf_context_data_get(ctx);
   Ecore_X_Window keyboard;
   XEvent event;
   Hildon_IM_Surrounding_Content_Message *surrounding_content_msg = NULL;
   int flag;
   char *surrounding, *str;
   int cpos;
   int has_surrounding;

   flag = HILDON_IM_MSG_START;
   keyboard = _window_id_get(ctxd->display,
			     ecore_x_atom_get(HILDON_IM_WINDOW_NAME));

   has_surrounding = ecore_imf_context_surrounding_get(ctx,
						       &surrounding, &cpos);
   if (!has_surrounding)
     {
	_send_surrounding_header(ctxd, 0);
	return;
     }

   /* Split surrounding context into pieces that are small enough
      to send in a x message */
   str = surrounding;
   do
     {
	char *next_start;
	unsigned int len;

	next_start = _get_next_packet_start(str);
	len = next_start - str;

	/* this call will take care of adding the null terminator */
	memset(&event, 0, sizeof(XEvent));
	event.xclient.message_type = ecore_x_atom_get(HILDON_IM_SURROUNDING_CONTENT_NAME);
	event.xclient.format = HILDON_IM_SURROUNDING_CONTENT_FORMAT;

	surrounding_content_msg = (Hildon_IM_Surrounding_Content_Message *) &event.xclient.data;
	surrounding_content_msg->msg_flag = flag;
	memcpy(surrounding_content_msg->surrounding, str, len);

	_send_event(ctxd, keyboard, &event);

	str = next_start;
	flag = HILDON_IM_MSG_CONTINUE;
     } while (*str);

   _send_surrounding_header(ctxd, cpos);

   free(surrounding);
}

static void
_send_fake_key(Ecore_X_Display *display, KeySym keysym, int is_press)
{
   XTestFakeKeyEvent(display,
		     XKeysymToKeycode(display, keysym),
		     is_press, 0);
}

static int
_send_enter(Ecore_X_Display *display)
{
   _send_fake_key(display, XK_KP_Enter, 1);
   _send_fake_key(display, XK_KP_Enter, 0);
   return 0;
}

static int
_changes_case(const char *chr)
{
   uint32_t uni;
   if (chr == NULL)
     return 1;

   uni = hildon_imf_utils_utf8_char_get(chr);
   return (uni == '.' || uni == '!' || uni == '?' ||
	   uni == 0x00a1 || /* inverted exclamation mark */
	   uni == 0x00bf);  /* inverted question mark */
}

/* Updates the IM with the autocap state at the active cursor position */
static void
_check_sentence_start(Ecore_IMF_Context *ctx)
{
   Hildon_IMF_Context *ctxd = ecore_imf_context_data_get(ctx);
   char *surrounding, *iter;
   int cpos;
   int has_surrounding, space, ignore_punct = 1;
   uint32_t ch;

   if ((ctxd->input_mode & (ECORE_IMF_INPUT_MODE_ALPHA | ECORE_IMF_INPUT_MODE_AUTOCAP)) !=
       (ECORE_IMF_INPUT_MODE_ALPHA | ECORE_IMF_INPUT_MODE_AUTOCAP))
     {
	/* If autocap is off, but the mode contains alpha, send autocap message.
	 * The important part is that when entering a numerical entry the autocap
	 * is not defined, and the plugin sets the mode appropriate for the language */
	if (ctxd->input_mode & ECORE_IMF_INPUT_MODE_ALPHA)
	  _send_command(ctxd, HILDON_IM_LOW);
	return;
     }

   has_surrounding = ecore_imf_context_surrounding_get(ctx,
						       &surrounding,
						       &cpos);
   if (!has_surrounding)
     {
	_send_command(ctxd, HILDON_IM_LOW);
	return;
     }

   iter = surrounding + cpos;
   space = 0;

   while (1)
     {
	iter = hildon_imf_utils_utf8_find_prev_char(surrounding, iter);

	if (iter == NULL)
	  break;

	ch = hildon_imf_utils_utf8_char_get(iter);

	if (hildon_imf_utils_unichar_isspace(ch))
	  {
	     space = 1;
	     ignore_punct = 0;
	  }
	else if (ignore_punct && hildon_imf_utils_unichar_ispunct(ch))
	  continue;
	else
	  break;
     }

   if (iter == NULL || (space == 1 && _changes_case(iter)))
     _send_command(ctxd, HILDON_IM_UPP);
   else
     _send_command(ctxd, HILDON_IM_LOW);

   if (has_surrounding)
     free(surrounding);
}

/* Check whether auto correction should be made for the text. */
static int
_autocorrection_check_character(const char *p)
{
   uint32_t uni;
   int retval = 0;

   uni = hildon_imf_utils_utf8_char_get(p);
   switch (uni)
     {
      case '.':
      case ',':
      case '?':
      case '!':
	 retval = 1;
	 break;
      case 0x00A1:
      case 0x00BF:
	 retval = 2;
	 break;
      default:
	 retval = 0;
	 break;
     }
   return retval;
}

/* Ask the client widget to insert the specified text at the cursor
   position, by triggering the commit signal on the context */
static void
_insert_utf8(Ecore_IMF_Context *ctx, int flag, const char *text)
{
   Hildon_IMF_Context *ctxd = ecore_imf_context_data_get(ctx);
   int char_count;
   int cpos;
   int to_copy;
   char *surrounding, *text_clean = (char *) text;
   char tmp[3] = { 0, 0, 0 };
   int has_surrounding, free_text = 0;

   if (ctxd->auto_correction)
     {
	has_surrounding = ecore_imf_context_surrounding_get(ctx,
							    &surrounding,
							    &cpos);

	if (has_surrounding)
	  {
	     if (surrounding[cpos] == 0)
	       {
		  if (surrounding[cpos - 1] == ' ')
		    {
		       to_copy = _autocorrection_check_character(text);

		       if (to_copy > 0)
			 {
			    int len;

			    memcpy (tmp, text, to_copy);

			    len = strlen(tmp) + 1 + strlen(text + to_copy) + 1;
			    text_clean = malloc(len * sizeof(char));
			    snprintf(text_clean, len, "%s %s", tmp, text + to_copy);

			    ecore_imf_context_delete_surrounding_event_add(ctx, -1, 1);
			    free_text = 1;
			 }
		    }
	       }
	     free(surrounding);
	  }

	ctxd->auto_correction = 0;
     }

   ctxd->last_internal_change = 1;

   if (ctxd->pre_edit_buffer)
     {
	/* If string is empty, emit 'commit' signal to delete highlighted text */
	if (strlen(ctxd->pre_edit_buffer) == 0)
	  {
	     ecore_imf_context_commit_event_add(ctx, "");
	  }
	else
	  {
	     char_count = hildon_imf_utils_utf8_strlen(ctxd->pre_edit_buffer, -1);
	     ecore_imf_context_delete_surrounding_event_add(ctx, -char_count, char_count);
	  }

	if (flag == HILDON_IM_MSG_START)
	  {
	     free(ctxd->pre_edit_buffer);
	     ctxd->pre_edit_buffer = strdup(text_clean);
	  }
	else
	  {
	     ctxd->pre_edit_buffer = (char *) realloc(ctxd->pre_edit_buffer,
					              strlen(ctxd->pre_edit_buffer) + strlen(text_clean) + 1);
	     strcat(ctxd->pre_edit_buffer, text_clean);
	  }

	if (free_text == 1)
	  {
	     free(text_clean);
	     free_text = 0;
	  }

	text_clean = ctxd->pre_edit_buffer;
     }
   else
     {
	/* first commit "" to delete highlighted text */
	ecore_imf_context_commit_event_add(ctx, "");
     }

   /* This last "commit" signal adds the actual text. We're assuming it sends
      0 or 1 "changed" signals (we try to guarantee that by sending a "" commit
      above to delete highlights).

      If we get more than one "changed" signal, it means that the
      application's "changed" signal handler went and changed the text
      as a result of the change, and we need to clear IM. */
   ctxd->last_internal_change = 1;
   ecore_imf_context_commit_event_add(ctx, text_clean);

   if (free_text == 1)
     free(text_clean);
}

static void
_commit_preedit_data(Hildon_IMF_Context *ctxd)
{
  if (ctxd->pre_edit_buffer)
    {
	free(ctxd->pre_edit_buffer);
	ctxd->pre_edit_buffer = strdup("");
    }
}

static void
_check_commit_mode(Hildon_IMF_Context *ctxd)
{
   if (ctxd->commit_mode == HILDON_IM_COMMIT_REDIRECT)
     ctxd->commit_mode = HILDON_IM_COMMIT_SURROUNDING;
}

static void
_commit_surrounding(Ecore_IMF_Context *ctx)
{
   Hildon_IMF_Context *ctxd = ecore_imf_context_data_get(ctx);
   char *surrounding;
   int cpos;
   int has_surrounding;

   has_surrounding = ecore_imf_context_surrounding_get(ctx,
						       &surrounding,
						       &cpos);
   if (has_surrounding)
     {
	int offset_start, offset_end;
	char *str, *start, *end = NULL;

	str = &surrounding[cpos];

	start = surrounding;
	for (end = str; end[0] != '\0'; end++);

	if (end - start > 0)
	  {
	     /* Offset to the start of the line, from the insertion point */
	     offset_start = str - start;
	     /* Offset to the end of the line */
	     offset_end = end - start;

	     /* Remove the surrounding context on the line with the cursor */
	     ecore_imf_context_delete_surrounding_event_add(ctx,
							    -offset_start,
							    offset_end);
	  }
     }

   /* Place the new surrounding context at the insertion point */
   ecore_imf_context_commit_event_add(ctx, ctxd->surrounding);

   if (has_surrounding)
     free(surrounding);
}

static int
_client_message_cb(void *data, int type, void *event)
{
   Ecore_IMF_Context *ctx = data;
   Hildon_IMF_Context *ctxd = ecore_imf_context_data_get(ctx);
   Ecore_X_Event_Client_Message *ev = event;

   if (ctxd->input_window != ev->win)
     return 1;

   if (type == ECORE_X_EVENT_CLIENT_MESSAGE)
     {
	if (ev->message_type == ecore_x_atom_get(HILDON_IM_INSERT_UTF8_NAME) &&
	    ev->format == HILDON_IM_INSERT_UTF8_FORMAT)
	  {
	     Hildon_IM_Insert_Utf8_Message *msg = (Hildon_IM_Insert_Utf8_Message *) &ev->data;
	     _insert_utf8(ctx, msg->msg_flag, msg->utf8_str);
	  }
	else if (ev->message_type == ecore_x_atom_get(HILDON_IM_COM_NAME) &&
		 ev->format == HILDON_IM_COM_FORMAT)
	  {
	     Hildon_IM_Com_Message *msg = (Hildon_IM_Com_Message *) &ev->data;
	     switch (msg->type)
	       {
		case HILDON_IM_CONTEXT_WIDGET_CHANGED:
		   break;
		case HILDON_IM_CONTEXT_HANDLE_ENTER:
		case HILDON_IM_CONTEXT_ENTER_ON_FOCUS:
		   ecore_idler_add((int (*)(void *)) _send_enter, ctxd->display);
		   break;
		case HILDON_IM_CONTEXT_CONFIRM_SENTENCE_START:
		   _check_sentence_start(ctx);
		   break;
		case HILDON_IM_CONTEXT_HANDLE_TAB:
		   _send_fake_key(ctxd->display, XK_Tab, 1);
		   _send_fake_key(ctxd->display, XK_Tab, 0);
		   break;
		case HILDON_IM_CONTEXT_HANDLE_BACKSPACE:
		   if (ctxd->commit_mode != HILDON_IM_COMMIT_REDIRECT)
		     {
			_send_fake_key(ctxd->display, XK_BackSpace, 1);
			_send_fake_key(ctxd->display, XK_BackSpace, 0);
		     }
		   break;
		case HILDON_IM_CONTEXT_HANDLE_SPACE:
		   _insert_utf8(ctx, HILDON_IM_MSG_CONTINUE, " ");
		   _commit_preedit_data(ctxd);
		   break;
		case HILDON_IM_CONTEXT_BUFFERED_MODE:
		   if (!ctxd->pre_edit_buffer)
		     ctxd->pre_edit_buffer = strdup("");
		   ctxd->commit_mode = HILDON_IM_COMMIT_BUFFERED;
		   break;
		case HILDON_IM_CONTEXT_DIRECT_MODE:
		   if (ctxd->pre_edit_buffer)
		     {
			_commit_preedit_data(ctxd);
			free(ctxd->pre_edit_buffer);
			ctxd->pre_edit_buffer = NULL;
		     }
		   ctxd->commit_mode = HILDON_IM_COMMIT_DIRECT;
		   break;
		case HILDON_IM_CONTEXT_REDIRECT_MODE:
		   if (ctxd->pre_edit_buffer)
		     {
			_commit_preedit_data(ctxd);
			free(ctxd->pre_edit_buffer);
			ctxd->pre_edit_buffer = NULL;
		     }
		   ctxd->commit_mode = HILDON_IM_COMMIT_REDIRECT;
		   _check_commit_mode(ctxd);
		   break;
		case HILDON_IM_CONTEXT_SURROUNDING_MODE:
		   if (ctxd->pre_edit_buffer)
		     {
			_commit_preedit_data(ctxd);
			free(ctxd->pre_edit_buffer);
			ctxd->pre_edit_buffer = NULL;
		     }
		  ctxd->commit_mode = HILDON_IM_COMMIT_SURROUNDING;
		  break;
		case HILDON_IM_CONTEXT_REQUEST_SURROUNDING:
		   _check_commit_mode(ctxd);
		   _send_surrounding(ctx);
		   break;
		case HILDON_IM_CONTEXT_FLUSH_PREEDIT:
		   _commit_preedit_data(ctxd);
		   break;
		case HILDON_IM_CONTEXT_CLIPBOARD_COPY:
		   break;
		case HILDON_IM_CONTEXT_CLIPBOARD_CUT:
		   break;
		case HILDON_IM_CONTEXT_CLIPBOARD_PASTE:
		   break;
		case HILDON_IM_CONTEXT_CLIPBOARD_SELECTION_QUERY:
		   break;
		case HILDON_IM_CONTEXT_CLEAR_STICKY:
		   break;
		case HILDON_IM_CONTEXT_OPTION_CHANGED:
		   break;
		default:
		   fprintf(stderr, "** hildon_imf: Invalid communication message %d from keyboard\n", msg->type);
		   break;
	       }
	  }
	else if (ev->message_type == ecore_x_atom_get(HILDON_IM_SURROUNDING_CONTENT_NAME) &&
		 ev->format == HILDON_IM_SURROUNDING_CONTENT_FORMAT)
	  {
	     Hildon_IM_Surrounding_Content_Message *msg = (Hildon_IM_Surrounding_Content_Message *) &ev->data;
	     char *new_surrounding;

	     if (ctxd->surrounding)
	       {
		  if (msg->msg_flag == HILDON_IM_MSG_START)
		    {
		       free(ctxd->surrounding);
		       ctxd->surrounding = strdup("");
		    }
		  else if (msg->msg_flag == HILDON_IM_MSG_END)
		    {
		       _commit_surrounding(ctx);
		       return 1;
		    }
	       }

	     if (ctxd->surrounding && !msg->surrounding)
	       new_surrounding = strdup(ctxd->surrounding);
	     else if (!ctxd->surrounding && msg->surrounding)
	       new_surrounding = strdup(msg->surrounding);
	     else
	       {
		  int len;
		  len = strlen(ctxd->surrounding) + strlen(msg->surrounding) + 1;
		  new_surrounding = malloc(len * sizeof(char));
		  snprintf(new_surrounding, len, "%s%s",
			   ctxd->surrounding, msg->surrounding);
	       }

	     if (ctxd->surrounding)
	       free(ctxd->surrounding);

	     ctxd->surrounding = new_surrounding;
	  }
     }

  return 1;
}

static Hildon_IMF_Context *
_hildon_imf_context_new(void)
{
   Hildon_IMF_Context *ctxd;
   Ecore_X_Display *display;

   display = ecore_x_display_get();
   if (!display) return NULL;

   ctxd = malloc(sizeof(Hildon_IMF_Context));
   ctxd->display = display;
   ctxd->window = 0;
   ctxd->input_window = ecore_x_window_input_new(None, 0, 0, 8, 8);
   ctxd->client_message_event_handler = NULL;
   ctxd->input_mode = ECORE_IMF_INPUT_MODE_FULL;
   ctxd->trigger = HILDON_IM_TRIGGER_FINGER;
   ctxd->commit_mode = HILDON_IM_COMMIT_REDIRECT;
   _check_commit_mode(ctxd);
   ctxd->launch_delay = HILDON_IM_DEFAULT_LAUNCH_DELAY;
   ctxd->launch_delay_timer = 0;
   ctxd->pre_edit_buffer = NULL;
   ctxd->last_internal_change = 0;
   ctxd->has_focus = 0;
   ctxd->auto_correction = 0;
   ctxd->prev_cursor_pos = 0;
   ctxd->surrounding = strdup("");

   return ctxd;
}

static void
_hildon_imf_context_del(Ecore_IMF_Context *ctx)
{
   Hildon_IMF_Context *ctxd = ecore_imf_context_data_get(ctx);

   _hildon_imf_context_client_window_set(ctx, NULL);
   if (ctxd->surrounding) free(ctxd->surrounding);
   if (ctxd->launch_delay_timer) ecore_timer_del(ctxd->launch_delay_timer);
   ecore_x_window_del(ctxd->input_window);
}

static void
_hildon_imf_context_client_window_set(Ecore_IMF_Context *ctx, void *window)
{
   Hildon_IMF_Context *ctxd = ecore_imf_context_data_get(ctx);

   if (ctxd->window == (Ecore_X_Window) window) return;

   if (ctxd->window)
     _send_command(ctxd, HILDON_IM_HIDE);

   if (ctxd->client_message_event_handler)
     {
	ecore_event_handler_del(ctxd->client_message_event_handler);
	ctxd->client_message_event_handler = NULL;
     }

   ctxd->window = (Ecore_X_Window) window;

   if (window)
     ctxd->client_message_event_handler =
	ecore_event_handler_add(ECORE_X_EVENT_CLIENT_MESSAGE,
				_client_message_cb, ctx);

   _commit_preedit_data(ctxd);
}

static int
_hildon_imf_context_show_cb(Ecore_IMF_Context *ctx)
{
   Hildon_IMF_Context *ctxd = ecore_imf_context_data_get(ctx);

   ctxd->launch_delay_timer = 0;

   if (ctxd->has_focus)
     _check_sentence_start(ctx);

   _send_command(ctxd, HILDON_IM_SETNSHOW);

   return 0;
}

static void
_hildon_imf_context_show(Ecore_IMF_Context *ctx)
{
   Hildon_IMF_Context *ctxd = ecore_imf_context_data_get(ctx);
   ctxd->launch_delay_timer = ecore_timer_add(ctxd->launch_delay,
					      (int (*)(void *)) _hildon_imf_context_show_cb,
					      ctx);
}

static void
_hildon_imf_context_hide(Ecore_IMF_Context *ctx)
{
   Hildon_IMF_Context *ctxd = ecore_imf_context_data_get(ctx);
   _send_command(ctxd, HILDON_IM_HIDE);
}

static void
_hildon_imf_context_focus_in(Ecore_IMF_Context *ctx)
{
   Hildon_IMF_Context *ctxd = ecore_imf_context_data_get(ctx);

   ctxd->has_focus = 1;
   _send_command(ctxd, HILDON_IM_SETCLIENT);
}

static void
_hildon_imf_context_focus_out(Ecore_IMF_Context *ctx)
{
   Hildon_IMF_Context *ctxd = ecore_imf_context_data_get(ctx);
   ctxd->has_focus = 0;
}

static void
_hildon_imf_context_reset(Ecore_IMF_Context *ctx)
{
   Hildon_IMF_Context *ctxd = ecore_imf_context_data_get(ctx);
   _commit_preedit_data(ctxd);
   _send_command(ctxd, HILDON_IM_CLEAR);
}

static void
_hildon_imf_context_cursor_position_set(Ecore_IMF_Context *ctx, int cursor_pos)
{
   Hildon_IMF_Context *ctxd = ecore_imf_context_data_get(ctx);

   if (!ctxd->has_focus)
     return;

   if (ctxd->last_internal_change)
     {
	/* Our own change */
	_check_sentence_start(ctx);
	ctxd->last_internal_change = 0;
     }
   else
     {
	if (ctxd->prev_cursor_pos != cursor_pos)
	  {
	     /* Moved, clear IM. */
	     _check_sentence_start(ctx);
	     _commit_preedit_data(ctxd);
	     _send_command(ctxd, HILDON_IM_CLEAR);
	  }
     }

   ctxd->prev_cursor_pos = cursor_pos;
}

static void
_hildon_imf_context_input_mode_set(Ecore_IMF_Context *ctx, Ecore_IMF_Input_Mode input_mode)
{
   Hildon_IMF_Context *ctxd = ecore_imf_context_data_get(ctx);
   ctxd->input_mode = input_mode;
}

static int
_hildon_imf_context_filter_event(Ecore_IMF_Context *ctx, Ecore_IMF_Event_Type type, Ecore_IMF_Event *event)
{
   if (type == ECORE_IMF_EVENT_MOUSE_UP)
      _hildon_imf_context_show(ctx);

   return 0;
}

static const
Ecore_IMF_Context_Info hildon_imf_info =
{
   "hildon-input-method", /* ID */
   "Hildon Input Method", /* Description */
   "*",                   /* Default locales */
   NULL,                  /* Canvas type */
   0                      /* Canvas required */
};

static Ecore_IMF_Context_Class hildon_imf_class =
{
   NULL,                                    /* add */
   _hildon_imf_context_del,                 /* del */
   _hildon_imf_context_client_window_set,   /* client_window_set */
   NULL,                                    /* client_canvas_set */
   _hildon_imf_context_show,                /* show */
   _hildon_imf_context_hide,                /* hide */
   NULL,                                    /* get_preedit_string */
   _hildon_imf_context_focus_in,            /* focus_in */
   _hildon_imf_context_focus_out,           /* focus_out */
   _hildon_imf_context_reset,               /* reset */
   _hildon_imf_context_cursor_position_set, /* cursor_position_set */
   NULL,                                    /* use_preedit_set */
   _hildon_imf_context_input_mode_set,      /* input_mode_set */
   _hildon_imf_context_filter_event         /* filter_event */
};

int imf_module_init(const Ecore_IMF_Context_Info **info)
{
   *info = &hildon_imf_info;
   return 1;
}

void imf_module_exit(void)
{
}

Ecore_IMF_Context *imf_module_create(void)
{
   Ecore_IMF_Context *ctx;
   Hildon_IMF_Context *ctxd;

   ctxd = _hildon_imf_context_new();
   if (!ctxd) return NULL;
   ctx = ecore_imf_context_new(&hildon_imf_class);
   ecore_imf_context_data_set(ctx, ctxd);
   return ctx;
}
