/*
 * Copyright (C) 2009-2010 Marco Barisione <marco@barisione.org>
 * Copyright (C) 2010 Collabora Ltd.
 *   @author Marco Barisione <marco.barisione@collabora.co.uk>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include "config.h"

#include <libosso-abook/osso-abook.h>

#include "wizard.h"
#include "main.h"
#include "merge.h"


static gchar uninstall_text[] = "You can now uninstall this program.";

enum
{
    WELCOME_PAGE,
    BACKUP_PAGE,
    FINISHED_PAGE,
    NUMBER_OF_PAGES
};

typedef struct
{
    gboolean aggregator_ready;

    GList *candidates;

    GtkWidget *notebook;

    GtkWidget *welcome_label;
    GtkWidget *finished_label;

    GtkWidget *progress_note;
    GtkWidget *progress_bar;
} FBWizardData;

static GQuark
fb_wizard_data_quark ()
{
    static GQuark q = 0;
    if (G_UNLIKELY (q == 0))
        q = g_quark_from_static_string ("fb-wizard-data-quark");
    return q;
}

static FBWizardData *
fb_wizard_get_data (GtkWidget *wizard)
{
    FBWizardData *data;

    data = g_object_get_qdata (G_OBJECT (wizard), fb_wizard_data_quark ());
    g_return_val_if_fail (data != NULL, data);

    return data;
}

static void
fb_wizard_data_free (FBWizardData *data)
{
    if (!data)
        return;

    merge_candidates_free (data->candidates);

    g_free (data);
}

static void
page_size_allocate_cb (GtkWidget     *page,
                       GtkAllocation *allocation,
                       GtkWidget     *label)
{
    int label_width;

    label_width = allocation->width - HILDON_MARGIN_DOUBLE;
    gtk_widget_set_size_request (label, label_width, -1);
}

static void
page_fix_label_width (GtkWidget *page,
                      GtkWidget *label)
{
    /* Thanks Maemo for being invaluable as usual... */
    g_signal_connect (page, "size-allocate",
            G_CALLBACK (page_size_allocate_cb), label);
}

static GtkWidget *
create_welcome_page (GtkWidget *wizard)
{
    FBWizardData *data;
    GtkWidget *page;
    GtkWidget *label;

    data = fb_wizard_get_data (wizard);

    label = gtk_label_new ("Loading your contacts...");
    gtk_misc_set_alignment (GTK_MISC (label), 0, 0.5);
    gtk_label_set_line_wrap (GTK_LABEL (label), TRUE);
    gtk_label_set_line_wrap_mode (GTK_LABEL (label), PANGO_WRAP_WORD_CHAR);

    page = gtk_vbox_new (FALSE, HILDON_MARGIN_DOUBLE);
    gtk_box_pack_start (GTK_BOX (page), label, FALSE, FALSE, 0);

    page_fix_label_width (page, label);

    gtk_widget_show_all (page);

    data->welcome_label = label;

    return page;
}

static void
backup_clicked_cb (GtkWidget *button,
                   GtkWidget *wizard)
{
    osso_application_top (get_osso_context (), "com.nokia.backup", NULL);
    gtk_widget_destroy (wizard);
}

static GtkWidget *
create_backup_page (GtkWidget *wizard)
{
    GtkWidget *label;
    GtkWidget *button;
    GtkWidget *button_label;
    GdkPixbuf *pixbuf;
    GtkWidget *icon;
    GtkWidget *hbox;
    GtkWidget *page;

    /* Label */
    label = gtk_label_new (
            "If you didn't take one already, it's recommended that you "
            "take a backup of your contacts and then launch this program "
            "again.");
    gtk_label_set_line_wrap (GTK_LABEL (label), TRUE);
    gtk_label_set_line_wrap_mode (GTK_LABEL (label), PANGO_WRAP_WORD_CHAR);

    /* Button */
    pixbuf = gtk_icon_theme_load_icon (gtk_icon_theme_get_default (),
            "general_backup", HILDON_ICON_PIXEL_SIZE_FINGER, 0, NULL);
    icon = gtk_image_new_from_pixbuf (pixbuf);
    g_object_unref (pixbuf);
    gtk_misc_set_alignment (GTK_MISC (icon), 1.0, 0.5);

    button_label = g_object_new (GTK_TYPE_LABEL,
            "label",  "Take a backup now",
            "xalign", 0.0,
            "yalign", 0.5,
            NULL);

    hbox = gtk_hbox_new (FALSE, HILDON_MARGIN_DOUBLE);
    gtk_box_pack_start (GTK_BOX (hbox), icon, TRUE, TRUE, 0);
    gtk_box_pack_start (GTK_BOX (hbox), button_label, TRUE, TRUE, 0);

    button = gtk_button_new ();
    GTK_WIDGET_UNSET_FLAGS (button, GTK_CAN_FOCUS | GTK_CAN_DEFAULT);
    gtk_button_set_focus_on_click (GTK_BUTTON (button), FALSE);
    hildon_gtk_widget_set_theme_size (button, HILDON_SIZE_FINGER_HEIGHT);
    gtk_container_add (GTK_CONTAINER (button), hbox);
    g_signal_connect (button, "clicked", G_CALLBACK (backup_clicked_cb),
            wizard);

    /* Page */
    page = gtk_vbox_new (FALSE, HILDON_MARGIN_DOUBLE);
    gtk_box_pack_start (GTK_BOX (page), label, FALSE, FALSE, 0);
    gtk_box_pack_start (GTK_BOX (page), button, FALSE, FALSE, 0);

    page_fix_label_width (page, label);

    gtk_widget_show_all (page);

    return page;
}

static GtkWidget *
create_finished_page (GtkWidget *wizard)
{
    FBWizardData *data;
    GtkWidget *page;
    GtkWidget *label;

    data = fb_wizard_get_data (wizard);

    label = gtk_label_new ("");
    gtk_misc_set_alignment (GTK_MISC (label), 0, 0.5);
    gtk_label_set_line_wrap (GTK_LABEL (label), TRUE);
    gtk_label_set_line_wrap_mode (GTK_LABEL (label), PANGO_WRAP_WORD_CHAR);

    page = gtk_vbox_new (FALSE, HILDON_MARGIN_DOUBLE);
    gtk_box_pack_start (GTK_BOX (page), label, FALSE, FALSE, 0);

    page_fix_label_width (page, label);

    gtk_widget_show_all (page);

    data->finished_label = label;

    return page;
}

static void
merge_updated_cb (gint     merged_count,
                  gint     failed_count,
                  gint     total_count,
                  gpointer user_data)
{
    GtkWidget *wizard;
    FBWizardData *data;

    wizard = user_data;
    data = fb_wizard_get_data (wizard);

    gtk_progress_bar_set_fraction (GTK_PROGRESS_BAR (data->progress_bar),
        (merged_count + failed_count) / (gdouble) total_count);
}

static void
merge_finished_cb (gint     merged_count,
                   gint     failed_count,
                   gint     total_count,
                   gpointer user_data)
{
    GtkWidget *wizard;
    FBWizardData *data;
    gchar *text;

    wizard = user_data;
    data = fb_wizard_get_data (wizard);

    gtk_widget_destroy (data->progress_note);

    if (failed_count == 0)
        text = g_strdup_printf (
                "All of your Facebook contacts have been merged back.\n\n%s",
                uninstall_text);
    else
        text = g_strdup_printf (
                "Some of your Facebook contacts have been merged back, "
                "but %d failed to be merged.", failed_count);

    /* The address book doesn't seem to update correctly in this case,
     * so we kill it (and I'm too lazy to kill it in a better way).
     * Sigh :( */
    if (merged_count > 0)
        system ("pkill osso-address");

    gtk_label_set_text (GTK_LABEL (data->finished_label), text);
    g_free (text);
}

static void
start_merge (GtkWidget *wizard)
{
    FBWizardData *data;

    data = fb_wizard_get_data (wizard);

    data->progress_bar = gtk_progress_bar_new ();
    data->progress_note = hildon_note_new_cancel_with_progress_bar (
            GTK_WINDOW (wizard), "Migrating contacts",
            GTK_PROGRESS_BAR (data->progress_bar));
    gtk_widget_hide (
            gtk_dialog_get_action_area (GTK_DIALOG (data->progress_note)));
    gtk_widget_show (data->progress_note);

    merge_candidates_do_merge (data->candidates, merge_updated_cb,
            merge_finished_cb, wizard);
}

static void
nothing_to_do (GtkWidget *wizard)
{
    FBWizardData *data;
    gchar *text;

    data = fb_wizard_get_data (wizard);

    if (data->aggregator_ready)
        text = g_strdup_printf (
                "Oh joy!\n"
                "All of your Facebook contacts are correctly merged "
                "and there is nothing to do for me.\n\n%s",
                uninstall_text);
    else
        text = g_strdup ("Impossible to retrieve the addressbook :(");

    gtk_label_set_text (GTK_LABEL (data->finished_label), text);

    /* Not sure why, but in this case the button is wrongly sensitive */
    gtk_dialog_set_response_sensitive (GTK_DIALOG (wizard),
            HILDON_WIZARD_DIALOG_NEXT, FALSE);

    g_free (text);
}

static void
switch_page_cb (GtkNotebook     *notebook,
                GtkNotebookPage *page_unused,
                int              page_num,
                GtkWidget       *wizard)
{
    FBWizardData *data;
    const gchar *title;

    data = fb_wizard_get_data (wizard);

    gtk_dialog_set_response_sensitive (GTK_DIALOG (wizard),
            HILDON_WIZARD_DIALOG_FINISH, FALSE);
    gtk_dialog_set_response_sensitive (GTK_DIALOG (wizard),
            HILDON_WIZARD_DIALOG_CANCEL, TRUE);

    switch (page_num) {
        case WELCOME_PAGE:
            if (data->aggregator_ready) {
                title = "Welcome";
                gtk_dialog_set_response_sensitive (GTK_DIALOG (wizard),
                        HILDON_WIZARD_DIALOG_NEXT, TRUE);
            }
            else {
                title = "Facebook contacts merger";
                gtk_dialog_set_response_sensitive (GTK_DIALOG (wizard),
                        HILDON_WIZARD_DIALOG_NEXT, FALSE);
            }
            break;

        case BACKUP_PAGE:
            title = "Backup";
            break;

        case FINISHED_PAGE:
            title = "Completed";

            gtk_dialog_set_response_sensitive (GTK_DIALOG (wizard),
                    HILDON_WIZARD_DIALOG_PREVIOUS, FALSE);
            gtk_dialog_set_response_sensitive (GTK_DIALOG (wizard),
                    HILDON_WIZARD_DIALOG_CANCEL, FALSE);
            gtk_dialog_set_response_sensitive (GTK_DIALOG (wizard),
                    HILDON_WIZARD_DIALOG_FINISH, TRUE);

            if (data->candidates)
                start_merge (wizard);
            else
                nothing_to_do (wizard);

            break;

        default:
            g_warning ("Invalid page number: %d", page_num);

            return;
    }

    gtk_window_set_title (GTK_WINDOW (wizard), title);
}

static void
aggregator_ready_cb (OssoABookWaitable *aggregator,
                     const GError      *error_in,
                     gpointer           user_data)
{
    GtkWidget *wizard;
    FBWizardData *data;
    gchar *text;

    wizard = user_data;
    data = fb_wizard_get_data (wizard);

    hildon_gtk_window_set_progress_indicator (GTK_WINDOW (wizard), 0);

    if (error_in) {
        g_critical ("Error while waiting for the aggregator to be ready: "
                "%s\n", error_in->message);
        gtk_notebook_set_current_page (GTK_NOTEBOOK (data->notebook),
                FINISHED_PAGE);
        return;
    }

    data->aggregator_ready = TRUE;

    data->candidates = merge_candidates_get ();

    text = g_strdup_printf (
            "Due to an update of the Facebook servers, %d of your contacts "
            "were unmerged.\n"
            "Press “Next” if you want to fix your contacts automatically.",
            g_list_length (data->candidates));
    gtk_label_set_text (GTK_LABEL (data->welcome_label), text);
    g_free (text);

    if (data->candidates)
        switch_page_cb (GTK_NOTEBOOK (data->notebook), NULL,
                WELCOME_PAGE, wizard);
    else
        gtk_notebook_set_current_page (GTK_NOTEBOOK (data->notebook),
                FINISHED_PAGE);
}

static void
response_cb (GtkDialog *dialog,
             gint       response_id,
             GtkWidget *wizard)
{
    if (response_id == HILDON_WIZARD_DIALOG_FINISH)
        gtk_widget_destroy (wizard);
}

static void
delete_event_cb (GtkWidget *window,
                 GdkEvent  *event,
                 GtkWidget *wizard)
{
    gtk_widget_destroy (wizard);
}

GtkWidget *
fb_wizard_new (void)
{
    FBWizardData *data;
    GtkWidget *wizard;
    GtkWidget *page;
    OssoABookRoster *aggregator;
    GError *error = NULL;

    data = g_new0 (FBWizardData, 1);

    data->notebook = gtk_notebook_new ();

    wizard = hildon_wizard_dialog_new (NULL, "",
            GTK_NOTEBOOK (data->notebook));
    g_object_set_qdata_full (G_OBJECT (wizard), fb_wizard_data_quark (), data,
            (GDestroyNotify) fb_wizard_data_free);
    g_object_set (G_OBJECT (wizard), "autotitle", FALSE, NULL);
    hildon_gtk_window_set_progress_indicator (GTK_WINDOW (wizard), 1);
    gtk_container_set_border_width (GTK_CONTAINER (wizard),
            HILDON_MARGIN_DOUBLE);
    g_signal_connect (wizard, "response", G_CALLBACK (response_cb), wizard);
    g_signal_connect (wizard, "delete-event", G_CALLBACK (delete_event_cb),
            wizard);

    g_signal_connect (G_OBJECT (data->notebook), "switch-page",
            G_CALLBACK (switch_page_cb), wizard);

    /* Create the pages */
    page = create_welcome_page (wizard);
    gtk_notebook_append_page (GTK_NOTEBOOK (data->notebook), page, NULL);

    page = create_backup_page (wizard);
    gtk_notebook_append_page (GTK_NOTEBOOK (data->notebook), page, NULL);

    page = create_finished_page (wizard);
    gtk_notebook_append_page (GTK_NOTEBOOK (data->notebook), page, NULL);

    aggregator = osso_abook_aggregator_get_default (&error);
    if (aggregator) {
        osso_abook_waitable_call_when_ready (OSSO_ABOOK_WAITABLE (aggregator),
            aggregator_ready_cb, wizard, NULL);
    }
    else {
        aggregator_ready_cb (OSSO_ABOOK_WAITABLE (aggregator), error, wizard);
        g_clear_error (&error);
    }


    return wizard;
}
