/*
 *  desktop clock home widget for the maemo desktop.
 *  Copyright (C) 2010 Nicolai Hess
 *  
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *  
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#include <gtk/gtk.h>
#include <hildon/hildon.h>
#include <libintl.h>
#include <locale.h>
#include <libhildondesktop/libhildondesktop.h>
#include <hildon/hildon-file-chooser-dialog.h>
#include <librsvg/rsvg.h>
#include <librsvg/rsvg-cairo.h>
#include <math.h>
#include <cityinfo.h>

#include "desktop-clock-widget.h"

#define HD_APP_MGR_DBUS_NAME "com.nokia.HildonDesktop.AppMgr"
#define HD_APP_MGR_DBUS_PATH "/com/nokia/HildonDesktop/AppMgr"
#define HD_APP_MGR_DBUS_IFACE "com.nokia.HildonDesktop.AppMgr"
#define HD_APP_MGR_LAUNCH_APPLICATION "LaunchApplication"
#define WORLD_CLOCK_APP_NAME "worldclock"

#define DESKTOP_CLOCK_WIDGET_HOME_PLUGIN_SETTINGS_FILE "config"
#define DESKTOP_CLOCK_WIDGET_HOME_PLUGIN_SETTINGS_FOLDER ".desktop_clock_widget"
#define DESKTOP_CLOCK_WIDGET_CONFIG_KEY_ZONE_ID "zone_id"
#define DESKTOP_CLOCK_WIDGET_CONFIG_KEY_WIDTH "width"
#define DESKTOP_CLOCK_WIDGET_CONFIG_KEY_HEIGHT "height"
#define DESKTOP_CLOCK_WIDGET_CONFIG_KEY_SHOW_SECONDS "show_seconds"
#define DESKTOP_CLOCK_WIDGET_CONFIG_KEY_USE_STOCK_THEME "use_stock_theme"
#define DESKTOP_CLOCK_WIDGET_CONFIG_KEY_SHOW_CITYNAME "show_cityname"
#define DESKTOP_CLOCK_WIDGET_CONFIG_KEY_TWENTY_FOUR_HOURS "twenty_four_hours"
#define DESKTOP_CLOCK_WIDGET_CONFIG_KEY_CUSTOM_THEME_NAME "custom_theme_name"

#define CLOCK_DROP_SHADOW_SVG "clock-drop-shadow.svg"
#define CLOCK_FACE_SHADOW_SVG "clock-face-shadow.svg"
#define CLOCK_FACE_SVG "clock-face.svg"
#define CLOCK_FRAME_SVG "clock-frame.svg"
#define CLOCK_GLASS_SVG "clock-glass.svg"
#define CLOCK_MARKS_SVG "clock-marks.svg"
#define CLOCK_HOUR_HAND_SVG "clock-hour-hand.svg"
#define CLOCK_HOUR_HAND_SHADOW_SVG "clock-hour-hand-shadow.svg"
#define CLOCK_MINUTE_HAND_SVG "clock-minute-hand.svg"
#define CLOCK_MINUTE_HAND_SHADOW_SVG "clock-minute-hand-shadow.svg"
#define CLOCK_SECOND_HAND_SVG "clock-second-hand.svg"
#define CLOCK_SECOND_HAND_SHADOW_SVG "clock-second-hand-shadow.svg"

#define DESKTOP_CLOCK_WIDGET_BUILTIN_THEME "/usr/share/pixmaps/desktop-clock-widget/"

HD_DEFINE_PLUGIN_MODULE (DesktopClockWidgetHomePlugin, desktop_clock_widget_home_plugin, HD_TYPE_HOME_PLUGIN_ITEM)

#define DESKTOP_CLOCK_WIDGET_HOME_PLUGIN_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE(obj, \
DESKTOP_CLOCK_WIDGET_TYPE_HOME_PLUGIN, DesktopClockWidgetHomePluginPrivate))

typedef struct _DesktopClockSVGTheme
{
  RsvgHandle* drop_shadow;
  RsvgHandle* face_shadow;
  RsvgHandle* face;
  RsvgHandle* frame;
  RsvgHandle* glass;
  RsvgHandle* hour_hand_shadow;
  RsvgHandle* hour_hand;
  RsvgHandle* marks;
  RsvgHandle* minute_hand_shadow;
  RsvgHandle* minute_hand;
  RsvgHandle* second_hand_shadow;
  RsvgHandle* second_hand;
  gchar* path;
} DesktopClockSVGTheme;

struct _DesktopClockWidgetHomePluginPrivate
{
  gchar* iD;
  gfloat clock_width;
  gfloat clock_height;
  gfloat image_width;
  gfloat image_height;
  gboolean show_seconds;
  gboolean twenty_four_hours;
  gboolean show_cityname;
  gint city_info_id;
  gboolean use_stock_theme;
  gchar* custom_theme_name;
  gchar* theme_path;
  cairo_surface_t* background;
  cairo_surface_t* foreground;
  RsvgHandle* hour_svg_handle;
  RsvgHandle* hour_shadow_svg_handle;
  RsvgHandle* minute_svg_handle;
  RsvgHandle* minute_shadow_svg_handle;
  RsvgHandle* second_svg_handle;
  RsvgHandle* second_shadow_svg_handle;
  Cityinfo* city_info;
  guint time_out_handler;
};

// to prevent unloading the module, as
// reloading conflicts with the statically registered
// rsvg types
const gchar *
g_module_check_init (GModule *module)
{
  g_module_make_resident(module);
  return NULL;
} 


static gboolean
desktop_clock_widget_load_svg_file(const gchar* path,
				   const gchar* file_name, 
				   RsvgHandle** pHandle)
{
  GError* error = NULL;
  gchar* full_path = g_build_filename(path, file_name, NULL);
  RsvgHandle* handle = rsvg_handle_new_from_file(full_path, &error);
  g_free(full_path);
  if(error)
  {

    g_error_free(error);
    error = NULL;
    pHandle = NULL;
    return FALSE;
  }

  *pHandle = handle;

  return handle != NULL;
}

static void
desktop_clock_widget_free_svg_theme(DesktopClockSVGTheme* theme)
{
  if(theme->drop_shadow)
  {
    g_object_unref(theme->drop_shadow);
  }
  if(theme->face_shadow)
  {
    g_object_unref(theme->face_shadow);
  }
  if(theme->face)
  {
    g_object_unref(theme->face);
  }
  if(theme->frame)
  {
    g_object_unref(theme->frame);
  }
  if(theme->glass)
  {
    g_object_unref(theme->glass);
  }
  if(theme->hour_hand)
  {
    g_object_unref(theme->hour_hand);
  }
  if(theme->hour_hand_shadow)
  {
    g_object_unref(theme->hour_hand_shadow);
  }
  if(theme->marks)
  {
    g_object_unref(theme->marks);
  }
  if(theme->minute_hand)
  {
    g_object_unref(theme->minute_hand);
  }
  if(theme->minute_hand_shadow)
  {
    g_object_unref(theme->minute_hand_shadow);
  }
  if(theme->second_hand)
  {
    g_object_unref(theme->second_hand);
  }
  if(theme->second_hand_shadow)
  {
    g_object_unref(theme->second_hand_shadow);
  }
  if(theme->path)
    g_free(theme->path);
  g_free(theme);
}

static gboolean
desktop_clock_widget_read_theme(DesktopClockSVGTheme* theme)
{
  gboolean one_found = FALSE;
  one_found |= desktop_clock_widget_load_svg_file(theme->path, 
                                                  CLOCK_DROP_SHADOW_SVG,
     					          &theme->drop_shadow);
     one_found |= desktop_clock_widget_load_svg_file(theme->path, CLOCK_FACE_SHADOW_SVG,
     					 &theme->face_shadow);
     one_found |= desktop_clock_widget_load_svg_file(theme->path, CLOCK_FACE_SVG,
     					 &theme->face);
     one_found |= desktop_clock_widget_load_svg_file(theme->path, CLOCK_FRAME_SVG,
     					 &theme->frame);
     one_found |= desktop_clock_widget_load_svg_file(theme->path, CLOCK_GLASS_SVG,
     					 &theme->glass);
     one_found |= desktop_clock_widget_load_svg_file(theme->path, CLOCK_HOUR_HAND_SHADOW_SVG,
     					 &theme->hour_hand_shadow);
     one_found |= desktop_clock_widget_load_svg_file(theme->path, CLOCK_HOUR_HAND_SVG,
     					 &theme->hour_hand);
     one_found |= desktop_clock_widget_load_svg_file(theme->path, CLOCK_MARKS_SVG,
     					 &theme->marks);
     one_found |= desktop_clock_widget_load_svg_file(theme->path, CLOCK_MINUTE_HAND_SHADOW_SVG,
     					 &theme->minute_hand_shadow);
     one_found |= desktop_clock_widget_load_svg_file(theme->path, CLOCK_MINUTE_HAND_SVG,
     					 &theme->minute_hand);
     one_found |= desktop_clock_widget_load_svg_file(theme->path, CLOCK_SECOND_HAND_SHADOW_SVG,
     					 &theme->second_hand_shadow);
     one_found |= desktop_clock_widget_load_svg_file(theme->path, CLOCK_SECOND_HAND_SVG,
					 &theme->second_hand);
    return one_found; 
}

static void
desktop_clock_widget_render_background(cairo_t* main_cr,
				       DesktopClockSVGTheme* theme,
				       DesktopClockWidgetHomePlugin* desktop_plugin)
{
  if(desktop_plugin->priv->background!=NULL)
    cairo_surface_destroy(desktop_plugin->priv->background);

  desktop_plugin->priv->background = cairo_surface_create_similar(cairo_get_target(main_cr), 
								  CAIRO_CONTENT_COLOR_ALPHA, 
								  desktop_plugin->priv->clock_width, 
								  desktop_plugin->priv->clock_height); 

  cairo_t* cr = cairo_create(desktop_plugin->priv->background);
  cairo_scale(cr,  
	      desktop_plugin->priv->clock_width/desktop_plugin->priv->image_width,  
	      desktop_plugin->priv->clock_height/desktop_plugin->priv->image_height);








  cairo_set_operator(cr, CAIRO_OPERATOR_OVER);
  rsvg_handle_render_cairo(theme->drop_shadow, cr);
  rsvg_handle_render_cairo(theme->face, cr);
  rsvg_handle_render_cairo(theme->marks, cr);
  cairo_destroy(cr);    
  cairo_destroy(cr);
}

static void
desktop_clock_widget_render_foreground(cairo_t* main_cr,
				       DesktopClockSVGTheme* theme,
				       DesktopClockWidgetHomePlugin* desktop_plugin)
{
  if(desktop_plugin->priv->foreground!=NULL)
    cairo_surface_destroy(desktop_plugin->priv->foreground);

  desktop_plugin->priv->foreground = cairo_surface_create_similar(cairo_get_target(main_cr),
								  CAIRO_CONTENT_COLOR_ALPHA,
								  desktop_plugin->priv->clock_width, 
								  desktop_plugin->priv->clock_height);
  cairo_t* cr = cairo_create(desktop_plugin->priv->foreground);
  cairo_scale(cr,  
   	      desktop_plugin->priv->clock_width/desktop_plugin->priv->image_width, 
   	      desktop_plugin->priv->clock_height/desktop_plugin->priv->image_height); 
  
  cairo_set_operator(cr, CAIRO_OPERATOR_OVER);
  rsvg_handle_render_cairo(theme->face_shadow, cr);
  rsvg_handle_render_cairo(theme->glass, cr);
  rsvg_handle_render_cairo(theme->frame, cr);
  cairo_destroy(cr);
}

static void
desktop_clock_widget_register_time_out_handler(DesktopClockWidgetHomePlugin* desktop_plugin);

static gboolean
desktop_clock_widget_update_time(gpointer data)
{
  gboolean on;
  g_object_get(GTK_WIDGET(data), "is-on-current-desktop", &on, NULL);
  if(!on)
  {
    return FALSE;
  }
  gdk_threads_enter();
  gtk_widget_queue_draw(GTK_WIDGET(data));
  gdk_threads_leave();
  if(DESKTOP_CLOCK_WIDGET_HOME_PLUGIN(data)->priv->show_seconds)
    return TRUE;
  else
  {
    desktop_clock_widget_register_time_out_handler(DESKTOP_CLOCK_WIDGET_HOME_PLUGIN(data));
    return FALSE;
  }
}

static void
desktop_clock_widget_get_time(int* hour, int* min, int* sec, Cityinfo* city_info)
{
  struct tm time;
  time_get_remote(time_get_time(),
		  cityinfo_get_zone(city_info),
		  &time);
  *sec = time.tm_sec;
  *min = time.tm_min;
  *hour = time.tm_hour;
}

void static 
desktop_clock_widget_read_settings(DesktopClockWidgetHomePlugin* desktop_plugin)
{
  GKeyFile *key_file;
  gchar* file_name;
  gboolean file_exists;

  key_file = g_key_file_new();
  file_name = g_build_filename(g_get_home_dir(), 
			       DESKTOP_CLOCK_WIDGET_HOME_PLUGIN_SETTINGS_FOLDER,
			       DESKTOP_CLOCK_WIDGET_HOME_PLUGIN_SETTINGS_FILE, NULL);
  file_exists = g_key_file_load_from_file (key_file, file_name, G_KEY_FILE_KEEP_COMMENTS, NULL);
  if(file_exists)
  {
    if(g_key_file_has_key(key_file, desktop_plugin->priv->iD, 
			  DESKTOP_CLOCK_WIDGET_CONFIG_KEY_WIDTH, 
			  NULL))   
      desktop_plugin->priv->clock_width = g_key_file_get_integer(key_file, 
								 desktop_plugin->priv->iD, 
								 DESKTOP_CLOCK_WIDGET_CONFIG_KEY_WIDTH, 
								 NULL);
    if(g_key_file_has_key(key_file, 
			  desktop_plugin->priv->iD, 
			  DESKTOP_CLOCK_WIDGET_CONFIG_KEY_HEIGHT, 
			  NULL))
      desktop_plugin->priv->clock_height = g_key_file_get_integer(key_file, 
								  desktop_plugin->priv->iD, 
								  DESKTOP_CLOCK_WIDGET_CONFIG_KEY_HEIGHT, 
								  NULL);
    if(g_key_file_has_key(key_file, desktop_plugin->priv->iD, 
			  DESKTOP_CLOCK_WIDGET_CONFIG_KEY_SHOW_SECONDS, 
			  NULL))
      desktop_plugin->priv->show_seconds = g_key_file_get_boolean(key_file, 
								  desktop_plugin->priv->iD, 
								  DESKTOP_CLOCK_WIDGET_CONFIG_KEY_SHOW_SECONDS, 
								  NULL);
    if(g_key_file_has_key(key_file, 
			  desktop_plugin->priv->iD, 
			  DESKTOP_CLOCK_WIDGET_CONFIG_KEY_TWENTY_FOUR_HOURS, 
			  NULL))
      desktop_plugin->priv->twenty_four_hours = g_key_file_get_boolean(key_file, 
								       desktop_plugin->priv->iD, 
								       DESKTOP_CLOCK_WIDGET_CONFIG_KEY_TWENTY_FOUR_HOURS, 
								       NULL);
    if(g_key_file_has_key(key_file, 
			  desktop_plugin->priv->iD, 
			  DESKTOP_CLOCK_WIDGET_CONFIG_KEY_SHOW_CITYNAME, 
			  NULL))
      desktop_plugin->priv->show_cityname = g_key_file_get_boolean(key_file, 
								   desktop_plugin->priv->iD, 
								   DESKTOP_CLOCK_WIDGET_CONFIG_KEY_SHOW_CITYNAME, 
								   NULL);
    if(g_key_file_has_key(key_file, 
			  desktop_plugin->priv->iD, 
			  DESKTOP_CLOCK_WIDGET_CONFIG_KEY_ZONE_ID, 
			  NULL))
      desktop_plugin->priv->city_info_id = g_key_file_get_integer(key_file, 
								  desktop_plugin->priv->iD, 
								  DESKTOP_CLOCK_WIDGET_CONFIG_KEY_ZONE_ID, 
								  NULL);
    if(g_key_file_has_key(key_file, 
			  desktop_plugin->priv->iD, 
			  DESKTOP_CLOCK_WIDGET_CONFIG_KEY_USE_STOCK_THEME, 
			  NULL))
      desktop_plugin->priv->use_stock_theme = g_key_file_get_boolean(key_file, 
								     desktop_plugin->priv->iD, 
								     DESKTOP_CLOCK_WIDGET_CONFIG_KEY_USE_STOCK_THEME, 
								     NULL);
    if(g_key_file_has_key(key_file, 
			  desktop_plugin->priv->iD, 
			  DESKTOP_CLOCK_WIDGET_CONFIG_KEY_CUSTOM_THEME_NAME, 
			  NULL))
      desktop_plugin->priv->custom_theme_name = g_key_file_get_string(key_file, 
								      desktop_plugin->priv->iD, 
								      DESKTOP_CLOCK_WIDGET_CONFIG_KEY_CUSTOM_THEME_NAME, 
								      NULL);
    if(desktop_plugin->priv->city_info_id != CITYINFO_WRONG_ID)    
      desktop_plugin->priv->city_info = cityinfo_from_id(desktop_plugin->priv->city_info_id);
  }
  if(desktop_plugin->priv->custom_theme_name != NULL)
  {
    desktop_plugin->priv->theme_path = g_build_filename(g_get_home_dir(),
							DESKTOP_CLOCK_WIDGET_HOME_PLUGIN_SETTINGS_FOLDER, "theme",
							desktop_plugin->priv->iD, NULL);
  }
  else
  {
    desktop_plugin->priv->theme_path = g_strdup(DESKTOP_CLOCK_WIDGET_BUILTIN_THEME);
  }
  g_key_file_free(key_file);
  g_free(file_name);
}

void static
desktop_clock_widget_write_settings(DesktopClockWidgetHomePlugin* desktop_plugin)
{
  gchar* config_file_folder = g_build_filename(g_get_home_dir(), 
					       DESKTOP_CLOCK_WIDGET_HOME_PLUGIN_SETTINGS_FOLDER, 
					       NULL);
  if(!g_file_test(config_file_folder, (G_FILE_TEST_EXISTS | G_FILE_TEST_IS_DIR)))
  {
    if(g_mkdir(config_file_folder, 0755)!=0)
    {
      g_free(config_file_folder);
      return;
    }
  }
  GKeyFile *key_file;
  gchar* file_name;
  gchar *file_data;
  gsize size;
  key_file = g_key_file_new();
  file_name = g_build_filename(config_file_folder, 
			       DESKTOP_CLOCK_WIDGET_HOME_PLUGIN_SETTINGS_FILE, 
			       NULL);
  g_free(config_file_folder);
  g_key_file_load_from_file (key_file, file_name, G_KEY_FILE_KEEP_COMMENTS, NULL);

  g_key_file_set_integer(key_file,
 			 desktop_plugin->priv->iD,  
 			 DESKTOP_CLOCK_WIDGET_CONFIG_KEY_WIDTH,  
			 desktop_plugin->priv->clock_width);
  g_key_file_set_integer(key_file,
 			 desktop_plugin->priv->iD,  
 			 DESKTOP_CLOCK_WIDGET_CONFIG_KEY_HEIGHT,  
			 desktop_plugin->priv->clock_height);
  g_key_file_set_boolean(key_file, 
			 desktop_plugin->priv->iD, 
			 DESKTOP_CLOCK_WIDGET_CONFIG_KEY_SHOW_SECONDS,			 
			 desktop_plugin->priv->show_seconds);
  g_key_file_set_boolean(key_file, 
			 desktop_plugin->priv->iD, 
			 DESKTOP_CLOCK_WIDGET_CONFIG_KEY_TWENTY_FOUR_HOURS,
			 desktop_plugin->priv->twenty_four_hours);
  g_key_file_set_boolean(key_file, 
			 desktop_plugin->priv->iD, 
			 DESKTOP_CLOCK_WIDGET_CONFIG_KEY_SHOW_CITYNAME,			 
			 desktop_plugin->priv->show_cityname);
  g_key_file_set_boolean(key_file, 
			 desktop_plugin->priv->iD, 
			 DESKTOP_CLOCK_WIDGET_CONFIG_KEY_USE_STOCK_THEME,			 
			 desktop_plugin->priv->use_stock_theme);  
  g_key_file_set_integer(key_file, 
			 desktop_plugin->priv->iD, 
			 DESKTOP_CLOCK_WIDGET_CONFIG_KEY_ZONE_ID, 
			 desktop_plugin->priv->city_info_id);
  if(desktop_plugin->priv->custom_theme_name!=NULL)
  {
    g_key_file_set_string(key_file, 
			  desktop_plugin->priv->iD, 
			  DESKTOP_CLOCK_WIDGET_CONFIG_KEY_CUSTOM_THEME_NAME,			 
			  desktop_plugin->priv->custom_theme_name);
  }
  else
  {
    if(g_key_file_has_key(key_file, 
			  desktop_plugin->priv->iD, 
			  DESKTOP_CLOCK_WIDGET_CONFIG_KEY_CUSTOM_THEME_NAME, 
			  NULL))
    {
      g_key_file_remove_key(key_file,
			    desktop_plugin->priv->iD, 
			    DESKTOP_CLOCK_WIDGET_CONFIG_KEY_CUSTOM_THEME_NAME,
			    NULL);
    }
  }
  file_data = g_key_file_to_data (key_file, &size, NULL);
  g_file_set_contents(file_name, file_data, size, NULL);
  
  g_key_file_free (key_file); 
  g_free(file_name); 
  g_free(file_data); 
}

static void
desktop_clock_widget_register_time_out_handler(DesktopClockWidgetHomePlugin* desktop_plugin)
{
  if(desktop_plugin->priv->time_out_handler)
  {
    g_source_remove(desktop_plugin->priv->time_out_handler);
    desktop_plugin->priv->time_out_handler = 0;
  }
  if(desktop_plugin->priv->show_seconds)
  {
    desktop_plugin->priv->time_out_handler = g_timeout_add_seconds(1, 
								   desktop_clock_widget_update_time,
								   desktop_plugin);
  }
  else
  {
    int hour, min, sec;
    desktop_clock_widget_get_time(&hour, &min, &sec, desktop_plugin->priv->city_info);
    desktop_plugin->priv->time_out_handler = g_timeout_add_seconds(60-sec, 
								   desktop_clock_widget_update_time,
								   (gpointer) desktop_plugin);
  }
}

static void
desktop_clock_widget_init_theme(DesktopClockWidgetHomePlugin* desktop_plugin)
{
  DesktopClockSVGTheme* theme = g_new0(DesktopClockSVGTheme, 1);
  if(desktop_plugin->priv->use_stock_theme || desktop_plugin->priv->theme_path==NULL)
  {
    theme->path = g_strdup(DESKTOP_CLOCK_WIDGET_BUILTIN_THEME);
  }
  else
  {
    theme->path = g_strdup(desktop_plugin->priv->theme_path);
  }
  if(desktop_clock_widget_read_theme(theme))
  {
    RsvgDimensionData dimension_data;
    rsvg_handle_get_dimensions(theme->drop_shadow,
			       &dimension_data);
    desktop_plugin->priv->image_width = dimension_data.width;
    desktop_plugin->priv->image_height = dimension_data.height;
    cairo_t* main_cr = gdk_cairo_create(GDK_DRAWABLE(GTK_WIDGET(desktop_plugin)->window));

    desktop_clock_widget_render_background(main_cr,
					   theme,
					   desktop_plugin);
    desktop_clock_widget_render_foreground(main_cr,
					   theme,
					   desktop_plugin);
    if(desktop_plugin->priv->hour_svg_handle != NULL)
      g_object_unref(desktop_plugin->priv->hour_svg_handle);
    if(desktop_plugin->priv->hour_shadow_svg_handle != NULL)
      g_object_unref(desktop_plugin->priv->hour_shadow_svg_handle);
    if(desktop_plugin->priv->minute_svg_handle != NULL)
      g_object_unref(desktop_plugin->priv->minute_svg_handle);
    if(desktop_plugin->priv->minute_shadow_svg_handle != NULL)
      g_object_unref(desktop_plugin->priv->minute_shadow_svg_handle);
    if(desktop_plugin->priv->second_svg_handle != NULL)
      g_object_unref(desktop_plugin->priv->second_svg_handle);
    if(desktop_plugin->priv->second_shadow_svg_handle != NULL)
      g_object_unref(desktop_plugin->priv->second_shadow_svg_handle);
    desktop_plugin->priv->hour_svg_handle = theme->hour_hand;
    desktop_plugin->priv->hour_shadow_svg_handle = theme->hour_hand_shadow;
    desktop_plugin->priv->minute_svg_handle = theme->minute_hand;
    desktop_plugin->priv->minute_shadow_svg_handle = theme->minute_hand_shadow;
    desktop_plugin->priv->second_svg_handle = theme->second_hand;
    desktop_plugin->priv->second_shadow_svg_handle = theme->second_hand_shadow;
    g_object_ref(desktop_plugin->priv->hour_svg_handle);
    g_object_ref(desktop_plugin->priv->hour_shadow_svg_handle);
    g_object_ref(desktop_plugin->priv->minute_svg_handle);
    g_object_ref(desktop_plugin->priv->minute_shadow_svg_handle);
    g_object_ref(desktop_plugin->priv->second_svg_handle);
    g_object_ref(desktop_plugin->priv->second_shadow_svg_handle);
  }
  desktop_clock_widget_free_svg_theme(theme);
}

static int
clock_city_name_sort_func(GtkTreeModel *model,
			  GtkTreeIter* a,
			  GtkTreeIter* b,
			  gpointer user_data)
{
  gchar* val1;
  gchar* val2;
  gtk_tree_model_get(model, a,
		     0, &val1, -1);
  gtk_tree_model_get(model, b,
		     0, &val2, -1);
  int ret = g_strcmp0(val1, val2);
  g_free(val1);
  g_free(val2);
  return ret;
}
static gchar* 
world_clock_gmt_offset_string(Cityinfo* cityinfo)
{
  gchar* name = cityinfo_get_name(cityinfo);
  gchar* country = cityinfo_get_country(cityinfo);
  gchar* timeoffset = NULL;
  if(!name) 
    return timeoffset;

  int offset = time_get_utc_offset(cityinfo_get_zone(cityinfo));
  if((offset % 3600)==0)
    timeoffset = g_strdup_printf("%s, %s, (UTC %+d)", name, country, -offset/3600);
  else
    timeoffset = g_strdup_printf("%s, %s, (UTC %+d:%d)", name, country, -offset/3600, 30);
  return timeoffset;
}

static void
desktop_clock_widget_init_clock_list(GtkListStore* list_store)
{
  Cityinfo** cities = cityinfo_get_all();
  Cityinfo** cities_iter = cities;
  GtkTreeIter iter;
  while(*cities_iter)
  {
    gchar* name = world_clock_gmt_offset_string(*cities_iter);
    gint id = cityinfo_get_id(*cities_iter);
    
    if(name)
    {
      gtk_list_store_append (list_store, &iter); 
      gtk_list_store_set(list_store, &iter, 
			 0, name, 
			 1, id, -1);

    }
    g_free(name);
    ++cities_iter;
  }
  cityinfo_free_all(cities);
  
  gtk_tree_sortable_set_sort_column_id(GTK_TREE_SORTABLE(list_store),
				       0,
				       GTK_SORT_ASCENDING);
  gtk_tree_sortable_set_sort_func(GTK_TREE_SORTABLE(list_store),
                                  0, 
                                  clock_city_name_sort_func,NULL,NULL);
}


static void
desktop_clock_widget_select_theme(GtkButton* button, gpointer data)
{
  GtkWidget* dialog;
  GtkWidget* parent = gtk_widget_get_toplevel(GTK_WIDGET(button));
  DesktopClockSVGTheme* theme = (DesktopClockSVGTheme*)data;
  parent = gtk_widget_get_parent(parent);
  dialog = hildon_file_chooser_dialog_new(GTK_WINDOW(parent),
					  GTK_FILE_CHOOSER_ACTION_SELECT_FOLDER);
  gtk_widget_show_all(GTK_WIDGET(dialog));
  
  int ret = gtk_dialog_run(GTK_DIALOG(dialog));
  if(ret == GTK_RESPONSE_OK)
  {
    gchar* filename = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(dialog));
    theme->path=g_strdup(filename);

    if(desktop_clock_widget_read_theme(theme))
    {
      GFile* gfile = g_file_new_for_path(filename);
      gchar* folder_name = g_file_get_basename(gfile);
      hildon_button_set_value(HILDON_BUTTON(button),
			      folder_name);
      g_free(folder_name);
      g_object_unref(gfile);
    }
    else
    {
      g_free(theme->path);
      theme->path = NULL;
      hildon_banner_show_information(NULL, NULL, "No theme files found");
    }
    g_free(filename);
  }
  gtk_widget_destroy(dialog);
}

static void
desktop_clock_widget_toggle_stock_theme(HildonCheckButton* button, gpointer data)
{
  if(GTK_IS_BUTTON(data))
  {
    gtk_widget_set_sensitive(GTK_WIDGET(data), 
			     !hildon_check_button_get_active(button));
  }
}

static void
desktop_clock_widget_cleanup_old_theme_files(gchar* old_theme_path)
{
  gchar* svg_image_path = NULL;
  svg_image_path = g_build_filename(old_theme_path, CLOCK_DROP_SHADOW_SVG, NULL);
  g_remove(svg_image_path);
  g_free(svg_image_path);

  svg_image_path = g_build_filename(old_theme_path, CLOCK_FACE_SHADOW_SVG, NULL);
  g_remove(svg_image_path);
  g_free(svg_image_path);

  svg_image_path = g_build_filename(old_theme_path, CLOCK_FACE_SVG, NULL);
  g_remove(svg_image_path);
  g_free(svg_image_path);

  svg_image_path = g_build_filename(old_theme_path, CLOCK_FRAME_SVG, NULL);
  g_remove(svg_image_path);
  g_free(svg_image_path);

  svg_image_path = g_build_filename(old_theme_path, CLOCK_GLASS_SVG, NULL);
  g_remove(svg_image_path);
  g_free(svg_image_path);

  svg_image_path = g_build_filename(old_theme_path, CLOCK_MARKS_SVG, NULL);
  g_remove(svg_image_path);
  g_free(svg_image_path);

  svg_image_path = g_build_filename(old_theme_path, CLOCK_HOUR_HAND_SVG, NULL);
  g_remove(svg_image_path);
  g_free(svg_image_path);

  svg_image_path = g_build_filename(old_theme_path, CLOCK_HOUR_HAND_SHADOW_SVG, NULL);
  g_remove(svg_image_path);
  g_free(svg_image_path);

  svg_image_path = g_build_filename(old_theme_path, CLOCK_MINUTE_HAND_SVG, NULL);
  g_remove(svg_image_path);
  g_free(svg_image_path);

  svg_image_path = g_build_filename(old_theme_path, CLOCK_MINUTE_HAND_SHADOW_SVG, NULL);
  g_remove(svg_image_path);
  g_free(svg_image_path);

  svg_image_path = g_build_filename(old_theme_path, CLOCK_SECOND_HAND_SVG, NULL);
  g_remove(svg_image_path);
  g_free(svg_image_path);

  svg_image_path = g_build_filename(old_theme_path, CLOCK_SECOND_HAND_SHADOW_SVG, NULL);
  g_remove(svg_image_path);
  g_free(svg_image_path);
}

static void
desktop_clock_widget_copy_theme_file(const gchar* source_dir, const gchar* dest_dir, const gchar* file)
{
  gchar* source_path= g_build_filename(source_dir, file, NULL);
  gchar* dest_path= g_build_filename(dest_dir, file, NULL);
  GFile* source = g_file_new_for_path(source_path);
  GFile* dest = g_file_new_for_path(dest_path);
  GError* error = NULL;

  if(!g_file_copy(source, dest,
		  G_FILE_COPY_NONE,
		  NULL,
		  NULL,
		  NULL,
		  &error))
  {
    g_warning("can not copy %s to %s \n", source_path, dest_path);
  }
  if(error)
  {
    g_warning("erro on copy %s to %s (%s)\n", source_path, dest_path, error->message);
    g_error_free(error);
    error = NULL;
  }
  g_object_unref(source);
  g_object_unref(dest);
  g_free(source_path);
  g_free(dest_path);
}

static void
desktop_clock_widget_copy_theme_files_to_config(gchar* theme_path, gchar* config_theme_path)
{
  desktop_clock_widget_copy_theme_file(theme_path, config_theme_path, CLOCK_DROP_SHADOW_SVG);
  desktop_clock_widget_copy_theme_file(theme_path, config_theme_path, CLOCK_FACE_SHADOW_SVG);
  desktop_clock_widget_copy_theme_file(theme_path, config_theme_path, CLOCK_FACE_SVG);
  desktop_clock_widget_copy_theme_file(theme_path, config_theme_path, CLOCK_FRAME_SVG);
  desktop_clock_widget_copy_theme_file(theme_path, config_theme_path, CLOCK_GLASS_SVG);
  desktop_clock_widget_copy_theme_file(theme_path, config_theme_path, CLOCK_MARKS_SVG);
  desktop_clock_widget_copy_theme_file(theme_path, config_theme_path, CLOCK_HOUR_HAND_SVG);
  desktop_clock_widget_copy_theme_file(theme_path, config_theme_path, CLOCK_HOUR_HAND_SHADOW_SVG);
  desktop_clock_widget_copy_theme_file(theme_path, config_theme_path, CLOCK_MINUTE_HAND_SVG);
  desktop_clock_widget_copy_theme_file(theme_path, config_theme_path, CLOCK_MINUTE_HAND_SHADOW_SVG);
  desktop_clock_widget_copy_theme_file(theme_path, config_theme_path, CLOCK_SECOND_HAND_SVG);
  desktop_clock_widget_copy_theme_file(theme_path, config_theme_path, CLOCK_SECOND_HAND_SHADOW_SVG);
}

static void
desktop_clock_widget_save_custom_theme(gchar* theme_path, DesktopClockWidgetHomePlugin* desktop_plugin)
{
  GFile* gfile = g_file_new_for_path(theme_path);
  GFile* parent = g_file_get_parent(gfile);
  gchar* folder_name = g_file_get_basename(gfile);
  desktop_plugin->priv->custom_theme_name = folder_name;

  gchar* config_theme_path =
    g_build_filename(g_get_home_dir(),
		     DESKTOP_CLOCK_WIDGET_HOME_PLUGIN_SETTINGS_FOLDER, "theme",
		     desktop_plugin->priv->iD,
		     NULL);
  if(g_strcmp0(theme_path, config_theme_path)==0)
  {
    // we are done
    g_object_unref(gfile);
    g_object_unref(parent);
    return;
  }
  desktop_clock_widget_cleanup_old_theme_files(config_theme_path);
  if(g_mkdir_with_parents(config_theme_path, 0755)!=0)
  {
    g_warning("can not create config theme path\n");
  }
  else
  {
    desktop_clock_widget_copy_theme_files_to_config(theme_path, config_theme_path);
  }
  g_object_unref(gfile);
  g_object_unref(parent);
}

static void
desktop_clock_widget_show_settings_dialog(GtkWidget* widget, gpointer data)
{
  DesktopClockWidgetHomePlugin* desktop_plugin = DESKTOP_CLOCK_WIDGET_HOME_PLUGIN(widget);
  GtkWidget *dialog;
  GtkListStore *timezone_list = NULL;
  GtkWidget* selector = NULL;
  GtkWidget* show_seconds_checkbutton;
  GtkWidget* twenty_four_hour_checkbutton;
  GtkWidget* timezone_button;
  GtkWidget* show_cityname_checkbutton;
  GtkWidget* clock_width_slider;
  GtkWidget* clock_height_slider;
  GtkWidget* stock_theme_checkbutton;
  GtkWidget* custom_theme_selection_button;

  timezone_list = gtk_list_store_new(2, G_TYPE_STRING, G_TYPE_INT);
  desktop_clock_widget_init_clock_list(timezone_list);
  selector = hildon_touch_selector_entry_new_text();
  hildon_touch_selector_entry_set_text_column(HILDON_TOUCH_SELECTOR_ENTRY(selector), 0);
  hildon_touch_selector_set_model(HILDON_TOUCH_SELECTOR(selector),
				  0, GTK_TREE_MODEL(timezone_list));

  timezone_button = hildon_picker_button_new(HILDON_SIZE_FINGER_HEIGHT,
					     HILDON_BUTTON_ARRANGEMENT_VERTICAL);

  hildon_button_set_title(HILDON_BUTTON(timezone_button),
			  "Timezone");
  hildon_picker_button_set_selector(HILDON_PICKER_BUTTON(timezone_button),
				   HILDON_TOUCH_SELECTOR(selector));

  gchar* cityname = world_clock_gmt_offset_string(desktop_plugin->priv->city_info);
  hildon_button_set_value(HILDON_BUTTON(timezone_button),
			  cityname);
  g_free(cityname);
  show_seconds_checkbutton = hildon_check_button_new(HILDON_SIZE_FINGER_HEIGHT);
  hildon_check_button_set_active(HILDON_CHECK_BUTTON(show_seconds_checkbutton),
				 desktop_plugin->priv->show_seconds);
  gtk_button_set_label(GTK_BUTTON(show_seconds_checkbutton),
		       "Show Seconds");
  
  twenty_four_hour_checkbutton = hildon_check_button_new(HILDON_SIZE_FINGER_HEIGHT);
  hildon_check_button_set_active(HILDON_CHECK_BUTTON(twenty_four_hour_checkbutton),
				 desktop_plugin->priv->twenty_four_hours);

  gtk_button_set_label(GTK_BUTTON(twenty_four_hour_checkbutton),
		       "24 Hours");

  show_cityname_checkbutton = hildon_check_button_new(HILDON_SIZE_FINGER_HEIGHT);
  hildon_check_button_set_active(HILDON_CHECK_BUTTON(show_cityname_checkbutton),
				 desktop_plugin->priv->show_cityname);
  gtk_button_set_label(GTK_BUTTON(show_cityname_checkbutton),
		       "Display Name");

  clock_width_slider = hildon_gtk_hscale_new();
  gtk_adjustment_configure(gtk_range_get_adjustment(GTK_RANGE(clock_width_slider)),
			   desktop_plugin->priv->clock_width,
			   32, 351,
			   1, 1, 1);
  gtk_scale_set_value_pos(GTK_SCALE(clock_width_slider), GTK_POS_RIGHT);
  gtk_scale_set_draw_value(GTK_SCALE(clock_width_slider), TRUE);
  gtk_scale_set_digits(GTK_SCALE(clock_width_slider), 0);
  clock_height_slider = hildon_gtk_hscale_new();
  gtk_adjustment_configure(gtk_range_get_adjustment(GTK_RANGE(clock_height_slider)),
			   desktop_plugin->priv->clock_height,
			   32, 351,
			   1, 1, 1);
  gtk_scale_set_value_pos(GTK_SCALE(clock_height_slider), GTK_POS_RIGHT);
  gtk_scale_set_draw_value(GTK_SCALE(clock_height_slider), TRUE);
  gtk_scale_set_digits(GTK_SCALE(clock_height_slider), 0);

  stock_theme_checkbutton = hildon_check_button_new(HILDON_SIZE_FINGER_HEIGHT);
  gtk_button_set_label(GTK_BUTTON(stock_theme_checkbutton),
		       "Built-in Theme");
  hildon_check_button_set_active(HILDON_CHECK_BUTTON(stock_theme_checkbutton),
				 desktop_plugin->priv->use_stock_theme);
  custom_theme_selection_button = hildon_picker_button_new(HILDON_SIZE_FINGER_HEIGHT,
							   HILDON_BUTTON_ARRANGEMENT_VERTICAL);
  hildon_button_set_title(HILDON_BUTTON(custom_theme_selection_button),
	  "Custom");	  
  hildon_button_set_value(HILDON_BUTTON(custom_theme_selection_button),
                          desktop_plugin->priv->custom_theme_name);	

  gtk_widget_set_sensitive(custom_theme_selection_button,
			   !desktop_plugin->priv->use_stock_theme);

  DesktopClockSVGTheme* theme = NULL;
  theme = g_new0(DesktopClockSVGTheme, 1);
  g_signal_connect(custom_theme_selection_button, "clicked", G_CALLBACK(desktop_clock_widget_select_theme), theme);
  
  g_signal_connect(stock_theme_checkbutton, "toggled", G_CALLBACK(desktop_clock_widget_toggle_stock_theme), custom_theme_selection_button);
  dialog = gtk_dialog_new_with_buttons("Settings",
				       NULL,
				       0,
				       dgettext("hildon-libs", "wdgt_bd_done"),
				       GTK_RESPONSE_OK,
				       NULL);
  GtkWidget* box1 = gtk_hbox_new(TRUE, 3);
  gtk_box_pack_start(GTK_BOX(box1), show_seconds_checkbutton, TRUE, TRUE, 1);
  gtk_box_pack_start(GTK_BOX(box1), twenty_four_hour_checkbutton, TRUE, TRUE, 1);

  GtkWidget* box2 = gtk_hbox_new(FALSE, 3);
  GtkWidget* box2a = gtk_vbox_new(TRUE, 3);
  GtkWidget* box2b = gtk_vbox_new(TRUE, 3);
  GtkWidget* box3 = gtk_hbox_new(TRUE, 3);
  GtkWidget* box4 = gtk_hbox_new(TRUE, 3);
  gtk_box_pack_start(GTK_BOX(box3), stock_theme_checkbutton, TRUE, TRUE, 1);
  gtk_box_pack_start(GTK_BOX(box3), custom_theme_selection_button, TRUE, TRUE, 1);

  GtkWidget* wlabel = gtk_label_new("Width:");
  GtkWidget* hlabel = gtk_label_new("Height:");
  
  gtk_box_pack_start(GTK_BOX(box2a), wlabel, TRUE, TRUE, 1);
  gtk_box_pack_start(GTK_BOX(box2a), hlabel, TRUE, TRUE, 1);

  gtk_box_pack_start(GTK_BOX(box2b), clock_width_slider, TRUE, TRUE, 1);
  gtk_box_pack_start(GTK_BOX(box2b), clock_height_slider, TRUE, TRUE, 1);

  gtk_box_pack_start(GTK_BOX(box2), box2a, FALSE, FALSE, 1);
  gtk_box_pack_start(GTK_BOX(box2), box2b, TRUE, TRUE, 1);


  gtk_box_pack_start(GTK_BOX(box4), show_cityname_checkbutton, TRUE, TRUE, 1);
  gtk_box_pack_start(GTK_BOX(box4), timezone_button, TRUE, TRUE, 1);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), box2, TRUE, TRUE, 1);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), box3, TRUE, TRUE, 1);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), box4, TRUE, TRUE, 1);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), box1, TRUE, TRUE, 1);

  gtk_widget_show_all(dialog);

  if(gtk_dialog_run(GTK_DIALOG(dialog)) == GTK_RESPONSE_OK)
  {
    GtkTreeIter iter;
    int new_width = gtk_range_get_value(GTK_RANGE(clock_width_slider));
    int new_height = gtk_range_get_value(GTK_RANGE(clock_height_slider));
    gboolean use_stock_theme = hildon_check_button_get_active(HILDON_CHECK_BUTTON(stock_theme_checkbutton));
    
    gboolean reread_theme = 
      new_width!=desktop_plugin->priv->clock_width ||
      new_height!=desktop_plugin->priv->clock_height ||
      use_stock_theme!=desktop_plugin->priv->use_stock_theme ||
      theme->path!=NULL;

    desktop_plugin->priv->show_seconds = hildon_check_button_get_active(HILDON_CHECK_BUTTON(show_seconds_checkbutton));
    desktop_plugin->priv->twenty_four_hours = hildon_check_button_get_active(HILDON_CHECK_BUTTON(twenty_four_hour_checkbutton));
    desktop_plugin->priv->clock_width = new_width;
    desktop_plugin->priv->clock_height = new_height;
    desktop_plugin->priv->use_stock_theme = use_stock_theme;
    desktop_plugin->priv->show_cityname = hildon_check_button_get_active(HILDON_CHECK_BUTTON(show_cityname_checkbutton));
    hildon_touch_selector_get_selected(HILDON_TOUCH_SELECTOR(selector),
				       0, &iter);
    if(gtk_list_store_iter_is_valid(timezone_list, &iter))
    {
      gint id = 0;
      gtk_tree_model_get(GTK_TREE_MODEL(timezone_list),
			 &iter, 1, &id, -1);
      if(desktop_plugin->priv->city_info)
	cityinfo_free(desktop_plugin->priv->city_info);
      desktop_plugin->priv->city_info = cityinfo_from_id(id);
      desktop_plugin->priv->city_info_id = id;
    }
    
    if(!desktop_plugin->priv->use_stock_theme && theme->path!=NULL)
    {
      desktop_clock_widget_save_custom_theme(theme->path, desktop_plugin);
      GFile* gfile = g_file_new_for_path(theme->path);
      GFile* parent = g_file_get_parent(gfile);
      gchar* folder_name = g_file_get_basename(gfile);
      desktop_plugin->priv->custom_theme_name = folder_name;
      desktop_plugin->priv->theme_path = g_strdup(theme->path);
      g_object_unref(gfile);
      g_object_unref(parent);
    }
    if(reread_theme)
    {
      desktop_clock_widget_init_theme(desktop_plugin);
    }

    int height = desktop_plugin->priv->clock_height;
    if(desktop_plugin->priv->show_cityname)
      height +=8;
    gtk_window_resize(GTK_WINDOW(desktop_plugin), 
		      desktop_plugin->priv->clock_width,
		      height);

    desktop_clock_widget_write_settings(desktop_plugin);
    gtk_widget_queue_draw(GTK_WIDGET(desktop_plugin));
    desktop_clock_widget_register_time_out_handler(desktop_plugin);
  }
  desktop_clock_widget_free_svg_theme(theme);
  gtk_widget_destroy (dialog);
  g_object_unref(timezone_list);
}

static void
desktop_clock_widget_on_current_desktop(GtkWidget* widget,
					gpointer data)
{
  DesktopClockWidgetHomePlugin* desktop_plugin = DESKTOP_CLOCK_WIDGET_HOME_PLUGIN(widget);
  gboolean on;
  g_object_get(widget, "is-on-current-desktop", &on, NULL);
  if(on)
  {
    if(desktop_plugin->priv->time_out_handler==0)
    {
      desktop_clock_widget_register_time_out_handler(desktop_plugin);
    }
    if(!desktop_plugin->priv->show_seconds)
      gtk_widget_queue_draw(GTK_WIDGET(desktop_plugin));
  }
  else
  {
    if(desktop_plugin->priv->time_out_handler!=0)
    {
      g_source_remove(desktop_plugin->priv->time_out_handler);
      desktop_plugin->priv->time_out_handler = 0;
    }
  }
}

static void
desktop_clock_widget_init_default_timezone(DesktopClockWidgetHomePlugin* desktop_plugin)
{
  int ret = 0;
  gchar current_tz[32];
  ret = time_get_timezone(&current_tz[0],
			  32);
  if(ret>0 && ret < 32)
  {
    Cityinfo** cities = cityinfo_get_all();
    Cityinfo** cities_iter = cities;
    gboolean current_found = FALSE;
    while(*cities_iter && !current_found)
    {
      gchar* tz = cityinfo_get_zone(*cities_iter);

      if(g_strcmp0(tz, current_tz+1)==0)
      {
	current_found = TRUE;
	desktop_plugin->priv->city_info = cityinfo_clone(*cities_iter);
      }
      ++cities_iter;
    }
    cityinfo_free_all(cities);
  }
}

static GtkWidget*
build_ui(void)
{
  GtkWidget* contents = gtk_event_box_new();
  gtk_event_box_set_visible_window(GTK_EVENT_BOX(contents), FALSE);
  gtk_container_set_border_width(GTK_CONTAINER(contents), 0);
  gtk_widget_show(contents);
  return contents;
}

static gboolean
desktop_clock_widget_open_clock_app(GtkWidget* widget, GdkEventButton* event, DesktopClockWidgetHomePlugin* desktop_plugin)
{
  DBusGConnection* dbus_conn = hd_home_plugin_item_get_dbus_g_connection(&desktop_plugin->hitem, 
									 DBUS_BUS_SESSION, 
									 NULL);
  DBusGProxy* dbus_proxy = dbus_g_proxy_new_for_name(dbus_conn,
						     HD_APP_MGR_DBUS_NAME,
						     HD_APP_MGR_DBUS_PATH,
						     HD_APP_MGR_DBUS_IFACE);
  if(dbus_proxy)
  {
    
    dbus_g_proxy_call_no_reply(dbus_proxy,
			       HD_APP_MGR_LAUNCH_APPLICATION,
			       G_TYPE_STRING,
			       WORLD_CLOCK_APP_NAME,
			       G_TYPE_INVALID,
			       G_TYPE_INVALID);
    
    g_object_unref(dbus_proxy);
  }
  return FALSE;
}

static gboolean
desktop_clock_widget_expose(GtkWidget* widget, GdkEventExpose *event)
{
  DesktopClockWidgetHomePlugin* desktop_plugin = DESKTOP_CLOCK_WIDGET_HOME_PLUGIN(widget);
  if(desktop_plugin->priv->background == NULL)
  {
    desktop_clock_widget_init_theme(desktop_plugin);
  }

  int seconds = 0;
  int minutes = 0;
  int hours = 0;
  desktop_clock_widget_get_time(&hours, &minutes, &seconds, desktop_plugin->priv->city_info);
  float hour_angle = M_PI/6.0*(hours+minutes/60.0);

  if(desktop_plugin->priv->twenty_four_hours)
  {
    hour_angle/=2.0;
  }

  float min_angle = M_PI/30.0*minutes; 

  cairo_t *cr;
  cr = gdk_cairo_create(GDK_DRAWABLE(widget->window));
  gdk_cairo_region(cr, event->region);
  cairo_clip(cr);
  cairo_set_operator(cr, CAIRO_OPERATOR_OVER);

  cairo_set_source_surface(cr, desktop_plugin->priv->background, 0, 0);


  cairo_paint(cr);
  cairo_save(cr);

  cairo_scale(cr, 
	      desktop_plugin->priv->clock_width/desktop_plugin->priv->image_width, 
	      desktop_plugin->priv->clock_height/desktop_plugin->priv->image_height); 
  cairo_translate(cr, desktop_plugin->priv->image_width/2.0, desktop_plugin->priv->image_height/2.0);
  cairo_rotate(cr, -M_PI/2.0);
  cairo_save(cr);

  cairo_rotate(cr, hour_angle);
  rsvg_handle_render_cairo(desktop_plugin->priv->hour_svg_handle, cr);
  cairo_restore(cr);

  cairo_save(cr);
  cairo_translate(cr, 
		  -0.75, 
		  0.75);
  cairo_rotate(cr, hour_angle);
  rsvg_handle_render_cairo(desktop_plugin->priv->hour_shadow_svg_handle, cr);
  cairo_restore(cr);

  cairo_save(cr);
  cairo_rotate(cr, min_angle);
  rsvg_handle_render_cairo(desktop_plugin->priv->minute_svg_handle, cr);
  cairo_restore(cr);

  cairo_save(cr);
  cairo_translate(cr, 
		  -0.75, 
		  0.75);
  cairo_rotate(cr, min_angle);
  rsvg_handle_render_cairo(desktop_plugin->priv->minute_shadow_svg_handle, cr);
  cairo_restore(cr);

  if(desktop_plugin->priv->show_seconds)
  {
    cairo_save(cr);
    cairo_rotate(cr, 2*M_PI/60.0*seconds);
    rsvg_handle_render_cairo(desktop_plugin->priv->second_svg_handle, cr);
    cairo_restore(cr);

    cairo_save(cr);
    cairo_translate(cr, 
		    -0.75, 
		    0.75);
    cairo_rotate(cr, 2*M_PI/60.0*seconds);
    rsvg_handle_render_cairo(desktop_plugin->priv->second_shadow_svg_handle, cr);
    cairo_restore(cr);
  }

  cairo_restore(cr);
  cairo_set_source_surface(cr, desktop_plugin->priv->foreground, 0, 0);
  cairo_paint(cr);

  if(desktop_plugin->priv->show_cityname)
  {
    PangoLayout *pangoLayout;
    PangoFontDescription *fontDescription;
    pangoLayout = pango_cairo_create_layout(cr);
    const char* font = "Sans 10";
    fontDescription = pango_font_description_from_string(font);
    pango_layout_set_font_description(pangoLayout, fontDescription);
    pango_font_description_free(fontDescription);
    pango_layout_set_width(pangoLayout, 85*1024);
    pango_layout_set_ellipsize(pangoLayout, PANGO_ELLIPSIZE_END);
    pango_layout_set_text(pangoLayout,  cityinfo_get_name(desktop_plugin->priv->city_info), -1);

    int width;
    int height;
    int x = desktop_plugin->priv->clock_width/2;

    pango_layout_get_pixel_size(pangoLayout, &width, &height);
    int y = desktop_plugin->priv->clock_height -height/2;
    cairo_set_line_cap(cr, CAIRO_LINE_CAP_ROUND);
    cairo_set_line_width(cr, height);
    cairo_move_to(cr, x-width/2.0, y+(height/2.0));
    cairo_line_to(cr, x+width/2.0, y+(height/2.0));
    cairo_set_source_rgba(cr, 0.3, 0.3, 0.3, 0.8);
    cairo_stroke(cr);
    cairo_move_to(cr, x-width/2.0, y);
    cairo_set_source_rgba(cr, 1.0, 1.0, 1.0, 1.0);
    pango_cairo_show_layout(cr, pangoLayout);
  }
  cairo_destroy(cr);
  return GTK_WIDGET_CLASS(desktop_clock_widget_home_plugin_parent_class)->expose_event(widget, event);
}

static void
desktop_clock_widget_realize(GtkWidget* widget)
{
  GdkScreen *screen = gtk_widget_get_screen(widget);
  gtk_widget_set_colormap(widget, gdk_screen_get_rgba_colormap(screen));
  gtk_widget_set_app_paintable(widget, TRUE);

  DesktopClockWidgetHomePlugin* desktop_plugin = DESKTOP_CLOCK_WIDGET_HOME_PLUGIN(widget);
  desktop_plugin->priv->iD = hd_home_plugin_item_get_applet_id (HD_HOME_PLUGIN_ITEM (widget));
  desktop_clock_widget_read_settings(desktop_plugin);
  if(desktop_plugin->priv->city_info == NULL || desktop_plugin->priv->city_info_id == CITYINFO_WRONG_ID)
  {
    desktop_clock_widget_init_default_timezone(desktop_plugin);
  }
  int height = desktop_plugin->priv->clock_height;
  if(desktop_plugin->priv->show_cityname)
    height += 16;
  gtk_window_resize(GTK_WINDOW(widget),
		    desktop_plugin->priv->clock_width,
		    height);
		    
  desktop_clock_widget_register_time_out_handler(desktop_plugin);
  GTK_WIDGET_CLASS(desktop_clock_widget_home_plugin_parent_class)->realize(widget);
}

static void
desktop_clock_widget_finalize(GObject* object)
{
  DesktopClockWidgetHomePlugin* desktop_plugin = DESKTOP_CLOCK_WIDGET_HOME_PLUGIN(object);
  g_free(desktop_plugin->priv->iD);
  if(desktop_plugin->priv->time_out_handler)
  {
    g_source_remove(desktop_plugin->priv->time_out_handler);
    desktop_plugin->priv->time_out_handler = 0;
  }
  if(desktop_plugin->priv->background)
    cairo_surface_destroy(desktop_plugin->priv->background);
  if(desktop_plugin->priv->foreground)
    cairo_surface_destroy(desktop_plugin->priv->foreground);
  if(desktop_plugin->priv->hour_svg_handle)
    g_object_unref(desktop_plugin->priv->hour_svg_handle);
  if(desktop_plugin->priv->hour_shadow_svg_handle)
    g_object_unref(desktop_plugin->priv->hour_shadow_svg_handle);
  if(desktop_plugin->priv->minute_svg_handle)
    g_object_unref(desktop_plugin->priv->minute_svg_handle);
  if(desktop_plugin->priv->minute_shadow_svg_handle)
    g_object_unref(desktop_plugin->priv->minute_shadow_svg_handle);
  if(desktop_plugin->priv->second_svg_handle)
    g_object_unref(desktop_plugin->priv->second_svg_handle);
  if(desktop_plugin->priv->second_shadow_svg_handle)
    g_object_unref(desktop_plugin->priv->second_shadow_svg_handle);

  cityinfo_free(desktop_plugin->priv->city_info);


  G_OBJECT_CLASS(desktop_clock_widget_home_plugin_parent_class)->finalize(object);
}

static void
desktop_clock_widget_home_plugin_init(DesktopClockWidgetHomePlugin* desktop_plugin)
{
  desktop_plugin->priv = DESKTOP_CLOCK_WIDGET_HOME_PLUGIN_GET_PRIVATE(desktop_plugin);
  desktop_plugin->priv->iD = NULL;
  desktop_plugin->priv->clock_width = 100;
  desktop_plugin->priv->clock_height = 100;
  desktop_plugin->priv->image_width = 100;
  desktop_plugin->priv->image_height = 100;
  desktop_plugin->priv->show_seconds = TRUE;
  desktop_plugin->priv->twenty_four_hours = FALSE;
  desktop_plugin->priv->show_cityname = FALSE;
  desktop_plugin->priv->city_info_id = CITYINFO_WRONG_ID;
  desktop_plugin->priv->use_stock_theme = TRUE;
  desktop_plugin->priv->custom_theme_name = NULL;
  desktop_plugin->priv->theme_path = NULL;
  desktop_plugin->priv->background = NULL;
  desktop_plugin->priv->foreground = NULL;
  desktop_plugin->priv->hour_svg_handle = NULL;
  desktop_plugin->priv->hour_shadow_svg_handle = NULL;
  desktop_plugin->priv->minute_svg_handle = NULL;
  desktop_plugin->priv->minute_shadow_svg_handle = NULL;
  desktop_plugin->priv->second_svg_handle = NULL;
  desktop_plugin->priv->second_shadow_svg_handle = NULL;
  desktop_plugin->priv->city_info = NULL;
  desktop_plugin->priv->time_out_handler = 0;
  GtkWidget* contents = build_ui();
  hd_home_plugin_item_set_settings (HD_HOME_PLUGIN_ITEM (desktop_plugin), TRUE);
  g_signal_connect(desktop_plugin, "show-settings", G_CALLBACK(desktop_clock_widget_show_settings_dialog), NULL);
  g_signal_connect(desktop_plugin, "notify::is-on-current-desktop", G_CALLBACK(desktop_clock_widget_on_current_desktop), NULL);
  g_signal_connect(GTK_CONTAINER(contents), "button-release-event", G_CALLBACK(desktop_clock_widget_open_clock_app), desktop_plugin);
  gtk_container_add(GTK_CONTAINER(desktop_plugin), contents);
}

static void
desktop_clock_widget_home_plugin_class_init(DesktopClockWidgetHomePluginClass* klass)
{
  g_type_class_add_private(klass, sizeof(DesktopClockWidgetHomePluginPrivate));
  GtkWidgetClass* widget_class = GTK_WIDGET_CLASS(klass);
  widget_class->realize = desktop_clock_widget_realize;
  widget_class->expose_event = desktop_clock_widget_expose;

  G_OBJECT_CLASS(klass)->finalize = desktop_clock_widget_finalize;
}

static void
desktop_clock_widget_home_plugin_class_finalize(DesktopClockWidgetHomePluginClass* klass)
{
}
