#include "dailymotionvideodescriptiontab.h"
#include "dailymotionplaylistdialog.h"
#include "dailymotionmetadatadialog.h"
#include "sharingdialog.h"
#include "sharetofacebookdialog.h"
#include "sharetotwitterdialog.h"
#include "dailymotionvideolistwindow.h"
#include "dailymotionchannelwindow.h"
#include "videoplaybackwindow.h"
#include "../base/dailymotion.h"
#include "../base/sharing.h"
#include "../base/thumbnailcache.h"
#include "../base/settings.h"
#include "../base/transfermanager.h"
#include "../base/playbacklistmodel.h"
#include "../base/urlgrabber.h"
#include "windowstack.h"
#include "../base/videoitem.h"
#include "thumbnaillabel.h"
#include "ratingindicator.h"
#include <QAction>
#include <QLabel>
#include <QScrollArea>
#include <QGridLayout>
#include <QMaemo5Style>

DailymotionVideoDescriptionTab::DailymotionVideoDescriptionTab(QSharedPointer<VideoItem> video, QWidget *parent) :
    AbstractTab(parent),
    m_video(video),
    m_editAction(new QAction(tr("Edit"), this)),
    m_favouritesAction(new QAction(m_video.data()->favourite() ? tr("Delete from favourites") : tr("Add to favourites"), this)),
    m_addToPlaylistAction(new QAction(tr("Add to playlist"), this)),
    m_queueAction(new QAction(tr("Add to playback queue"), this)),
    m_videoDownloadAction(new QAction(tr("Download"), this)),
    m_audioDownloadAction(new QAction(tr("Download (audio)"), this)),
    m_shareAction(new QAction(tr("Share"), this)),
    m_thumbnail(new ThumbnailLabel(m_video.data()->duration(), this)),
    m_avatar(new ThumbnailLabel(this)),
    m_ratingIndicator(new RatingIndicator(m_video.data()->rating(), m_video.data()->ratingCount(), this)),
    m_titleLabel(new QLabel(m_video.data()->title(), this)),
    m_dateLabel(new QLabel(QString("%1 - %2").arg(m_video.data()->date()).arg(m_video.data()->artist()), this)),
    m_viewsLabel(new QLabel(QString("%1   <font color='%2'>%3</font>").arg(tr("Views")).arg(QMaemo5Style::standardColor("SecondaryTextColor").name()).arg(m_video.data()->viewCount()), this)),
    m_descriptionLabel(new QLabel(QString("%1<br><font color='%2'>%3</font>").arg(tr("Description")).arg(QMaemo5Style::standardColor("SecondaryTextColor").name()).arg(m_video.data()->description().isEmpty() ? tr("No description") : m_video.data()->description()), this)),
    m_tagsLabel(new QLabel(this->tagsToString(m_video.data()->tags()), this)),
    m_scrollArea(new QScrollArea(this)),
    m_grid(new QGridLayout(this))
{
    this->addAction(m_editAction);
    this->addAction(m_favouritesAction);
    this->addAction(m_addToPlaylistAction);
    this->addAction(m_queueAction);
    this->addAction(m_videoDownloadAction);
    this->addAction(m_audioDownloadAction);
    this->addAction(m_shareAction);

    QWidget *scrollWidget = new QWidget(m_scrollArea);
    QGridLayout *grid = new QGridLayout(scrollWidget);
    QFont smallFont;
    smallFont.setPixelSize(20);
    m_avatar->setFixedSize(60, 60);
    m_titleLabel->setWordWrap(true);
    m_dateLabel->setWordWrap(true);
    m_dateLabel->setFont(smallFont);
    m_viewsLabel->setFont(smallFont);
    m_descriptionLabel->setWordWrap(true);
    m_descriptionLabel->setFont(smallFont);
    m_tagsLabel->setWordWrap(true);
    m_tagsLabel->setFont(smallFont);

    grid->addWidget(m_titleLabel, 0, 0, 1, 2, Qt::AlignTop);
    grid->addWidget(m_avatar, 1, 0);
    grid->addWidget(m_dateLabel, 1, 1, Qt::AlignTop);
    grid->addWidget(m_ratingIndicator, 2, 0, 1, 2);
    grid->addWidget(m_viewsLabel, 3, 0, 1, 2, Qt::AlignTop);
    grid->addWidget(m_descriptionLabel, 4, 0, 1, 2, Qt::AlignTop);
    grid->addWidget(m_tagsLabel, 5, 0, 1, 2);

    m_scrollArea->setWidgetResizable(true);
    m_scrollArea->setWidget(scrollWidget);
    m_scrollArea->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);

    this->connect(m_thumbnail, SIGNAL(clicked()), this, SLOT(playVideo()));
    this->connect(m_avatar, SIGNAL(clicked()), this, SLOT(showUserChannel()));
    this->connect(m_editAction, SIGNAL(triggered()), this, SLOT(showEditMetadataDialog()));
    this->connect(m_favouritesAction, SIGNAL(triggered()), this, SLOT(toggleFavourite()));
    this->connect(m_addToPlaylistAction, SIGNAL(triggered()), this, SLOT(showPlaylistDialog()));
    this->connect(m_queueAction, SIGNAL(triggered()), this, SLOT(addVideoToPlaybackQueue()));
    this->connect(m_videoDownloadAction, SIGNAL(triggered()), this, SLOT(addVideoDownload()));
    this->connect(m_audioDownloadAction, SIGNAL(triggered()), this, SLOT(addAudioDownload()));
    this->connect(m_shareAction, SIGNAL(triggered()), this, SLOT(showShareDialog()));
    this->connect(m_tagsLabel, SIGNAL(linkActivated(QString)), this, SLOT(search(QString)));
    this->connect(m_video.data(), SIGNAL(favouriteChanged(bool)), this, SLOT(onFavouriteChanged(bool)));

    this->setMenuActions();
    this->setThumbnails();
    this->onOrientationChanged();
}

void DailymotionVideoDescriptionTab::setLandscapeLayout() {
    m_thumbnail->setFixedSize(280, 210);
    m_grid->setContentsMargins(10, 10, 0, 0);

    if (m_grid->isEmpty()) {
        m_grid->addWidget(m_thumbnail, 0, 0, Qt::AlignTop | Qt::AlignHCenter);
        m_grid->addWidget(m_scrollArea, 0, 1, Qt::AlignTop);
    }
    else {
        m_grid->removeWidget(m_scrollArea);
        m_grid->addWidget(m_scrollArea, 0, 1, Qt::AlignTop);
    }
}

void DailymotionVideoDescriptionTab::setPortraitLayout() {
    m_thumbnail->setFixedSize(460, 345);
    m_grid->setContentsMargins(0, 10, 0, 0);

    if (m_grid->isEmpty()) {
        m_grid->addWidget(m_thumbnail, 0, 0, Qt::AlignTop | Qt::AlignHCenter);
        m_grid->addWidget(m_scrollArea, 1, 0, Qt::AlignTop);
    }
    else {
        m_grid->removeWidget(m_scrollArea);
        m_grid->addWidget(m_scrollArea, 1, 0, Qt::AlignTop);
    }
}

void DailymotionVideoDescriptionTab::setMenuActions() {
    bool signedIn = Dailymotion::instance()->userSignedIn();

    m_editAction->setEnabled((signedIn) && (m_video.data()->artistId() == Dailymotion::instance()->username()));
    m_favouritesAction->setEnabled(signedIn);
    m_addToPlaylistAction->setEnabled(signedIn);
}

QString DailymotionVideoDescriptionTab::tagsToString(QStringList tags) {
    QString tagString("Tags<br>");

    if (tags.isEmpty()) {
        tagString.append(QString("<p style='color:%1;'>%2</p>").arg(QMaemo5Style::standardColor("SecondaryTextColor").name()).arg(tr("No tags")));
    }
    else {
        while (!tags.isEmpty()) {
            tagString.append(QString("<a href='%1'>%1</a>   ").arg(tags.takeFirst()));
        }
    }

    return tagString;
}

void DailymotionVideoDescriptionTab::setThumbnails() {
    QImage thumbnail = ThumbnailCache::instance()->thumbnail(m_video.data()->largeThumbnailUrl());
    QImage avatar = ThumbnailCache::instance()->thumbnail(m_video.data()->artistAvatarUrl(), QSize(60, 60));

    if ((thumbnail.isNull()) || (avatar.isNull())) {
        this->connect(ThumbnailCache::instance(), SIGNAL(thumbnailReady()), this, SLOT(setThumbnails()));
    }
    else {
        this->disconnect(ThumbnailCache::instance(), SIGNAL(thumbnailReady()), this, SLOT(setThumbnails()));
    }
    if (!thumbnail.isNull()) {
        m_thumbnail->setImage(thumbnail);
    }
    if (!avatar.isNull()) {
        m_avatar->setImage(avatar);
    }
}

void DailymotionVideoDescriptionTab::showEditMetadataDialog() {
    DailymotionMetadataDialog *dialog = new DailymotionMetadataDialog(m_video, this);
    dialog->open();
}

void DailymotionVideoDescriptionTab::toggleFavourite() {
    if (m_video.data()->favourite()) {
        Dailymotion::instance()->deleteFromFavourites(QStringList(m_video.data()->videoId()));
    }
    else {
        Dailymotion::instance()->addToFavourites(QStringList(m_video.data()->videoId()));
    }
}

void DailymotionVideoDescriptionTab::onFavouriteChanged(bool favourite) {
    m_favouritesAction->setText(favourite ? tr("Delete from favourites") : tr("Add to favourites"));
}

void DailymotionVideoDescriptionTab::showPlaylistDialog() {
    DailymotionPlaylistDialog *dialog = new DailymotionPlaylistDialog(this);
    dialog->open();
    this->connect(dialog, SIGNAL(playlistChosen(QString)), this, SLOT(addVideoToPlaylist(QString)));
    this->connect(dialog, SIGNAL(addToNewPlaylist(QVariantMap)), this, SLOT(addVideoToNewPlaylist(QVariantMap)));
}

void DailymotionVideoDescriptionTab::showShareDialog() {
    SharingDialog *dialog = new SharingDialog(this);
    dialog->open();
    this->connect(dialog, SIGNAL(valueChanged(Services::SharingService)), this, SLOT(shareVideo(Services::SharingService)));
}

void DailymotionVideoDescriptionTab::shareVideo(Services::SharingService service) {
    switch (service) {
    case Services::Facebook:
        this->showFacebookDialog();
        return;
    case Services::Twitter:
        this->showTwitterDialog();
        return;
    case Services::Email:
        Sharing::instance()->shareViaEmail(m_video.data()->title(), m_video.data()->url());
        return;
    case Services::Clipboard:
        Sharing::instance()->copyToClipboard(m_video.data()->url());
        return;
    default:
        qWarning() << "DailymotionVideoDescriptionTab::shareVideo(): Invalid service.";
        return;
    }
}

void DailymotionVideoDescriptionTab::showFacebookDialog() {
    ShareToFacebookDialog *dialog = new ShareToFacebookDialog(m_video, this);
    dialog->open();
}

void DailymotionVideoDescriptionTab::showTwitterDialog() {
    ShareToTwitterDialog *dialog = new ShareToTwitterDialog(m_video, this);
    dialog->open();
}

void DailymotionVideoDescriptionTab::addVideoToPlaylist(const QString &id) {
    Dailymotion::instance()->addToPlaylist(QStringList(m_video.data()->videoId()), id);
}

void DailymotionVideoDescriptionTab::addVideoToNewPlaylist(const QVariantMap &playlist) {
    Dailymotion::instance()->createPlaylist(playlist, QStringList(m_video.data()->videoId()));
}

void DailymotionVideoDescriptionTab::addVideoToPlaybackQueue() {
    PlaybackListModel::playbackQueue()->addVideo(m_video);
}

void DailymotionVideoDescriptionTab::addVideoDownload() {
    TransferManager::instance()->addDownloadTransfer(m_video);
}

void DailymotionVideoDescriptionTab::addAudioDownload() {
    TransferManager::instance()->addDownloadTransfer(m_video, true);
}

void DailymotionVideoDescriptionTab::playVideo() {
    if (Settings::instance()->mediaPlayer() == "cutetube") {
        VideoPlaybackWindow *window = new VideoPlaybackWindow(QList< QSharedPointer<VideoItem> >() << m_video, WindowStack::instance()->currentWindow());
        window->show();
    }
    else {
        UrlGrabber::instance()->getDailymotionVideoUrl(m_video.data()->videoId());
    }
}

void DailymotionVideoDescriptionTab::showUserChannel() {
    DailymotionChannelWindow *window = new DailymotionChannelWindow(m_video.data()->artistId(), WindowStack::instance()->currentWindow());
    window->show();
}

void DailymotionVideoDescriptionTab::search(const QString &query) {
    DailymotionVideoListWindow *window = new DailymotionVideoListWindow(WindowStack::instance()->currentWindow());
    window->search(query, Settings::instance()->searchOrder(), Settings::instance()->languageFilter());
    window->show();
}
