#include "sharing.h"
#include "json.h"
#include "utils.h"
#include "oauth.h"
#include <QNetworkAccessManager>
#include <QNetworkRequest>
#include <QNetworkReply>
#if QT_VERSION >= 0x050000
#include <QGuiApplication>
#include <QUrlQuery>
#else
#include <QApplication>
#endif
#include <QClipboard>
#include <QDateTime>
#include <QDesktopServices>
#ifdef MEEGO_EDITION_HARMATTAN
#include <MDataUri>
#include <maemo-meegotouch-interfaces/shareuiinterface.h>
#else
#include "settings.h"
#endif

using namespace QtJson;
using namespace QtOAuth;

Sharing* sharingInstance = 0;

const QRegExp YOUTUBE_SHARING_RE("(http(s|)://(www.|m.|)youtube.com/(v/|.+)(v=|)|http://youtu.be/)");
const QRegExp DAILYMOTION_SHARING_RE("(http(s|)://(www.|)dailymotion.com/video/|http://dai.ly/)\\w{6}");
const QRegExp VIMEO_SHARING_RE("http(s|)://vimeo.com/\\d+");

const QString FACEBOOK_CLIENT_ID("385476614847489");
const QString FACEBOOK_CLIENT_SECRET("711d38a113d298af9348386957c524e5");
const QString TWITTER_CLIENT_ID("YmfKe14OCdVxn9FCL1S77A");
const QString TWITTER_CLIENT_SECRET("evSW7txyhlvvPXkezKvJgaZ2lC97iXZc4ZLXLZxfsfc");
const QString REDIRECT_URI("https://sites.google.com/site/marxodian/home/cutetube");

Sharing::Sharing(QObject *parent) :
    QObject(parent),
    m_nam(0),
    m_facebookTimestamp(0),
    m_videoCache(new QList< QSharedPointer<VideoItem> >),
    m_busy(false)
{
    if (!sharingInstance) {
        sharingInstance = this;
    }
}

Sharing::~Sharing() {
    this->clearCache();    
}

Sharing* Sharing::instance() {
    return sharingInstance;
}

void Sharing::clearCache() {
    m_videoCache->clear();
}

void Sharing::setBusy(bool isBusy, const QString &message, int numberOfOperations) {
    if (isBusy != this->busy()) {
        m_busy = isBusy;
        emit busyChanged(isBusy);
    }

    if (isBusy) {
        emit busy(message, numberOfOperations);
    }
    else {
        emit busyProgressChanged(numberOfOperations);
    }
}

void Sharing::cancelCurrentOperation() {
    this->setBusy(false);
    emit currentOperationCancelled();
}

QUrl Sharing::facebookAuthUrl() const {
    QUrl url("https://www.facebook.com/dialog/oauth");
#if QT_VERSION >= 0x050000
    QUrlQuery query;
    query.addQueryItem("client_id", FACEBOOK_CLIENT_ID);
    query.addQueryItem("redirect_uri", "https://www.facebook.com/connect/login_success.html");
    query.addQueryItem("response_type", "token");
    query.addQueryItem("display", "touch");
    query.addQueryItem("scope", "publish_stream,read_stream");
    url.setQuery(query);
#else
    url.addQueryItem("client_id", FACEBOOK_CLIENT_ID);
    url.addQueryItem("redirect_uri", "https://www.facebook.com/connect/login_success.html");
    url.addQueryItem("response_type", "token");
    url.addQueryItem("display", "touch");
    url.addQueryItem("scope", "publish_stream,read_stream");
#endif
    return url;
}

void Sharing::signInToFacebook(const QUrl &response) {
    QUrl url(response.toString().replace('#', '?').replace("??", "?")); //Fix malformed response from Facebook
#if QT_VERSION >= 0x050000
    QUrlQuery query(url);
    QString token = query.queryItemValue("access_token");
#else
    QString token = url.queryItemValue("access_token");
#endif
    if (token.isEmpty()) {
#if QT_VERSION >= 0x050000
        QString errorString = query.queryItemValue("error");
#else
        QString errorString = url.queryItemValue("error");
#endif
        if (errorString == "access_denied") {
            emit info(tr("You have denied access to your facebook account"));
        }
        else {
            emit error(tr("Error obtaining facebook authorisation"));
        }
    }
    else {
        emit signedInToFacebook(token);
        emit alert(tr("You are signed in to your facebook account"));
    }
}

void Sharing::cancelFacebookSignIn() {
    this->disconnect(SIGNAL(signedInToFacebook(QString)));
}

QByteArray Sharing::getTwitterOAuthHeader(const QString &method, const QUrl &url, QMap<QString, QString> params) {
    params.insert("oauth_consumer_key", TWITTER_CLIENT_ID);
    params.insert("oauth_consumer_secret", TWITTER_CLIENT_SECRET);

    if (this->userSignedInToTwitter()) {
        params.insert("oauth_token", this->twitterToken());
        params.insert("oauth_token_secret", this->twitterTokenSecret());
    }

    return OAuth::createOAuthHeader(method, url, params);
}

void Sharing::getTwitterAuthUrl() {
    QUrl url("https://api.twitter.com/oauth/request_token");
    QMap<QString, QString> params;
    params.insert("oauth_callback", REDIRECT_URI);
    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", QString("cuteTube/%1 (Qt)").arg(Utils::versionNumberString()).toUtf8());
    request.setRawHeader("Authorization", this->getTwitterOAuthHeader("GET", url, params));
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(checkTwitterAuthUrl()));
}

void Sharing::checkTwitterAuthUrl() {
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    QUrl url("http://twitter.com?" + reply->readAll());
#if QT_VERSION >= 0x050000
    QUrlQuery query(url);
    QString token = query.queryItemValue("oauth_token");
    QString secret = query.queryItemValue("oauth_token_secret");
#else
    QString token = url.queryItemValue("oauth_token");
    QString secret = url.queryItemValue("oauth_token_secret");
#endif
    if ((token.isEmpty()) || (secret.isEmpty())) {
        emit error(tr("Error obtaining twitter authorisation URL"));
    }
    else {
        this->setTwitterRequestToken(token);
        this->setTwitterRequestSecret(secret);
        QUrl authUrl("http://api.twitter.com/oauth/authorize");
#if QT_VERSION >= 0x050000
        QUrlQuery query;
        query.addQueryItem("oauth_token", token);
        authUrl.setQuery(query);
#else
        authUrl.addQueryItem("oauth_token", token);
#endif
        emit gotTwitterAuthUrl(authUrl);
    }

    reply->deleteLater();
}

void Sharing::signInToTwitter(const QUrl &response) {
#if QT_VERSION >= 0x050000
    QUrlQuery query(response);
    QString verifier = query.queryItemValue("oauth_verifier");
#else
    QString verifier = response.queryItemValue("oauth_verifier");
#endif
    if (verifier.isEmpty()) {
        emit error(tr("Error obtaining twitter authorisation verifier"));
    }
    else {
        this->setBusy(true, tr("Signing in"));
        QUrl url("https://api.twitter.com/oauth/access_token");
        QMap<QString, QString> params;
        params.insert("oauth_token", this->twitterRequestToken());
        params.insert("oauth_token_secret", this->twitterRequestSecret());
        params.insert("oauth_verifier", verifier);
        QNetworkRequest request(url);
        request.setRawHeader("User-Agent", QString("cuteTube/%1 (Qt)").arg(Utils::versionNumberString()).toUtf8());
        request.setRawHeader("Authorization", this->getTwitterOAuthHeader("GET", url, params));
        QNetworkReply *reply = this->networkAccessManager()->get(request);
        this->connect(reply, SIGNAL(finished()), this, SLOT(checkIfSignedInToTwitter()));
    }
}

void Sharing::checkIfSignedInToTwitter() {
    this->setBusy(false);
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    QUrl url("http://twitter.com?" + reply->readAll());
#if QT_VERSION >= 0x050000
    QUrlQuery query(url);
    QString token = query.queryItemValue("oauth_token");
    QString secret = query.queryItemValue("oauth_token_secret");
#else
    QString token = url.queryItemValue("oauth_token");
    QString secret = url.queryItemValue("oauth_token_secret");
#endif
    if ((token.isEmpty()) || (secret.isEmpty())) {
        emit error(tr("Error obtaining twitter authorisation"));
    }
    else {
        this->setTwitterRequestToken(QString());
        this->setTwitterRequestSecret(QString());
        emit signedInToTwitter(token, secret);
        emit alert(tr("You are signed in to your twitter account"));
    }

    reply->deleteLater();
}

void Sharing::cancelTwitterSignIn() {
    this->disconnect(SIGNAL(signedInToTwitter(QString,QString)));
}

void Sharing::setFacebookAccessToken(const QString &token) {
    m_facebookToken = token;
    emit userSignedInToFacebookChanged();
}

void Sharing::signOutOfFacebook() {
    this->setFacebookAccessToken(QString());
    emit signedOutOfFacebook();
    emit info(tr("Access token deleted. Please visit the facebook website to revoke access"));
}

void Sharing::setTwitterAccount(const QString &token, const QString &secret) {
    this->setTwitterAccessToken(token);
    this->setTwitterTokenSecret(secret);
    emit userSignedInToTwitterChanged();
}

void Sharing::signOutOfTwitter() {
    this->setTwitterAccount(QString(), QString());
    emit signedOutOfTwitter();
    emit info(tr("Access token deleted. Please visit the twitter website to revoke access"));
}

#ifdef MEEGO_EDITION_HARMATTAN
void Sharing::shareViaHarmattanShareUI(VideoItem *video) {

    MDataUri duri;
    duri.setMimeType("text/x-url");
    duri.setTextData(video->url().toString());
    duri.setAttribute("title", video->title());
    duri.setAttribute("picture", video->largeThumbnailUrl().toString());
    duri.setAttribute("source", video->embedUrl().toString());

    if (!video->description().isEmpty()) {
        duri.setAttribute("description", video->description());
    }

    if (duri.isValid()) {
        ShareUiInterface shareIf("com.nokia.ShareUi");

        if (shareIf.isValid()) {
            shareIf.share(QStringList(duri.toString()));
        }
        else {
            emit error(tr("Unable to share video"));
        }
    }
    else {
        emit error(tr("Unable to share video"));
    }
}
#else
void Sharing::postToFacebook(const QVariantMap &post) {
    m_post = post;

    if (this->userSignedInToFacebook()) {
        this->postToFacebook();
    }
    else {
        emit reauthenticateForFacebook();
        this->connect(this, SIGNAL(signedInToFacebook(QString)), this, SLOT(postToFacebook()));
    }
}

void Sharing::postToFacebook() {
    QByteArray postData = "access_token=" + this->facebookToken().toUtf8()
            + "&message=" + m_post.value("body").toString().toUtf8()
            + "&link=" + m_post.value("videoUrl").toString().toUtf8()
            + "&source=" + m_post.value("embedUrl").toString().toUtf8()
            + "&picture=" + m_post.value("thumbnailUrl").toString().toUtf8()
            + "&name=" + m_post.value("title").toString().toUtf8()
            + "&description=" + m_post.value("description").toString().toUtf8();

    QNetworkRequest request(QUrl("https://graph.facebook.com/me/feed"));
    request.setRawHeader("User-Agent", QString("cuteTube/%1 (Qt)").arg(Utils::versionNumberString()).toUtf8());
    request.setHeader(QNetworkRequest::ContentTypeHeader, "application/x-www-form-urlencoded");
    QNetworkReply *reply = this->networkAccessManager()->post(request, postData);
    this->connect(reply, SIGNAL(finished()), this, SLOT(facebookPostFinished()));
    this->disconnect(this, SIGNAL(signedInToFacebook(QString)), this, SLOT(postToFacebook()));
}

void Sharing::facebookPostFinished() {
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    if (statusCode == 400) {
        emit reauthenticateForFacebook();
        this->connect(this, SIGNAL(signedInToFacebook(QString)), this, SLOT(postToFacebook()));
    }
    else {
        if ((statusCode == 200) || (statusCode == 201)) {
            emit alert(tr("Video shared to facebook"));
        }
        else {
            QString statusText = reply->attribute(QNetworkRequest::HttpReasonPhraseAttribute).toString();
            emit error(statusText);
        }
    }

    reply->deleteLater();
}

void Sharing::postToTwitter(const QVariantMap &post) {
    m_post = post;

    if (this->userSignedInToTwitter()) {
        this->postToTwitter();
    }
    else {
        emit reauthenticateForTwitter();
        this->connect(this, SIGNAL(signedInToTwitter(QString,QString)), this, SLOT(postToTwitter()));
    }
}

void Sharing::postToTwitter() {
    QUrl url("http://api.twitter.com/1.1/statuses/update.json");
    QMap<QString, QString> params;
    params.insert("status", m_post.value("body").toString());
    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", QString("cuteTube/%1 (Qt)").arg(Utils::versionNumberString()).toUtf8());
    request.setHeader(QNetworkRequest::ContentTypeHeader, "application/x-www-form-urlencoded");
    request.setRawHeader("Authorization", this->getTwitterOAuthHeader("POST", url, params));
    QNetworkReply *reply = this->networkAccessManager()->post(request, "status=" + m_post.value("body").toString().toUtf8().toPercentEncoding());
    this->connect(reply, SIGNAL(finished()), this, SLOT(twitterPostFinished()));
    this->disconnect(this, SIGNAL(signedInToTwitter(QString,QString)), this, SLOT(postToTwitter()));
}

void Sharing::twitterPostFinished() {
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    if (statusCode == 401) {
        emit reauthenticateForTwitter();
        this->connect(this, SIGNAL(signedInToTwitter(QString,QString)), this, SLOT(postToTwitter()));
    }
    else {
        if ((statusCode == 200) || (statusCode == 201)) {
            emit alert(tr("Video shared to twitter"));
        }
        else {
            QString statusText = reply->attribute(QNetworkRequest::HttpReasonPhraseAttribute).toString();
            emit error(statusText);
        }        
    }

    reply->deleteLater();
}

void Sharing::shareViaEmail(const QString &title, const QUrl &url) {
    QDesktopServices::openUrl(QUrl(QString("mailto:?subject=%1&body=%2").arg(title).arg(url.toString())));
}

void Sharing::copyToClipboard(const QUrl &url) {
    bool monitor = Settings::instance()->monitorClipboard();
    Settings::instance()->setMonitorClipboard(false);
#if QT_VERSION >= 0x050000
    QGuiApplication::clipboard()->setText(url.toString());
#else
    QApplication::clipboard()->setText(url.toString());
#endif
    emit alert(tr("Copied to clipboard"));

    if (monitor) {
        Settings::instance()->setMonitorClipboard(true);
    }
}
#endif
void Sharing::getVideos(bool forceRefresh) {
    if ((forceRefresh) || (this->videoRefreshIsDue())) {
        if (this->userSignedInToFacebook()) {
            this->getVideosFromFacebook();
        }
        else if (userSignedInToTwitter()) {
            this->getVideosFromTwitter();
        }
    }
    else if (this->userSignedInToFacebook() && (this->facebookTimeStamp() == 0)) {
        this->getVideosFromFacebook();
    }
    else if (this->userSignedInToTwitter() && (this->lastTweet().isEmpty())) {
        this->getVideosFromTwitter();
    }
}

void Sharing::getVideosFromFacebook() {
    this->setBusy(true);
    QUrl url("https://graph.facebook.com/me/feed");
#if QT_VERSION >= 0x050000
    QUrlQuery query;

    if (this->facebookTimeStamp() > 0) {
        query.addQueryItem("since", QString::number(this->facebookTimeStamp()));
    }

    query.addQueryItem("limit", "100");
    query.addQueryItem("access_token", this->facebookToken());
    url.setQuery(query);
#else
    if (this->facebookTimeStamp() > 0) {
        url.addQueryItem("since", QString::number(this->facebookTimeStamp()));
    }

    url.addQueryItem("limit", "100");
    url.addQueryItem("access_token", this->facebookToken());
#endif
    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", QString("cuteTube/%1 (Qt)").arg(Utils::versionNumberString()).toUtf8());
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(checkFacebookVideos()));
    this->disconnect(this, SIGNAL(signedInToFacebook(QString)), this, SLOT(getVideosFromFacebook()));
}

void Sharing::checkFacebookVideos() {
    this->setBusy(false);
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(this->sender());
    if (!reply) {
        emit error("Network error");
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    if (statusCode == 400) {
        emit reauthenticateForFacebook();
        this->connect(this, SIGNAL(signedInToFacebook(QString)), this, SLOT(getVideosFromFacebook()));
    }
    else {
        QString response(reply->readAll());
        bool ok;
        QVariantMap result = Json::parse(response, ok).toMap();

        if (!ok) {
            emit error("Error parsing server response");
        }
        else {
            this->setFacebookTimeStamp(QDateTime::currentMSecsSinceEpoch() / 1000);
            QVariantList results = result.value("data").toList();
            QStringList youtubeIds;
            QStringList dailymotionIds;
            QStringList vimeoIds;

            for (int i = 0; i < results.size(); i++) {
                QString link = results.at(i).toMap().value("link").toString();

                if (!link.isEmpty()) {
                    if (link.contains(YOUTUBE_SHARING_RE)) {
                        youtubeIds.append(link.section(QRegExp("v=|/"), -1).section(QRegExp("&|\\?"), 0, 0));
                    }
                    else if (link.contains(DAILYMOTION_SHARING_RE)) {
                        dailymotionIds.append(link.section('/', -1).left(6));
                    }
                    else if (link.contains(VIMEO_SHARING_RE)) {
                        vimeoIds.append(link.section('/', -1));
                    }
                }
            }

            if (!youtubeIds.isEmpty()) {
                youtubeIds.removeDuplicates();
                emit gotYouTubeIds(youtubeIds);
            }
            if (!dailymotionIds.isEmpty()) {
                dailymotionIds.removeDuplicates();
                emit gotDailymotionIds(dailymotionIds);
            }
            if (!vimeoIds.isEmpty()) {
                vimeoIds.removeDuplicates();
                emit gotVimeoIds(vimeoIds);
            }
        }

        if (this->userSignedInToTwitter()) {
            this->getVideosFromTwitter();
        }
        else {
            this->resetVideoRefreshTime();
        }        
    }

    reply->deleteLater();
}

void Sharing::getVideosFromTwitter() {
    this->setBusy(true);
    QUrl url("http://api.twitter.com/1.1/statuses/home_timeline.json");
#if QT_VERSION >= 0x050000
    QUrlQuery query;
    query.addQueryItem("count", "200");
    query.addQueryItem("include_entities", "true");
    query.addQueryItem("trim_user", "true");
    query.addQueryItem("include_rts", "false");

    if (!this->lastTweet().isEmpty()) {
        query.addQueryItem("since_id", this->lastTweet());
    }

    url.setQuery(query);
#else
    url.addQueryItem("count", "200");
    url.addQueryItem("include_entities", "true");
    url.addQueryItem("trim_user", "true");
    url.addQueryItem("include_rts", "false");

    if (!this->lastTweet().isEmpty()) {
        url.addQueryItem("since_id", this->lastTweet());
    }
#endif
    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", QString("cuteTube/%1 (Qt)").arg(Utils::versionNumberString()).toUtf8());
    request.setRawHeader("Authorization", this->getTwitterOAuthHeader("GET", url, QMap<QString, QString>()));
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(checkTwitterVideos()));
    this->disconnect(this, SIGNAL(signedInToTwitter(QString,QString)), this, SLOT(getVideosFromTwitter()));
}

void Sharing::checkTwitterVideos() {
    this->setBusy(false);
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    if (statusCode == 401) {
        emit reauthenticateForTwitter();
        this->connect(this, SIGNAL(signedInToTwitter(QString,QString)), this, SLOT(getVideosFromTwitter()));
    }
    else {
        QString response(reply->readAll());
        bool ok;
        QVariantList results = Json::parse(response, ok).toList();

        if (!ok) {
            emit error("Error parsing server response");
        }
        else {
            QStringList youtubeIds;
            QStringList dailymotionIds;
            QStringList vimeoIds;

            if (!results.isEmpty()) {
                this->setLastTweet(results.first().toMap().value("id_str").toString());
            }

            while (!results.isEmpty()) {
                QVariantMap tweet = results.takeFirst().toMap();
                QVariantList urls = tweet.value("entities").toMap().value("urls").toList();

                while (!urls.isEmpty()) {
                    QString url = urls.takeFirst().toMap().value("expanded_url").toString();

                    if (!url.isEmpty()) {
                        if (url.contains(YOUTUBE_SHARING_RE)) {
                            youtubeIds.append(url.section(QRegExp("v=|/"), -1).section(QRegExp("&|\\?"), 0, 0));
                        }
                        else if (url.contains(DAILYMOTION_SHARING_RE)) {
                            dailymotionIds.append(url.section('/', -1).section('_', 0, 0));
                        }
                        else if (url.contains(VIMEO_SHARING_RE)) {
                            vimeoIds.append(url.section('/', -1));
                        }
                    }
                }
            }

            if (!youtubeIds.isEmpty()) {
                youtubeIds.removeDuplicates();
                emit gotYouTubeIds(youtubeIds);
            }
            if (!dailymotionIds.isEmpty()) {
                dailymotionIds.removeDuplicates();
                emit gotDailymotionIds(dailymotionIds);
            }
            if (!vimeoIds.isEmpty()) {
                vimeoIds.removeDuplicates();
                emit gotVimeoIds(vimeoIds);
            }
        }
    }

    this->resetVideoRefreshTime();
    reply->deleteLater();
}

void Sharing::addVideosToCache(QList< QSharedPointer<VideoItem> > videos) {
    while (!videos.isEmpty()) {
        m_videoCache->insert(0, videos.takeFirst());
        emit videoAddedToCache(0);
    }
}
