#include "dailymotionmetadatadialog.h"
#include "../base/dailymotion.h"
#include "dailymotionuploadcategoryselector.h"
#include <QLineEdit>
#include <QTextEdit>
#include <QPushButton>
#include <QCheckBox>
#include <QDialogButtonBox>
#include <QScrollArea>
#include <QGridLayout>
#include <QLabel>

DailymotionMetadataDialog::DailymotionMetadataDialog(QSharedPointer<VideoItem> video, QWidget *parent) :
    AbstractDialog(parent),
    m_video(video),
    m_titleEdit(new QLineEdit(this)),
    m_descriptionEdit(new QTextEdit(this)),
    m_tagsEdit(new QLineEdit(this)),
    m_categorySelector(new DailymotionUploadCategorySelector(tr("Channel"), this)),
    m_commentsCheckbox(new QCheckBox(tr("Allow comments?"), this)),
    m_privateCheckbox(new QCheckBox(tr("Private?"), this)),
    m_buttonBox(new QDialogButtonBox(Qt::Vertical, this)),
    m_doneButton(m_buttonBox->addButton(tr("Done"), QDialogButtonBox::AcceptRole)),
    m_scrollArea(new QScrollArea(this)),
    m_grid(new QGridLayout(this))
{
    this->setWindowTitle(tr("Edit metadata"));

    QLabel *titleLabel = new QLabel(tr("Title"), this);
    QLabel *descriptionLabel = new QLabel(tr("Description"), this);
    QLabel *tagsLabel = new QLabel(tr("Tags"), this);
    QWidget *scrollWidget = new QWidget(m_scrollArea);
    QVBoxLayout *vbox = new QVBoxLayout(scrollWidget);

    vbox->addWidget(titleLabel);
    vbox->addWidget(m_titleEdit);
    vbox->addWidget(descriptionLabel);
    vbox->addWidget(m_descriptionEdit);
    vbox->addWidget(tagsLabel);
    vbox->addWidget(m_tagsEdit);
    vbox->addWidget(m_categorySelector);
    vbox->addWidget(m_commentsCheckbox);
    vbox->addWidget(m_privateCheckbox);

    m_scrollArea->setWidgetResizable(true);
    m_scrollArea->setWidget(scrollWidget);
    m_doneButton->setEnabled(false);
    m_descriptionEdit->setAcceptRichText(false);

    this->connect(m_titleEdit, SIGNAL(textChanged(QString)), this, SLOT(onTitleTextChanged(QString)));
    this->connect(m_buttonBox, SIGNAL(accepted()), this, SLOT(submitMetadata()));
    this->connect(Dailymotion::instance(), SIGNAL(gotVideoMetadata(QVariantMap)), this, SLOT(setMetadata(QVariantMap)));

    Dailymotion::instance()->getVideoMetadata(m_video.data()->videoId());

    this->toggleBusy(true);
    this->onOrientationChanged();
}

void DailymotionMetadataDialog::setLandscapeLayout() {
    this->setFixedHeight(340);
    m_buttonBox->setSizePolicy(QSizePolicy::Fixed, QSizePolicy::Fixed);

    if (m_grid->isEmpty()) {
        m_grid->addWidget(m_scrollArea, 0, 0);
        m_grid->addWidget(m_buttonBox, 0, 1, Qt::AlignBottom);
    }
    else {
        m_grid->removeWidget(m_buttonBox);
        m_grid->addWidget(m_buttonBox, 0, 1, Qt::AlignBottom);
    }
}

void DailymotionMetadataDialog::setPortraitLayout() {
    this->setFixedHeight(600);
    m_buttonBox->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Fixed);

    if (m_grid->isEmpty()) {
        m_grid->addWidget(m_scrollArea, 0, 0);
        m_grid->addWidget(m_buttonBox, 1, 0, Qt::AlignBottom);
    }
    else {
        m_grid->removeWidget(m_buttonBox);
        m_grid->addWidget(m_buttonBox, 1, 0, Qt::AlignBottom);
    }
}

void DailymotionMetadataDialog::onTitleTextChanged(const QString &text) {
    m_doneButton->setEnabled(!text.isEmpty());
}

void DailymotionMetadataDialog::setMetadata(const QVariantMap &metadata) {
    this->toggleBusy(false);

    m_titleEdit->setText(metadata.value("title").toString());
    m_descriptionEdit->setText(metadata.value("description").toString());
    m_tagsEdit->setText(metadata.value("tags").toString());
    m_categorySelector->setValue(metadata.value("category").toString());
    m_commentsCheckbox->setChecked(metadata.value("commentsPermission").toBool());
    m_privateCheckbox->setChecked(metadata.value("isPrivate").toBool());

    this->disconnect(Dailymotion::instance(), SIGNAL(gotVideoMetadata(QVariantMap)), this, SLOT(setMetadata(QVariantMap)));
}

void DailymotionMetadataDialog::submitMetadata() {
    QVariantMap metadata;
    metadata.insert("videoId", m_video.data()->videoId());
    metadata.insert("title", m_titleEdit->text());
    metadata.insert("description", m_descriptionEdit->toPlainText());
    metadata.insert("tags", m_tagsEdit->text());
    metadata.insert("category", m_categorySelector->currentValue());
    metadata.insert("commentsPermission", m_commentsCheckbox->isChecked() ? "true" : "false");
    metadata.insert("isPrivate", m_privateCheckbox->isChecked());

    Dailymotion::instance()->updateVideoMetadata(metadata);
    this->accept();
}
