#include "groupitem.h"
#include "utils.h"
#include "dailymotion.h"
#include "vimeo.h"

GroupItem::GroupItem(QObject *parent) :
    QObject(parent),
    m_videoCount(0),
    m_member(false),
    m_service(Services::NoService)
{
}

GroupItem::GroupItem(GroupItem *group, QObject *parent) :
    QObject(parent),
    m_id(group->id()),
    m_title(group->title()),
    m_artist(group->artist()),
    m_artistId(group->artistId()),
    m_artistAvatarUrl(group->artistAvatarUrl()),
    m_description(group->description()),
    m_date(group->date()),
    m_thumbnailUrl(group->thumbnailUrl()),
    m_videoCount(group->videoCount()),
    m_member(group->member()),
    m_service(group->service())
{
    switch (this->service()) {
    case Services::Dailymotion:
        this->connect(Dailymotion::instance(), SIGNAL(groupMembershipChanged(QString,bool)), this, SLOT(onMembershipChanged(QString,bool)));
        break;
    case Services::Vimeo:
        this->connect(Vimeo::instance(), SIGNAL(groupMembershipChanged(QString,bool)), this, SLOT(onMembershipChanged(QString,bool)));
        break;
    default:
        break;
    }
}

void GroupItem::loadGroup(GroupItem *group) {
    this->setId(group->id());
    this->setTitle(group->title());
    this->setArtist(group->artist());
    this->setArtistId(group->artistId());
    this->setArtistAvatarUrl(group->artistAvatarUrl());
    this->setDescription(group->description());
    this->setDate(group->date());
    this->setThumbnailUrl(group->thumbnailUrl());
    this->setVideoCount(group->videoCount());
    this->setMember(group->member());
    this->setService(group->service());

    switch (this->service()) {
    case Services::Dailymotion:
        this->connect(Dailymotion::instance(), SIGNAL(groupMembershipChanged(QString,bool)), this, SLOT(onMembershipChanged(QString,bool)));
        break;
    case Services::Vimeo:
        this->connect(Vimeo::instance(), SIGNAL(groupMembershipChanged(QString,bool)), this, SLOT(onMembershipChanged(QString,bool)));
        break;
    default:
        break;
    }
}

void GroupItem::loadDailymotionGroup(const QVariantMap &group, bool member) {
    this->setMember(member);
    this->setId(group.value("id").toString());
    this->setTitle(group.value("name").toString());
    this->setArtist(group.value("owner.username").toString());
    this->setArtistId(group.value("owner.id").toString());
    this->setArtistAvatarUrl(group.value("owner.avatar_medium_url").toUrl());
    this->setDescription(group.value("description").toString());
    this->setDate(Utils::dateFromSecs(group.value("created_time").toLongLong()));
    this->setService(Services::Dailymotion);
    this->connect(Dailymotion::instance(), SIGNAL(groupMembershipChanged(QString,bool)), this, SLOT(onMembershipChanged(QString,bool)));
}

void GroupItem::loadVimeoGroup(const QDomElement &group) {
    this->setMember(group.attribute("has_joined").toInt());
    this->setId(group.attribute("id"));
    this->setTitle(group.firstChildElement("name").text());
    this->setDescription(group.firstChildElement("description").text());
    this->setDate(Utils::localDateTimeFromString(group.firstChildElement("created_on").text()));
    this->setVideoCount(group.firstChildElement("total_videos").text().toInt());
    this->setThumbnailUrl(QUrl(group.firstChildElement("thumbnail_url").text()));

    QDomElement artistElement = group.firstChildElement("creator").toElement();
    this->setArtist(artistElement.attribute("display_name"));
    this->setArtistId(artistElement.attribute("id"));

    QDomNodeList avatars = artistElement.firstChildElement("portrait").childNodes();

    if (avatars.size() > 1) {
        this->setArtistAvatarUrl(QUrl(avatars.at(1).toElement().text()));
    }

    this->setService(Services::Vimeo);
    this->connect(Vimeo::instance(), SIGNAL(groupMembershipChanged(QString,bool)), this, SLOT(onMembershipChanged(QString,bool)));
}

void GroupItem::onMembershipChanged(const QString &groupId, bool member) {
    if (groupId == this->id()) {
        this->setMember(member);
    }
}
