#include "transferlistwindow.h"
#include "transferlistdelegate.h"
#include "transferprioritydialog.h"
#include "transferdetailsdialog.h"
#include "listview.h"
#include "transferlistmodel.h"
#include <QLabel>
#include <QMenu>
#include <QAction>
#include <QMaemo5Style>

TransferListWindow::TransferListWindow(QWidget *parent) :
    StackedWindow(parent),
    m_model(new TransferListModel(this)),
    m_listView(new ListView(this)),
    m_noResultsLabel(new QLabel(QString("<p style='color:%1;font-size:40px;'><center><b>%2</b></center></p>").arg(QMaemo5Style::standardColor("SecondaryTextColor").name()).arg(tr("No transfers queued")), this)),
    m_contextMenu(new QMenu(this)),
    m_saveAsAudioAction(m_contextMenu->addAction(tr("Save as audio"), this, SLOT(setTransferSaveAsAudio()))),
    m_priorityAction(m_contextMenu->addAction(tr("Set priority"), this, SLOT(showPriorityDialog()))),
    m_pauseResumeAction(m_contextMenu->addAction(tr("Pause"), this, SLOT(pauseOrResumeTransfer()))),
    m_cancelAction(m_contextMenu->addAction(tr("Cancel"), this, SLOT(cancelTransfer())))
{
    this->setWindowTitle(tr("Transfers"));
    this->setCentralWidget(m_listView);

    m_listView->setModel(m_model);
    m_listView->setItemDelegate(new TransferListDelegate(m_listView));
    m_saveAsAudioAction->setCheckable(true);

    this->connect(m_listView, SIGNAL(clicked(QModelIndex)), this, SLOT(onItemClicked(QModelIndex)));
    this->connect(m_listView, SIGNAL(customContextMenuRequested(QPoint)), this, SLOT(showContextMenu(QPoint)));
    this->connect(m_model, SIGNAL(countChanged(int)), this, SLOT(onCountChanged(int)));

    this->onCountChanged(m_model->rowCount());
    this->onOrientationChanged();
}

TransferListWindow::~TransferListWindow() {}

void TransferListWindow::setLandscapeLayout() {
    m_noResultsLabel->setFixedSize(800, 400);
}

void TransferListWindow::setPortraitLayout() {
    m_noResultsLabel->setFixedSize(480, 700);
}

void TransferListWindow::onItemClicked(const QModelIndex &index) {
    TransferDetailsDialog *dialog = new TransferDetailsDialog(m_model->get(index.row()), this);
    dialog->open();
}

void TransferListWindow::showContextMenu(const QPoint &pos) {
    QModelIndex index = m_listView->currentIndex();

    if (!index.isValid()) {
        return;
    }

    m_saveAsAudioAction->setEnabled(index.data(TransferListModel::ConvertibleToAudioRole).toBool());
    m_saveAsAudioAction->setChecked((m_saveAsAudioAction->isEnabled()) && (index.data(TransferListModel::SaveAsAudioRole).toBool()));

    Transfers::TransferStatus status = static_cast<Transfers::TransferStatus>(index.data(TransferListModel::StatusRole).toInt());

    switch (status) {
    case Transfers::Paused:
        m_pauseResumeAction->setText(tr("Resume"));
        break;
    case Transfers::Failed:
        m_pauseResumeAction->setText(tr("Resume"));
        break;
    default:
        m_pauseResumeAction->setText(tr("Pause"));
    }

    m_contextMenu->popup(pos, m_saveAsAudioAction);
}

void TransferListWindow::setTransferSaveAsAudio() {
    m_model->setData(m_listView->currentIndex(), m_saveAsAudioAction->isChecked(), TransferListModel::SaveAsAudioRole);
}

void TransferListWindow::showPriorityDialog() {
    TransferPriorityDialog *dialog = new TransferPriorityDialog(this);
    dialog->setValue(static_cast<Transfers::TransferPriority>(m_listView->currentIndex().data(TransferListModel::PriorityRole).toInt()));
    dialog->show();

    this->connect(dialog, SIGNAL(valueChanged(Transfers::TransferPriority)), this, SLOT(setTransferPriority(Transfers::TransferPriority)));
}

void TransferListWindow::setTransferPriority(Transfers::TransferPriority priority) {
    m_model->setData(m_listView->currentIndex(), priority, TransferListModel::PriorityRole);
}

void TransferListWindow::pauseOrResumeTransfer() {
    QModelIndex index = m_listView->currentIndex();

    if (!index.isValid()) {
        return;
    }

    Transfers::TransferStatus status = static_cast<Transfers::TransferStatus>(index.data(TransferListModel::StatusRole).toInt());

    switch (status) {
    case Transfers::Paused:
        m_model->setData(index, Transfers::Queued, TransferListModel::StatusRole);
        break;
    case Transfers::Failed:
        m_model->setData(index, Transfers::Queued, TransferListModel::StatusRole);
        break;
    default:
        m_model->setData(index, Transfers::Paused, TransferListModel::StatusRole);
    }
}

void TransferListWindow::cancelTransfer() {
    m_model->setData(m_listView->currentIndex(), Transfers::Cancelled, TransferListModel::StatusRole);
}

void TransferListWindow::onCountChanged(int count) {
    m_noResultsLabel->setVisible(count == 0);
}
