#include "socialvideolistwindow.h"
#include "youtubevideoinfowindow.h"
#include "dailymotionvideoinfowindow.h"
#include "vimeovideoinfowindow.h"
#include "videolistdelegate.h"
#include "socialvideoselectionwindow.h"
#include "videoplaybackwindow.h"
#include "../base/sharing.h"
#include "../base/youtube.h"
#include "../base/dailymotion.h"
#include "../base/vimeo.h"
#include "../base/transfermanager.h"
#include "../base/playbacklistmodel.h"
#include "../base/socialvideolistmodel.h"
#include "../base/settings.h"
#include "../base/urlgrabber.h"
#include "listview.h"
#include <QAction>
#include <QLabel>
#include <QMenuBar>
#include <QMaemo5Style>

SocialVideoListWindow::SocialVideoListWindow(QWidget *parent) :
    StackedWindow(parent),
    m_listView(new ListView(this)),
    m_model(new SocialVideoListModel(this)),
    m_refreshAction(this->menuBar()->addAction(tr("Refresh"), this, SLOT(refresh()))),
    m_addToQueueAction(this->menuBar()->addAction(tr("Add to playback queue"), this, SLOT(queueVideos()))),
    m_videoDownloadAction(this->menuBar()->addAction(tr("Download"), this, SLOT(addVideoDownloads()))),
    m_audioDownloadAction(this->menuBar()->addAction(tr("Download (audio)"), this, SLOT(addAudioDownloads()))),
    m_contextMenu(new QMenu(this)),
    m_addToQueueContextAction(m_contextMenu->addAction(tr("Add to playback queue"), this, SLOT(queueVideo()))),
    m_videoDownloadContextAction(m_contextMenu->addAction(tr("Download"), this, SLOT(addVideoDownload()))),
    m_audioDownloadContextAction(m_contextMenu->addAction(tr("Download (audio)"), this, SLOT(addAudioDownload()))),
    m_noResultsLabel(new QLabel(QString("<p style='color:%1;font-size:40px;'><center><b>%2</b></center></p>").arg(QMaemo5Style::standardColor("SecondaryTextColor").name()).arg(tr("No videos found")), this))
{
    this->setWindowTitle(tr("Social"));
    this->setCentralWidget(m_listView);

    VideoListDelegate *delegate = new VideoListDelegate(m_listView);

    m_listView->setModel(m_model);
    m_listView->setItemDelegate(delegate);
    m_noResultsLabel->hide();

    this->connect(m_model, SIGNAL(countChanged(int)), this, SLOT(onCountChanged(int)));
    this->connect(delegate, SIGNAL(clicked(QModelIndex)), this, SLOT(onItemClicked(QModelIndex)));
    this->connect(delegate, SIGNAL(play(QModelIndex)), this, SLOT(playVideo(QModelIndex)));
    this->connect(m_listView, SIGNAL(customContextMenuRequested(QPoint)), this, SLOT(showContextMenu(QPoint)));
    this->connect(Sharing::instance(), SIGNAL(videosAdded()), m_listView, SLOT(positionAtBeginning()));
    this->connect(YouTube::instance(), SIGNAL(busyChanged(bool)), this, SLOT(onLoadingChanged()));
    this->connect(Dailymotion::instance(), SIGNAL(busyChanged(bool)), this, SLOT(onLoadingChanged()));
    this->connect(Vimeo::instance(), SIGNAL(busyChanged(bool)), this, SLOT(onLoadingChanged()));
    this->connect(Sharing::instance(), SIGNAL(busyChanged()), this, SLOT(onLoadingChanged()));

    this->onOrientationChanged();
}

void SocialVideoListWindow::setLandscapeLayout() {
    m_noResultsLabel->setFixedSize(800, 400);
}

void SocialVideoListWindow::setPortraitLayout() {
    m_noResultsLabel->setFixedSize(480, 700);
}

void SocialVideoListWindow::onLoadingChanged() {
    bool busy = (YouTube::instance()->busy()) || (Dailymotion::instance()->busy()) || (Vimeo::instance()->busy()) || (Sharing::instance()->busy());
    this->toggleBusy(busy);
    m_noResultsLabel->setVisible((!busy) && (m_model->rowCount() == 0));
}

void SocialVideoListWindow::onCountChanged(int count) {
    m_noResultsLabel->setVisible(count == 0);
}

void SocialVideoListWindow::refresh() {
    Sharing::instance()->getVideos(true);
}

void SocialVideoListWindow::addVideoDownloads() {
    SocialVideoSelectionWindow *window = new SocialVideoSelectionWindow(m_model, tr("Download"), SLOT(addVideoDownloads()), this);
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void SocialVideoListWindow::addAudioDownloads() {
    SocialVideoSelectionWindow *window = new SocialVideoSelectionWindow(m_model, tr("Download"), SLOT(addAudioDownloads()), this);
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void SocialVideoListWindow::queueVideos() {
    SocialVideoSelectionWindow *window = new SocialVideoSelectionWindow(m_model, tr("Add to queue"), SLOT(queueVideos()), this);
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void SocialVideoListWindow::showContextMenu(const QPoint &pos) {
    m_contextMenu->popup(pos, m_addToQueueContextAction);
}

void SocialVideoListWindow::addVideoDownload() {
    TransferManager::instance()->addDownloadTransfer(m_model->get(m_listView->currentIndex().row()));
}

void SocialVideoListWindow::addAudioDownload() {
    TransferManager::instance()->addDownloadTransfer(m_model->get(m_listView->currentIndex().row()), true);
}

void SocialVideoListWindow::queueVideo() {
    PlaybackListModel::playbackQueue()->addVideo(m_model->get(m_listView->currentIndex().row()));
}

void SocialVideoListWindow::onItemClicked(const QModelIndex &index) {
    if (index.isValid()) {
        this->showVideoInfo(m_model->get(index.row()));
    }
}

void SocialVideoListWindow::showVideoInfo(QSharedPointer<VideoItem> video) {
    switch (video.data()->service()) {
    case Services::YouTube:
    {
        YouTubeVideoInfoWindow *youtubeWindow = new YouTubeVideoInfoWindow(video, this);
        youtubeWindow->show();
    }
        return;
    case Services::Dailymotion:
    {
        DailymotionVideoInfoWindow *dailymotionWindow = new DailymotionVideoInfoWindow(video, this);
        dailymotionWindow->show();
    }
        return;
    case Services::Vimeo:
    {
        VimeoVideoInfoWindow *vimeoWindow = new VimeoVideoInfoWindow(video, this);
        vimeoWindow->show();
    }
        return;
    default:
        qWarning() << "SocialVideoListWindow::showVideoInfo(): Invalid service.";
        return;
    }
}

void SocialVideoListWindow::playVideo(const QModelIndex &index) {
    if (Settings::instance()->mediaPlayer() == "cutetube") {
        VideoPlaybackWindow *window = new VideoPlaybackWindow(QList< QSharedPointer<VideoItem> >() << m_model->get(index.row()), this);
        window->show();
    }
    else {
        UrlGrabber::instance()->getYouTubeVideoUrl(index.data(AbstractVideoListModel::VideoIdRole).toString());
    }
}
