#include "utils.h"
#include <QDateTime>
#include <QFileInfo>
#include <QFile>

QString Utils::durationFromSecs(int secs) {
    QTime time;
    QString format = secs >= 3600 ? "hh:mm:ss" : "mm:ss";

    return time.addSecs(secs).toString(format);
}

QString Utils::durationFromMSecs(int msecs) {
    QTime time;
    QString format = msecs >= 3600000 ? "hh:mm:ss" : "mm:ss";

    return time.addMSecs(msecs).toString(format);
}

QString Utils::fileSizeFromPath(const QString &filePath) {
    QFileInfo file(filePath);
    return Utils::fileSizeFromBytes(file.size());
}

QString Utils::fileSizeFromBytes(double bytes) {
    double kb = 1024;
    double mb = kb * 1024;
    double gb = mb * 1024;

    QString size;

    if (bytes > gb) {
        size = QString::number(bytes / gb, 'f', 2) + "GB";
    }
    else if (bytes > mb) {
        size = QString::number(bytes / mb, 'f', 2) + "MB";
    }
    else if (bytes > kb){
        size = QString::number(bytes / kb, 'f', 2) + "KB";
    }
    else {
        size = QString::number(bytes) + "B";
    }

    return size;
}

QString Utils::dateFromSecs(qint64 secs, bool showTime) {
    return Utils::dateFromMSecs(secs * 1000, showTime);
}

QString Utils::dateFromMSecs(qint64 msecs, bool showTime) {
    QString date;

    if (showTime) {
        date = QDateTime::fromMSecsSinceEpoch(msecs).toString("dd/MM/yyyy | HH:mm");
    }
    else {
        date = QDateTime::fromMSecsSinceEpoch(msecs).toString("dd/MM/yyyy");
    }

    return date;
}

QString Utils::base64Id(const QString &artist, const QString &title) {
    return QString(artist.toAscii().toBase64() + "-" + title.toAscii().toBase64());
}

QString Utils::localDateTimeFromString(const QString &dateTimeString, Qt::DateFormat format) {
    QDateTime dt = QDateTime::fromString(dateTimeString, format);

    if (!dt.isValid()) {
        dt = QDateTime::currentDateTimeUtc();
    }

    return dt.toLocalTime().toString("dd/MM/yyyy | HH:mm");
}

QString Utils::serviceString(Services::VideoService service) {
    switch (service) {
    case Services::YouTube:
        return QString("YouTube");
    case Services::Dailymotion:
        return QString("Dailymotion");
    case Services::Vimeo:
        return QString("Vimeo");
    default:
        return QString();
    }
}

QString Utils::unescape(const QString &s) {
    int unescapes = 0;
    QByteArray us = s.toUtf8();

    while ((us.contains('%')) && (unescapes < 10)) {
        us = QByteArray::fromPercentEncoding(us);
        unescapes++;
    }

    return QString(us);
}

void Utils::log(const QString &filePath, const QByteArray &message) {
    QFile lf(filePath);

    if (lf.open(QIODevice::Append)) {
        lf.write(QDateTime::currentDateTime().toString().toUtf8() + ": " + message + "\n");
    }

    lf.close();
}

QString Utils::versionNumberString() {
#ifdef SYMBIAN_OS
    return QString("1.8.2");
#else
    return QString("1.7.9");
#endif
}
