#ifndef DAILYMOTIONVIDEOLISTMODEL_H
#define DAILYMOTIONVIDEOLISTMODEL_H

#include "abstractvideolistmodel.h"
#include "videoitem.h"

class DailymotionVideoListModel : public AbstractVideoListModel
{
    Q_OBJECT

    Q_PROPERTY(int count
               READ rowCount
               NOTIFY countChanged)
    Q_PROPERTY(bool loading
               READ loading
               NOTIFY loadingChanged)
    Q_PROPERTY(bool itemsSelected
               READ itemsSelected
               NOTIFY itemsSelectedChanged)
    Q_PROPERTY(QUrl feed
               READ feed
               WRITE setFeed
               NOTIFY feedChanged)
    Q_PROPERTY(bool refreshRequired
               READ refreshRequired
               NOTIFY refreshRequiredChanged)

public:
    explicit DailymotionVideoListModel(QObject *parent = 0);
    inline QUrl feed() const { return m_feed; }
    void setFeed(const QUrl &feed);
    inline QString playlistId() const { return m_playlistId; }
    inline QString groupId() const { return m_groupId; }
    inline bool loading() const { return m_loading; }
    inline bool refreshRequired() const { return m_refresh; }

public slots:
    void clear();
    void reset();
    void getVideos(QUrl feed = QUrl());
    void getMoreVideos();
    void getRelatedVideos(const QString &id);
    void getPlaylistVideos(const QString &id);
    void getGroupVideos(const QString &id);
    void search(const QString &query = QString(),
                int order = Queries::Relevance,
                const QString &language = "all");
    void deleteSelectedVideosFromUploads();
    void addSelectedVideosToFavourites();
    void deleteSelectedVideosFromFavourites();
    void addSelectedVideosToPlaylist(const QString &playlistId);
    void addSelectedVideosToNewPlaylist(const QVariantMap &playlist);
    void deleteSelectedVideosFromPlaylist(const QString &playlistId);
    void downloadSelectedVideos(bool saveAsAudio = false);
    void queueSelectedVideos();
    inline void toggleSelected(int row) { AbstractVideoListModel::toggleSelected(row); emit itemsSelectedChanged(this->itemsSelected()); }
    inline void selectAll() { AbstractVideoListModel::selectAll(); emit itemsSelectedChanged(true); }
    inline void selectNone() { AbstractVideoListModel::selectNone(); emit itemsSelectedChanged(false); }

private slots:
    void addVideos();
    void onDeletedFromUploads(const QString &videoId);
    void onFavouriteChanged(const QString &videoId, bool favourite);
    void onAddedToPlaylist(const QString &videoId, const QString &playlistId);
    void onDeletedFromPlaylist(const QString &videoId, const QString &playlistId);
    inline void onError() { this->setError(true); this->setLoading(false); }
    inline void stopLoading() { this->setLoading(false); }

private:
    inline void setPlaylistId(const QString &id) { m_playlistId = id; }
    inline void setGroupId(const QString &id) { m_groupId = id; }
    inline QString query() const { return m_query; }
    inline void setQuery(const QString &query) { m_query = query; }
    inline int order() const { return m_order; }
    inline void setOrder(int order) { m_order = order; }
    inline QString languageFilter() const { return m_language; }
    inline void setLanguageFilter(const QString &language) { m_language = language; }
    inline void setLoading(bool loading) { m_loading = loading; emit loadingChanged(loading); }
    inline bool moreResults() const { return m_moreResults; }
    inline void setMoreResults(bool more) { m_moreResults = more; }
    inline int offset() const { return m_offset; }
    inline void setOffset(int offset) { m_offset = offset; }
    inline bool error() const { return m_error; }
    inline void setError(bool error) { m_error = error; }
    inline void setRefreshRequired(bool refresh) { m_refresh = refresh; emit refreshRequiredChanged(refresh); }

signals:
    void loadingChanged(bool loading);
    void countChanged(int count);
    void itemsSelectedChanged(bool selected);
    void refreshRequiredChanged(bool refresh);
    void feedChanged(const QUrl &feed);

private:
    QUrl m_feed;
    QString m_playlistId;
    QString m_groupId;
    QString m_query;
    int m_order;
    QString m_language;
    bool m_loading;
    bool m_moreResults;
    bool m_error;
    int m_offset;
    bool m_refresh;
};

#endif // DAILYMOTIONVIDEOLISTMODEL_H
