#include "settings.h"
#include <QDir>
#include <QDesktopServices>
#include <QNetworkProxy>
#include <QAuthenticator>
#if (defined (SYMBIAN_OS)) && (defined (CUTETUBE_PLAYER_ENABLED))
#include <QSystemDeviceInfo>
#endif

Settings* settingsInstance = 0;

Settings::Settings(QObject *parent) :
    QSettings(QString("cuteTube"), QString("cuteTube"), parent),
    m_autoTransfers(true),
    m_maxConcurrentTransfers(1),
    m_safeSearch(false),
    m_searchSite(Services::YouTube),
    m_showSearchSite(true),
    m_searchOrder(Queries::Relevance),
    m_showSearchOrder(true),
    m_searchTime(Queries::AllTime),
    m_showSearchTime(false),
    m_searchDuration(Queries::Any),
    m_showSearchDuration(false),
    m_searchType(Queries::Videos),
    m_showSearchType(true),
    m_languageFilter("all"),
    m_showSearchLanguage(false),
  #if (defined (SYMBIAN_OS) || (defined (MEEGO_EDITION_HARMATTAN)))
    m_theme("dark"),
    m_activeColor("#0881cb"),
    m_activeColorString("color7"),
  #endif
    m_language("en"),
  #ifdef CUTETUBE_PLAYER_ENABLED
    m_mediaplayer("cutetube"),
  #ifndef SYMBIAN_OS
    m_backgroundPlaybackEnabled(false),
  #endif
  #else
    m_mediaplayer("mediaplayer"),
  #endif
    m_orientation(ScreenOrientation::Automatic),
    m_monitorClipboard(false),
    m_proxyType(NetworkProxyType::HttpProxy),
    m_proxyPort(80),
  #ifdef Q_WS_MAEMO_5
    m_eventFeedEnabled(false),
    m_eventFeedInstalled(false),
  #endif
  #ifdef CUTETUBE_PLAYER_ENABLED
    m_cuteTubePlayerEnabled(true)
  #else
    m_cuteTubePlayerEnabled(false)
  #endif
{
    if (!settingsInstance) {
        settingsInstance = this;
    }
}

Settings::~Settings() {
    this->saveSettings();
}

Settings* Settings::instance() {
    return settingsInstance;
}

void Settings::restoreSettings() {
    this->beginGroup("Media");
    this->setYouTubePlaybackFormats(this->value("youtubePlaybackFormats", QVariantList() << Videos::LQ).toList());
    this->setYouTubeDownloadFormats(this->value("youtubeDownloadFormats", QVariantList() << Videos::LQ).toList());
    this->setDailymotionPlaybackFormats(this->value("dailymotionPlaybackFormats", QVariantList() << Videos::Normal).toList());
    this->setDailymotionDownloadFormats(this->value("dailymotionDownloadFormats", QVariantList() << Videos::Normal).toList());
#ifdef Q_WS_MAEMO_5
    this->setVimeoPlaybackFormats(this->value("vimeoPlaybackFormats", QVariantList() << Videos::LQ).toList());
    this->setVimeoDownloadFormats(this->value("vimeoDownloadFormats", QVariantList() << Videos::LQ).toList());
#else
    this->setVimeoPlaybackFormats(this->value("vimeoPlaybackFormats", QVariantList() << Videos::LQ << Videos::Normal).toList());
    this->setVimeoDownloadFormats(this->value("vimeoDownloadFormats", QVariantList() << Videos::LQ << Videos::Normal).toList());
#endif
    this->setUploadAttribute(this->value("uploadAttribute", tr("Uploaded via cuteTube")).toString());
#ifdef SYMBIAN_OS
#ifdef CUTETUBE_PLAYER_ENABLED
    using namespace QtMobility;
    QSystemDeviceInfo info;
    this->setCuteTubePlayerEnabled((info.model() == "700") || (info.model() == "701") || (info.model() == "808"));

    if (this->cuteTubePlayerEnabled()) {
        this->setMediaPlayer(this->value("mediaPlayer", QString("cutetube")).toString());
    }
#endif
#else
#ifdef CUTETUBE_PLAYER_ENABLED
    this->setMediaPlayer(this->value("mediaPlayer", QString("cutetube")).toString());
    this->setBackgroundPlaybackEnabled(this->value("backgroundPlaybackEnabled", false).toBool());
#else
    this->setMediaPlayer(this->value("mediaPlayer", QString("mediaplayer")).toString());
#endif
    this->setMediaPlayerCommand(this->value("mediaPlayerCommand", QString()).toString());
#endif
    this->setLanguageFilter(this->value("languageFilter", QString("all")).toString());
    this->endGroup();

    this->beginGroup("Transfers");
    this->setStartTransfersAutomatically(this->value("startTransfersAutomatically", true).toBool());
    this->setMaximumConcurrentTransfers(this->value("maximumConcurrentTransfers", 1).toInt());
#if (defined (Q_WS_MAEMO_5)) || (defined (MEEGO_EDITION_HARMATTAN))
    this->setDownloadPath(this->value("downloadPath", QString("/home/user/MyDocs/cuteTube/")).toString());
#elif defined (SYMBIAN_OS)
    this->setDownloadPath(this->value("downloadPath", QString("E:/cuteTube/")).toString());
#else
    this->setDownloadPath(this->value("downloadPath", QDesktopServices::storageLocation(QDesktopServices::HomeLocation).append("/cuteTube/")).toString());
#endif
    this->endGroup();

    this->beginGroup("Search");
    this->setSafeSearch(this->value("safeSearch", false).toBool());
    this->setSearchSite(static_cast<Services::VideoService>(this->value("searchSite", Services::YouTube).toInt()));
    this->setShowSearchSite(this->value("showSearchSite", true).toBool());
    this->setSearchOrder(static_cast<Queries::QueryOrder>(this->value("searchOrder", Queries::Relevance).toInt()));
    this->setShowSearchOrder(this->value("showSearchOrder", true).toBool());
    this->setSearchTime(static_cast<Queries::TimeFilter>(this->value("searchTime", Queries::AllTime).toInt()));
    this->setShowSearchTime(this->value("showSearchTime", false).toBool());
    this->setSearchDuration(static_cast<Queries::DurationFilter>(this->value("searchDuration", Queries::Any).toInt()));
    this->setShowSearchDuration(this->value("showSearchDuration", false).toBool());
    this->setSearchType(static_cast<Queries::QueryType>(this->value("searchType", Queries::Videos).toInt()));
    this->setShowSearchType(this->value("showSearchType", true).toBool());
    this->setShowSearchLanguage(this->value("showSearchLanguage", true).toBool());
    this->endGroup();

#ifdef Q_WS_MAEMO_5
    this->beginGroup("Events");
    this->setEventFeedEnabled((this->value("eventFeedEnabled", false).toBool()) && (this->eventFeedInstalled()));
    this->endGroup();
#endif

#if (defined (SYMBIAN_OS)) || (defined (MEEGO_EDITION_HARMATTAN))
    this->beginGroup("Appearance");
#ifdef SYMBIAN_OS
    this->setTheme(this->value("theme", QString("dark")).toString());
#else
    this->setTheme(this->value("theme", QString("dark_plain")).toString());
    this->setActiveColor(this->value("activeColor", QString("#0881cb")).toString());
    this->setActiveColorString(this->value("activeColorString", QString("color7")).toString());
#endif
    this->endGroup();
#endif

    this->beginGroup("System");
    this->setLanguage(this->value("language", QString("en")).toString());
    this->setScreenOrientation(static_cast<ScreenOrientation::Orientation>(this->value("screenOrientation", 0).toInt()));
    this->setMonitorClipboard(this->value("monitorClipboard", false).toBool());
    this->setNetworkProxyType(static_cast<NetworkProxyType::ProxyType>(this->value("networkProxyType", NetworkProxyType::ProxyType(NetworkProxyType::HttpProxy)).toInt()));
    this->setNetworkProxyHostName(this->value("networkProxyHostName", QString()).toString());
    this->setNetworkProxyPort(this->value("networkProxyPort", 80).toUInt());
    this->setNetworkProxyUser(this->value("networkProxyUser", QString()).toString());
    this->setNetworkProxyPassword(QString(QByteArray::fromBase64(this->value("networkProxyPassword", QByteArray()).toByteArray())));
    this->setNetworkProxy();
    this->endGroup();
}

void Settings::saveSettings() {
    this->beginGroup("Media");
    this->setValue("youtubePlaybackFormats", this->toVariantList(m_youtubePlaybackFormats));
    this->setValue("youtubeDownloadFormats", this->toVariantList(m_youtubeDownloadFormats));
    this->setValue("dailymotionPlaybackFormats", this->toVariantList(m_dailymotionPlaybackFormats));
    this->setValue("dailymotionDownloadFormats", this->toVariantList(m_dailymotionDownloadFormats));
    this->setValue("vimeoPlaybackFormats", this->toVariantList(m_vimeoPlaybackFormats));
    this->setValue("vimeoDownloadFormats", this->toVariantList(m_vimeoDownloadFormats));
    this->setValue("uploadAttribute", this->uploadAttribute());
#if (!defined SYMBIAN_OS) || (defined (CUTETUBE_PLAYER_ENABLED))
    this->setValue("mediaPlayer", this->mediaPlayer());
#endif
#ifndef SYMBIAN_OS
#ifdef CUTETUBE_PLAYER_ENABLED
    this->setValue("backgroundPlaybackEnabled", this->backgroundPlaybackEnabled());
#endif
    this->setValue("mediaPlayerCommand", this->mediaPlayerCommand());
#endif
    this->setValue("languageFilter", this->languageFilter());
    this->endGroup();

    this->beginGroup("Transfers");
    this->setValue("startTransfersAutomatically", this->startTransfersAutomatically());
    this->setValue("maximumConcurrentTransfers", this->maximumConcurrentTransfers());
    this->setValue("downloadPath", this->downloadPath());
    this->endGroup();

    this->beginGroup("Search");
    this->setValue("safeSearch", this->safeSearch());
    this->setValue("searchSite", Services::VideoService(this->searchSite()));
    this->setValue("showSearchSite", this->showSearchSite());
    this->setValue("searchOrder", Queries::QueryOrder(this->searchOrder()));
    this->setValue("showSearchOrder", this->showSearchOrder());
    this->setValue("searchTime", Queries::TimeFilter(this->searchTime()));
    this->setValue("showSearchTime", this->showSearchTime());
    this->setValue("searchDuration", Queries::DurationFilter(this->searchDuration()));
    this->setValue("showSearchDuration", this->showSearchDuration());
    this->setValue("searchType", Queries::QueryType(this->searchType()));
    this->setValue("showSearchType", this->showSearchType());
    this->setValue("showSearchLanguage", this->showSearchLanguage());
    this->endGroup();

#ifdef Q_WS_MAEMO_5
    this->beginGroup("Events");
    this->setValue("eventFeedEnabled", (this->eventFeedEnabled()) && (this->eventFeedInstalled()));
    this->endGroup();
#endif

#if (defined (SYMBIAN_OS)) || (defined (MEEGO_EDITION_HARMATTAN))
    this->beginGroup("Appearance");
    this->setValue("theme", this->theme());
#ifndef SYMBIAN_OS
    this->setValue("activeColor", this->activeColor());
    this->setValue("activeColorString", this->activeColorString());
#endif
    this->endGroup();
#endif

    this->beginGroup("System");
    this->setValue("language", this->language());
    this->setValue("screenOrientation", ScreenOrientation::Orientation(this->screenOrientation()));
    this->setValue("monitorClipboard", this->monitorClipboard());
    this->setValue("networkProxyType", NetworkProxyType::ProxyType(this->networkProxyType()));
    this->setValue("networkProxyHostName", this->networkProxyHostName());
    this->setValue("networkProxyPort", this->networkProxyPort());
    this->setValue("networkProxyUser", this->networkProxyUser());
    this->setValue("networkProxyPassword", this->networkProxyPassword().toUtf8().toBase64());
    this->endGroup();
}

QVariantList Settings::toVariantList(const QSet<int> &set) {
    QVariantList variantList;

    foreach (int i, set) {
        variantList.append(i);
    }

    return variantList;
}

void Settings::setYouTubePlaybackFormats(QVariantList formats) {
    while (!formats.isEmpty()) {
        m_youtubePlaybackFormats.insert(formats.takeFirst().toInt());
    }

    m_youtubePlaybackFormats.insert(Videos::LQ);

    emit youtubePlaybackFormatsChanged(m_youtubePlaybackFormats);
    emit enableYouTube1080PlaybackChanged();
    emit enableYouTube720PlaybackChanged();
    emit enableYouTube480PlaybackChanged();
    emit enableYouTube360PlaybackChanged();
}

void Settings::setYouTubeDownloadFormats(QVariantList formats) {
    while (!formats.isEmpty()) {
        m_youtubeDownloadFormats.insert(formats.takeFirst().toInt());
    }

    m_youtubeDownloadFormats.insert(Videos::LQ);

    emit youtubeDownloadFormatsChanged(m_youtubeDownloadFormats);
    emit enableYouTube1080DownloadChanged();
    emit enableYouTube720DownloadChanged();
    emit enableYouTube480DownloadChanged();
    emit enableYouTube360DownloadChanged();
}

void Settings::setDailymotionPlaybackFormats(QVariantList formats) {
    while (!formats.isEmpty()) {
        m_dailymotionPlaybackFormats.insert(formats.takeFirst().toInt());
    }

    m_dailymotionPlaybackFormats.insert(Videos::Normal);

    emit dailymotionPlaybackFormatsChanged(m_dailymotionPlaybackFormats);
    emit enableDailymotion1080PlaybackChanged();
    emit enableDailymotion720PlaybackChanged();
    emit enableDailymotion480PlaybackChanged();
}

void Settings::setDailymotionDownloadFormats(QVariantList formats) {
    while (!formats.isEmpty()) {
        m_dailymotionDownloadFormats.insert(formats.takeFirst().toInt());
    }

    m_dailymotionDownloadFormats.insert(Videos::Normal);

    emit dailymotionDownloadFormatsChanged(m_dailymotionDownloadFormats);
    emit enableDailymotion1080DownloadChanged();
    emit enableDailymotion720DownloadChanged();
    emit enableDailymotion480DownloadChanged();
}

void Settings::setVimeoPlaybackFormats(QVariantList formats) {
    while (!formats.isEmpty()) {
        m_vimeoPlaybackFormats.insert(formats.takeFirst().toInt());
    }

    m_vimeoPlaybackFormats.insert(Videos::LQ);

    emit vimeoPlaybackFormatsChanged(m_vimeoPlaybackFormats);
    emit enableVimeo720PlaybackChanged();
    emit enableVimeo360PlaybackChanged();
}

void Settings::setVimeoDownloadFormats(QVariantList formats) {
    while (!formats.isEmpty()) {
        m_vimeoDownloadFormats.insert(formats.takeFirst().toInt());
    }

    m_vimeoDownloadFormats.insert(Videos::LQ);

    emit vimeoDownloadFormatsChanged(m_vimeoDownloadFormats);
    emit enableVimeo720DownloadChanged();
    emit enableVimeo360DownloadChanged();
}

void Settings::setEnableYouTube1080Playback(bool enable) {
    if (enable) {
        m_youtubePlaybackFormats.insert(Videos::SuperHD);
    }
    else {
        m_youtubePlaybackFormats.remove(Videos::SuperHD);
    }

    emit youtubePlaybackFormatsChanged(m_youtubePlaybackFormats);
    emit enableYouTube1080PlaybackChanged();
}

void Settings::setEnableYouTube720Playback(bool enable) {
    if (enable) {
        m_youtubePlaybackFormats.insert(Videos::HD);
    }
    else {
        m_youtubePlaybackFormats.remove(Videos::HD);
    }

    emit youtubePlaybackFormatsChanged(m_youtubePlaybackFormats);
    emit enableYouTube720PlaybackChanged();
}

void Settings::setEnableYouTube480Playback(bool enable) {
    if (enable) {
        m_youtubePlaybackFormats.insert(Videos::HQ);
    }
    else {
        m_youtubePlaybackFormats.remove(Videos::HQ);
    }

    emit youtubePlaybackFormatsChanged(m_youtubePlaybackFormats);
    emit enableYouTube480PlaybackChanged();
}

void Settings::setEnableYouTube360Playback(bool enable) {
    if (enable) {
        m_youtubePlaybackFormats.insert(Videos::Normal);
    }
    else {
        m_youtubePlaybackFormats.remove(Videos::Normal);
    }

    emit youtubePlaybackFormatsChanged(m_youtubePlaybackFormats);
    emit enableYouTube360PlaybackChanged();
}

void Settings::setEnableYouTube1080Download(bool enable) {
    if (enable) {
        m_youtubeDownloadFormats.insert(Videos::SuperHD);
    }
    else {
        m_youtubeDownloadFormats.remove(Videos::SuperHD);
    }

    emit youtubeDownloadFormatsChanged(m_youtubeDownloadFormats);
    emit enableYouTube1080DownloadChanged();
}

void Settings::setEnableYouTube720Download(bool enable) {
    if (enable) {
        m_youtubeDownloadFormats.insert(Videos::HD);
    }
    else {
        m_youtubeDownloadFormats.remove(Videos::HD);
    }

    emit youtubeDownloadFormatsChanged(m_youtubeDownloadFormats);
    emit enableYouTube720DownloadChanged();
}

void Settings::setEnableYouTube480Download(bool enable) {
    if (enable) {
        m_youtubeDownloadFormats.insert(Videos::HQ);
    }
    else {
        m_youtubeDownloadFormats.remove(Videos::HQ);
    }

    emit youtubeDownloadFormatsChanged(m_youtubeDownloadFormats);
    emit enableYouTube480DownloadChanged();
}

void Settings::setEnableYouTube360Download(bool enable) {
    if (enable) {
        m_youtubeDownloadFormats.insert(Videos::Normal);
    }
    else {
        m_youtubeDownloadFormats.remove(Videos::Normal);
    }

    emit youtubeDownloadFormatsChanged(m_youtubeDownloadFormats);
    emit enableYouTube360DownloadChanged();
}

void Settings::setEnableDailymotion1080Playback(bool enable) {
    if (enable) {
        m_dailymotionPlaybackFormats.insert(Videos::SuperHD);
    }
    else {
        m_dailymotionPlaybackFormats.remove(Videos::SuperHD);
    }

    emit dailymotionPlaybackFormatsChanged(m_dailymotionPlaybackFormats);
    emit enableDailymotion1080PlaybackChanged();
}

void Settings::setEnableDailymotion720Playback(bool enable) {
    if (enable) {
        m_dailymotionPlaybackFormats.insert(Videos::HD);
    }
    else {
        m_dailymotionPlaybackFormats.remove(Videos::HD);
    }

    emit dailymotionPlaybackFormatsChanged(m_dailymotionPlaybackFormats);
    emit enableDailymotion720PlaybackChanged();
}

void Settings::setEnableDailymotion480Playback(bool enable) {
    if (enable) {
        m_dailymotionPlaybackFormats.insert(Videos::HQ);
    }
    else {
        m_dailymotionPlaybackFormats.remove(Videos::HQ);
    }

    emit dailymotionPlaybackFormatsChanged(m_dailymotionPlaybackFormats);
    emit enableDailymotion480PlaybackChanged();
}

void Settings::setEnableDailymotion1080Download(bool enable) {
    if (enable) {
        m_dailymotionDownloadFormats.insert(Videos::SuperHD);
    }
    else {
        m_dailymotionDownloadFormats.remove(Videos::SuperHD);
    }

    emit dailymotionDownloadFormatsChanged(m_dailymotionDownloadFormats);
    emit enableDailymotion1080DownloadChanged();
}

void Settings::setEnableDailymotion720Download(bool enable) {
    if (enable) {
        m_dailymotionDownloadFormats.insert(Videos::HD);
    }
    else {
        m_dailymotionDownloadFormats.remove(Videos::HD);
    }

    emit dailymotionDownloadFormatsChanged(m_dailymotionDownloadFormats);
    emit enableDailymotion720DownloadChanged();
}

void Settings::setEnableDailymotion480Download(bool enable) {
    if (enable) {
        m_dailymotionDownloadFormats.insert(Videos::HQ);
    }
    else {
        m_dailymotionDownloadFormats.remove(Videos::HQ);
    }

    emit dailymotionDownloadFormatsChanged(m_dailymotionDownloadFormats);
    emit enableDailymotion480DownloadChanged();
}

void Settings::setEnableVimeo720Playback(bool enable) {
    if (enable) {
        m_vimeoPlaybackFormats.insert(Videos::HQ);
    }
    else {
        m_vimeoPlaybackFormats.remove(Videos::HQ);
    }

    emit vimeoPlaybackFormatsChanged(m_vimeoPlaybackFormats);
    emit enableVimeo720PlaybackChanged();
}

void Settings::setEnableVimeo360Playback(bool enable) {
    if (enable) {
        m_vimeoPlaybackFormats.insert(Videos::Normal);
    }
    else {
        m_vimeoPlaybackFormats.remove(Videos::Normal);
    }

    emit vimeoPlaybackFormatsChanged(m_vimeoPlaybackFormats);
    emit enableVimeo360PlaybackChanged();
}

void Settings::setEnableVimeo720Download(bool enable) {
    if (enable) {
        m_vimeoDownloadFormats.insert(Videos::HD);
    }
    else {
        m_vimeoDownloadFormats.remove(Videos::HD);
    }

    emit vimeoDownloadFormatsChanged(m_vimeoDownloadFormats);
    emit enableVimeo720DownloadChanged();
}

void Settings::setEnableVimeo360Download(bool enable) {
    if (enable) {
        m_vimeoDownloadFormats.insert(Videos::Normal);
    }
    else {
        m_vimeoDownloadFormats.remove(Videos::Normal);
    }

    emit vimeoDownloadFormatsChanged(m_vimeoDownloadFormats);
    emit enableVimeo360DownloadChanged();

}

void Settings::setStartTransfersAutomatically(bool automatic) {
    if (automatic != this->startTransfersAutomatically()) {
        m_autoTransfers = automatic;
        emit startTransfersAutomaticallyChanged(automatic);
    }
}

void Settings::setMaximumConcurrentTransfers(int maximum) {
    int oldMaximum = this->maximumConcurrentTransfers();

    if ((maximum != oldMaximum) && (maximum > 0) && (maximum <= 3)) {
        m_maxConcurrentTransfers = maximum;
        emit maximumConcurrentTransfersChanged(oldMaximum, maximum);
    }
}

void Settings::setDownloadPath(const QString &path) {
    m_path = path;

    if (!m_path.endsWith("/")) {
        m_path.append("/");
    }

    QDir pathCheck(m_path + ".thumbnails/");

    if (pathCheck.exists(m_path + "thumbnails/")) {
        pathCheck.rename(m_path + "thumbnails/", pathCheck.path());
    }

    pathCheck.mkpath(pathCheck.path());

    emit downloadPathChanged(m_path);
}

void Settings::setSafeSearch(bool safe) {
    if (safe != this->safeSearch()) {
        m_safeSearch = safe;
        emit safeSearchChanged(safe);
    }
}

void Settings::setSearchSite(Services::VideoService site) {
    if (site != this->searchSite()) {
        m_searchSite = site;
        emit searchSiteChanged(site);
    }
}

void Settings::setShowSearchSite(bool show) {
    if (show != this->showSearchSite()) {
        m_showSearchSite = show;
        emit showSearchSiteChanged(show);
    }
}

void Settings::setSearchOrder(Queries::QueryOrder order) {
    if (order != this->searchOrder()) {
        m_searchOrder = order;
        emit searchOrderChanged(order);
    }
}

void Settings::setShowSearchOrder(bool show) {
    if (show != this->showSearchOrder()) {
        m_showSearchOrder = show;
        emit showSearchOrderChanged(show);
    }
}

void Settings::setSearchTime(Queries::TimeFilter time) {
    if (time != this->searchTime()) {
        m_searchTime = time;
        emit searchTimeChanged(time);
    }
}

void Settings::setShowSearchTime(bool show) {
    if (show != this->showSearchTime()) {
        m_showSearchTime = show;
        emit showSearchTimeChanged(show);
    }
}

void Settings::setSearchDuration(Queries::DurationFilter duration) {
    if (duration != this->searchDuration()) {
        m_searchDuration = duration;
        emit searchDurationChanged(duration);
    }
}

void Settings::setShowSearchDuration(bool show) {
    if (show != this->showSearchDuration()) {
        m_showSearchDuration = show;
        emit showSearchDurationChanged(show);
    }
}

void Settings::setSearchType(Queries::QueryType type) {
    if (type != this->searchType()) {
        m_searchType = type;
        emit searchTypeChanged(type);
    }
}

void Settings::setShowSearchType(bool show) {
    if (show != this->showSearchType()) {
        m_showSearchType = show;
        emit showSearchTypeChanged(show);
    }
}

void Settings::setLanguageFilter(const QString &language) {
    if (language != this->languageFilter()) {
        m_languageFilter = language;
        emit languageFilterChanged(language);
    }
}

void Settings::setShowSearchLanguage(bool show) {
    if (show != this->showSearchLanguage()) {
        m_showSearchLanguage = show;
        emit showSearchLanguageChanged(show);
    }
}

#if (defined (SYMBIAN_OS)) || (defined (MEEGO_EDITION_HARMATTAN))
void Settings::setTheme(const QString &theme) {
    if (theme != this->theme()) {
        m_theme = theme;
        emit themeChanged(theme);
    }
}

#ifdef MEEGO_EDITION_HARMATTAN
void Settings::setActiveColor(const QString &color) {
    if (color != this->activeColor()) {
        m_activeColor = color;
        emit activeColorChanged(color);
    }
}

void Settings::setActiveColorString(const QString &colorString) {
    if (colorString != this->activeColorString()) {
        m_activeColorString = colorString;
        emit activeColorStringChanged(colorString);
    }
}
#endif
#endif

void Settings::setLanguage(const QString &lang) {
    if (lang != this->language()) {
        m_language = lang;
        emit languageChanged(lang);
    }
}

void Settings::setUploadAttribute(const QString &attribute) {
    if (attribute != this->uploadAttribute()) {
        m_attribute = attribute;
        emit uploadAttributeChanged(attribute);
    }
}

void Settings::setMediaPlayer(const QString &player) {
    if (player != this->mediaPlayer()) {
        m_mediaplayer = player;
        emit mediaPlayerChanged(player);
    }
}

#ifndef SYMBIAN_OS
void Settings::setMediaPlayerCommand(const QString &command) {
    if (command != this->mediaPlayerCommand()) {
        m_command = command;
        emit mediaPlayerCommandChanged(command);
    }
}

void Settings::setBackgroundPlaybackEnabled(bool enabled) {
    if (enabled != this->backgroundPlaybackEnabled()) {
        m_backgroundPlaybackEnabled = enabled;
        emit backgroundPlaybackEnabledChanged(enabled);
    }
}
#endif

void Settings::setScreenOrientation(ScreenOrientation::Orientation orientation) {
    if (orientation != this->screenOrientation()) {
        m_orientation = orientation;
        emit screenOrientationChanged(orientation);
    }
}

void Settings::setMonitorClipboard(bool monitor) {
    if (monitor != this->monitorClipboard()) {
        m_monitorClipboard = monitor;
        emit monitorClipboardChanged(monitor);
    }
}

QString Settings::networkProxyTypeString() const {
    switch (this->networkProxyType()) {
    case NetworkProxyType::Socks5Proxy:
        return QString("SOCKS5");
    case NetworkProxyType::HttpProxy:
        return QString("HTTP");
    case NetworkProxyType::HttpCachingProxy:
        return QString("HTTP %1").arg(tr("caching"));
    default:
        return QString("HTTP");
    }
}

void Settings::setNetworkProxyType(NetworkProxyType::ProxyType proxyType) {
    if (proxyType != this->networkProxyType()) {
        m_proxyType = proxyType;
        emit networkProxyTypeChanged(proxyType);
        emit networkProxyTypeStringChanged(this->networkProxyTypeString());
    }
}

void Settings::setNetworkProxyHostName(const QString &hostName) {
    if (hostName != this->networkProxyHostName()) {
        m_proxyHost = hostName;
        emit networkProxyHostNameChanged(hostName);
    }
}

void Settings::setNetworkProxyPort(quint16 port) {
    if (port != this->networkProxyPort()) {
        m_proxyPort = port;
        emit networkProxyPortChanged(port);
    }
}

void Settings::setNetworkProxyUser(const QString &user) {
    if (user != this->networkProxyUser()) {
        m_proxyUser = user;
        emit networkProxyUserChanged(user);
    }
}

void Settings::setNetworkProxyPassword(const QString &password) {
    if (password != this->networkProxyPassword()) {
        m_proxyPass = password;
        emit networkProxyPasswordChanged(password);
    }
}

void Settings::setNetworkProxy() {
    QNetworkProxy proxy;

    if (!this->networkProxyHostName().isEmpty()) {
        proxy.setHostName(this->networkProxyHostName());
        proxy.setType(QNetworkProxy::ProxyType(this->networkProxyType()));

        if (this->networkProxyPort() > 0) {
            proxy.setPort(this->networkProxyPort());
        }
        else {
            proxy.setPort(80);
        }

        if ((!this->networkProxyUser().isEmpty()) && (!this->networkProxyPassword().isEmpty())) {
            proxy.setUser(this->networkProxyUser());
            proxy.setPassword(this->networkProxyPassword());
        }
    }

    QNetworkProxy::setApplicationProxy(proxy);
}

void Settings::onNetworkProxyAuthenticationRequested(const QNetworkProxy &proxy, QAuthenticator *authenticator) {
    Q_UNUSED(proxy)

    authenticator->setUser(this->networkProxyUser());
    authenticator->setPassword(this->networkProxyPassword());
}

#ifdef Q_WS_MAEMO_5
void Settings::setEventFeedEnabled(bool enabled) {
    if (enabled != this->eventFeedEnabled()) {
        m_eventFeedEnabled = enabled;
        emit eventFeedEnabledChanged(enabled);
    }
}

bool Settings::eventFeedInstalled() const {
    return QDir().exists("/opt/hildonevents/bin/hildonevents");

}
#endif

void Settings::setCuteTubePlayerEnabled(bool enabled) {
    m_cuteTubePlayerEnabled = enabled;
}
