#include "dailymotioncommentlistmodel.h"
#include "feedurls.h"
#include "json.h"
#include "dailymotion.h"
#include <QNetworkReply>
#include <QNetworkRequest>
#include <QTimer>

using namespace QtJson;

DailymotionCommentListModel::DailymotionCommentListModel(QObject *parent) :
    AbstractCommentListModel(parent),
    m_loading(true),
    m_moreResults(true),
    m_error(false),
    m_offset(1)
{
    this->connect(Dailymotion::instance(), SIGNAL(commentAdded(QSharedPointer<CommentItem>)), this, SLOT(onCommentAdded(QSharedPointer<CommentItem>)));
}

void DailymotionCommentListModel::clear() {
    AbstractCommentListModel::clear();
    this->setLoading(false);
    this->setOffset(1);
    this->setMoreResults(true);
    this->setError(false);
}

void DailymotionCommentListModel::reset() {
    if (!this->loading()) {
        this->clear();
        this->getMoreComments();
    }
}

void DailymotionCommentListModel::setVideoId(const QString &id) {
    if (id != this->videoId()) {
        m_videoId = id;
        this->setFeed(QUrl(QString("https://api.dailymotion.com/video/%1/comments?limit=30").arg(id)));
    }
}

void DailymotionCommentListModel::getComments(const QString &id) {
    if (!id.isEmpty()) {
        this->setVideoId(id);
    }

    this->setLoading(true);
    QUrl feed(this->feed());
    feed.addQueryItem("fields", DAILYMOTION_COMMENT_FIELDS);
    QNetworkReply *reply = Dailymotion::instance()->createReply(feed, this->offset());
    reply->setParent(this);
    this->connect(reply, SIGNAL(finished()), this, SLOT(addComments()));
}

void DailymotionCommentListModel::addComments() {
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        this->setLoading(false);
        this->setError(true);
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    if (statusCode == 401) {
        this->connect(Dailymotion::instance(), SIGNAL(accessTokenRefreshed(QString,QString)), this, SLOT(getComments()));
        this->connect(Dailymotion::instance(), SIGNAL(refreshError()), this, SLOT(onError()));
        Dailymotion::instance()->refreshAccessToken();
    }
    else {
        QString response(reply->readAll());
        bool ok;
        QVariantMap res = Json::parse(response, ok).toMap();

        if (!ok) {
            this->setLoading(false);
            this->setError(true);
        }
        else {
            QVariantList entries = res.value("list").toList();

            for (int i = 0; i < entries.size(); i++) {
                CommentItem *comment = new CommentItem;
                comment->loadDailymotionComment(entries.at(i).toMap());
                QMetaObject::invokeMethod(this, "appendComment", Qt::QueuedConnection, Q_ARG(QSharedPointer<CommentItem>, QSharedPointer<CommentItem>(comment)));
            }

            QTimer::singleShot(1000, this, SLOT(stopLoading()));
            this->setMoreResults(res.value("has_more").toBool());
            this->setOffset(res.value("page").toInt() + 1);
        }

        this->disconnect(Dailymotion::instance(), SIGNAL(accessTokenRefreshed(QString,QString)), this, SLOT(getComments()));
        this->disconnect(Dailymotion::instance(), SIGNAL(refreshError()), this, SLOT(onError()));
    }

    emit countChanged(this->rowCount());

    reply->deleteLater();
}

void DailymotionCommentListModel::getMoreComments() {
    if ((this->moreResults()) && (!this->loading())) {
        this->getComments();
    }
}

void DailymotionCommentListModel::onCommentAdded(QSharedPointer<CommentItem> comment) {
    if ((!comment.isNull()) && (comment.data()->videoId() == this->videoId())) {
        this->insertComment(0, comment);
        emit countChanged(this->rowCount());
    }
}
