#include "transferitem.h"
#include "utils.h"
#include "youtube.h"
#include "settings.h"
#include <QNetworkAccessManager>
#include <QNetworkRequest>
#include <QNetworkReply>
#include <QDir>
#include <QRegExp>
#include <QPixmap>
#include <QDebug>

const QRegExp illegalChars("[\"@&~=\\/:?#!|<>*^]");

TransferItem::TransferItem(QObject *parent) :
    UrlGrabber(parent),
    m_reply(0),
    m_process(0),
    m_service(Services::NoService),
    m_status(Transfers::Paused),
    m_priority(Transfers::NormalPriority),
    m_progress(0),
    m_speed(0.0),
    m_resumeSize(0),
    m_size(0),
    m_convertible(false),
    m_checkedIfConvertible(false),
    m_saveAsAudio(false),
    m_format(Videos::Unknown),
    m_transferType(Transfers::Download),
    m_retries(0)
{
    m_progressTimer.setInterval(2000);
    m_progressTimer.setSingleShot(false);
    this->connect(&m_progressTimer, SIGNAL(timeout()), this, SLOT(updateProgress()));
    this->connect(this, SIGNAL(gotVideoUrl(QUrl,Videos::VideoFormat)), this, SLOT(performDownload(QUrl,Videos::VideoFormat)));
    this->connect(this, SIGNAL(error(QString)), this, SLOT(onUrlError(QString)));
}

TransferItem::TransferItem(const QString &id, VideoItem *video, Transfers::TransferStatus status, bool saveAsAudio, QObject *parent) :
    UrlGrabber(parent),
    m_reply(0),
    m_process(0),
    m_id(id),
    m_videoId(video->videoId()),
    m_title(video->title()),
    m_thumbnailUrl(video->thumbnailUrl()),
    m_service(video->service()),
    m_filename(this->title().replace(illegalChars, "_") + ".mp4"),
    m_tempFileName(this->id().replace(illegalChars, "_")),
    m_status(status),
    m_priority(Transfers::NormalPriority),
    m_progress(0),
    m_speed(0.0),
    m_resumeSize(0),
    m_size(0),
    m_convertible(false),
    m_checkedIfConvertible(false),
    m_saveAsAudio(saveAsAudio),
    m_format(Videos::Unknown),
    m_transferType(Transfers::Download),
    m_retries(0)
{
    video->setDownloaded(true);
    m_progressTimer.setInterval(2000);
    m_progressTimer.setSingleShot(false);
    this->connect(&m_progressTimer, SIGNAL(timeout()), this, SLOT(updateProgress()));
    this->connect(this, SIGNAL(gotVideoUrl(QUrl,Videos::VideoFormat)), this, SLOT(performDownload(QUrl,Videos::VideoFormat)));
    this->connect(this, SIGNAL(error(QString)), this, SLOT(onUrlError(QString)));
}

TransferItem::TransferItem(const QVariantMap &metadata, Transfers::TransferStatus status, QObject *parent) :
    UrlGrabber(parent),
    m_reply(0),
    m_process(0),
    m_title(metadata.value("title").toString()),
    m_service(Services::YouTube),
    m_filename(metadata.value("filePath").toString().section('/', -1)),
    m_status(status),
    m_priority(Transfers::HighPriority),
    m_progress(0),
    m_speed(0.0),
    m_resumeSize(0),
    m_size(0),
    m_convertible(false),
    m_checkedIfConvertible(false),
    m_saveAsAudio(false),
    m_uploadMetadata(metadata),
    m_format(Videos::Unknown),
    m_transferType(Transfers::Upload),
    m_retries(0)
{
    m_progressTimer.setInterval(2000);
    m_progressTimer.setSingleShot(false);
    this->connect(&m_progressTimer, SIGNAL(timeout()), this, SLOT(updateProgress()));
}

TransferItem::~TransferItem() {}

void TransferItem::setFileName(const QString &name) {
    if (name != this->fileName()) {
        m_filename = name;

        if (this->status() != Transfers::Converting) {
            emit fileNameChanged(name);
        }
    }
}

void TransferItem::setPriority(Transfers::TransferPriority priority) {
    if (priority != this->priority()) {
        m_priority = priority;
        emit priorityChanged(priority);
    }
}

void TransferItem::setSize(qint64 size) {
    if (size != this->size()) {
        m_size = size;
        emit sizeChanged(size);
    }
}

bool TransferItem::convertibleToAudio() const {
    if (!m_checkedIfConvertible) {
        m_checkedIfConvertible = true;
        m_convertible = (this->transferType() == Transfers::Download) && (QFile::exists("/usr/bin/ffmpeg")); // Should probably check other locations
    }

    return m_convertible;
}

void TransferItem::setSaveAsAudio(bool saveAsAudio) {
    if (saveAsAudio != this->saveAsAudio()) {
        m_saveAsAudio = saveAsAudio;
        emit saveAsAudioChanged(saveAsAudio);
    }
}

QString TransferItem::statusText() const {
    switch (status()) {
    case Transfers::Queued:
        return tr("Queued");
    case Transfers::Paused:
        return tr("Paused");
    case Transfers::Connecting:
        return tr("Connecting");
    case Transfers::Downloading:
        return tr("Downloading");
    case Transfers::Uploading:
        return tr("Uploading");
    case Transfers::Cancelled:
        return tr("Cancelled");
    case Transfers::Failed:
        return tr("Failed");
    case Transfers::Completed:
        return tr("Completed");
    case Transfers::Converting:
        return tr("Converting");
    default:
        return tr("Unknown");
    }
}

QString TransferItem::priorityText() const {
    switch (priority()) {
    case Transfers::HighPriority:
        return tr("High");
    case Transfers::NormalPriority:
        return tr("Normal");
    case Transfers::LowPriority:
        return tr("Low");
    default:
        return QString();
    }
}

void TransferItem::setStatus(Transfers::TransferStatus status) {
    if (status == this->status()) {
        return;
    }

    switch (status) {
    case Transfers::Paused:
        switch (this->status()) {
        case Transfers::Uploading:
            return;
        case Transfers::Downloading:
            m_status = status;
            this->pauseTransfer();
            return;
        default:
            break;
        }

        break;
    case Transfers::Cancelled:
        switch (this->status()) {
        case Transfers::Downloading:
            m_status = status;
            this->cancelTransfer();
            return;
        case Transfers::Uploading:
            m_status = status;
            this->cancelTransfer();
            return;
        default:
            break;
        }

        break;
    default:
        break;
    }

    m_status = status;
    emit statusChanged(status);
}

void TransferItem::setStatusInfo(const QString &info) {
    if (info != this->statusInfo()) {
        m_statusInfo = info;
        emit statusInfoChanged(info);
    }
}

void TransferItem::setVideoFormat(Videos::VideoFormat videoFormat) {
    if (videoFormat != this->videoFormat()) {
        m_format = videoFormat;
        emit videoFormatChanged(videoFormat);
    }
}

void TransferItem::startTransfer(bool resetRetries) {
    if (resetRetries) {
        m_retries = 0;
    }

    this->setStatus(Transfers::Connecting);

    if (this->transferType() == Transfers::Upload) {
        qDebug() << "Starting upload: " + this->title();
        this->startUpload();
    }
    else {
        qDebug() << "Starting download: " + this->title();
        switch (this->service()) {
        case Services::YouTube:
            switch (this->videoFormat()) {
            case Videos::Unknown:
                this->setYouTubeFormats(Settings::instance()->youtubeDownloadFormats());
                break;
            default:
                this->setYouTubeFormats(QSet<int>() << this->videoFormat());
                break;
            }

            break;
        case Services::Dailymotion:
            switch (this->videoFormat()) {
            case Videos::Unknown:
                this->setDailymotionFormats(Settings::instance()->dailymotionDownloadFormats());
                break;
            default:
                this->setDailymotionFormats(QSet<int>() << this->videoFormat());
                break;
            }

            break;
        case Services::Vimeo:
            switch (this->videoFormat()) {
            case Videos::Unknown:
                this->setVimeoFormats(Settings::instance()->vimeoDownloadFormats());
                break;
            default:
                this->setVimeoFormats(QSet<int>() << this->videoFormat());
                break;
            }

            break;
        default:
            qWarning() << "TransferItem::startTransfer(): No/invalid service.";
            this->setStatusInfo(tr("Invalid service"));
            this->setStatus(Transfers::Failed);
            return;
        }

        this->getVideoUrl(this->service(), this->videoId());
    }
}

void TransferItem::performDownload(const QUrl &url, Videos::VideoFormat videoFormat) {
    switch (videoFormat) {
    case Videos::Unknown:
        break;
    default:
        this->setVideoFormat(videoFormat);
        break;
    }

    QDir dir;
    dir.mkpath(this->downloadPath());
    m_file.setFileName(this->downloadPath() + this->tempFileName());
    m_resumeSize = m_file.size();

    if (m_file.exists()) {
        if (!m_file.open(QIODevice::Append)) {
            this->setStatusInfo(tr("Cannot write to file"));
            this->setStatus(Transfers::Failed);
            return;
        }
    }
    else if (!m_file.open(QIODevice::WriteOnly)) {
        this->setStatusInfo(tr("Cannot write to file"));
        this->setStatus(Transfers::Failed);
        return;
    }

    this->setStatus(Transfers::Downloading);
    QNetworkRequest request(url);
    qDebug() << "Download url: " + url.toString();

    if (m_resumeSize > 0) {
        request.setRawHeader("Range", "bytes=" + QByteArray::number(m_resumeSize) + "-"); // Set 'Range' header if resuming a download
    }

    m_reply = this->networkAccessManager()->get(request);
    m_transferTime.start();
    m_progressTimer.start();

    this->connect(m_reply, SIGNAL(metaDataChanged()), this, SLOT(onDownloadMetadataChanged()));
    this->connect(m_reply, SIGNAL(downloadProgress(qint64,qint64)), this, SLOT(onDownloadProgressChanged(qint64,qint64)));
    this->connect(m_reply, SIGNAL(readyRead()), this, SLOT(onDownloadReadyRead()));
    this->connect(m_reply, SIGNAL(finished()), this, SLOT(onDownloadFinished()));
}

void TransferItem::pauseTransfer() {
    if (m_reply) {
        m_reply->abort();
    }
}

void TransferItem::cancelTransfer() {
    if (m_reply) {
        m_reply->abort();
    }
}

void TransferItem::onDownloadMetadataChanged() {
    if (this->size() == 0) {
        qint64 size = m_reply->rawHeader("Content-Length").toLongLong();

        if (size > 0) {
            this->setSize(size + m_file.size());
        }
    }
}

void TransferItem::onDownloadProgressChanged(qint64 received, qint64 total) {
    Q_UNUSED(total)

    if (received) {
        this->setProgress((m_resumeSize + received) * 100 / this->size());
        this->setSpeed(received * 1000 / m_transferTime.elapsed());
    }
}

void TransferItem::updateProgress() {
    emit progressChanged(this->progress());
    emit speedChanged(this->speed());
}

void TransferItem::onDownloadReadyRead() {
    m_file.write(m_reply->readAll());
}

void TransferItem::onDownloadFinished() {
    if (!m_reply) {
        this->setStatusInfo(tr("Network error"));
        this->setStatus(Transfers::Failed);
        return;
    }

    m_progressTimer.stop();
    m_file.close();
    QUrl redirect = m_reply->attribute(QNetworkRequest::RedirectionTargetAttribute).toUrl();

    if (!redirect.isEmpty()) {
        m_reply->deleteLater();
        this->performDownload(redirect);
        return;
    }

    if (m_reply->error()) {
        if (m_reply->error() != QNetworkReply::OperationCanceledError) {
            if (m_retries > 2) {
                this->setStatusInfo(m_reply->attribute(QNetworkRequest::HttpReasonPhraseAttribute).toString());
                this->setStatus(Transfers::Failed);
                m_reply->deleteLater();
            }
            else {
                m_reply->deleteLater();
                m_retries++;
                qDebug() << "Retry number " + QString::number(m_retries);
                this->startTransfer(false);
            }
        }
        else {
            switch (this->status()) {
            case Transfers::Cancelled:
                m_file.remove();
                m_reply->deleteLater();
                emit statusChanged(Transfers::Cancelled);
                return;
            case Transfers::Paused:
                m_reply->deleteLater();
                emit statusChanged(Transfers::Paused);
                return;
            default:
                m_reply->deleteLater();
                return;
            }
        }
    }
    else {
        m_reply->deleteLater();
        this->downloadThumbnail();
    }
}

void TransferItem::startUpload() {
    m_file.setFileName(this->uploadMetadata().value("filePath").toString());

    if (!m_file.exists()) {
        this->setStatusInfo(tr("File not found"));
        this->setStatus(Transfers::Failed);
        return;
    }

    this->setSize(m_file.size());
    m_reply = YouTube::instance()->createUploadReply(this->uploadMetadata());
    this->connect(m_reply, SIGNAL(finished()), this, SLOT(setUploadUrl()));
}

void TransferItem::setUploadUrl() {
    if (!m_reply) {
        this->setStatusInfo(tr("Network error"));
        this->setStatus(Transfers::Failed);
        return;
    }

    if (m_reply->error() == QNetworkReply::OperationCanceledError) {
        m_reply->deleteLater();
        emit statusChanged(Transfers::Cancelled);
        return;
    }

    int statusCode = m_reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    if (statusCode == 401) {
        m_reply->deleteLater();
        this->connect(YouTube::instance(), SIGNAL(accessTokenRefreshed(QString)), this, SLOT(startTransfer()));
        YouTube::instance()->refreshAccessToken();
    }
    else {
        if (statusCode == 200) {
            QUrl url(m_reply->rawHeader("Location"));
            m_reply->deleteLater();
            this->performUpload(url);
        }
        else {
            this->setStatusInfo(m_reply->attribute(QNetworkRequest::HttpReasonPhraseAttribute).toString());
            m_reply->deleteLater();
            this->setStatus(Transfers::Failed);
        }

        this->disconnect(YouTube::instance(), SIGNAL(accessTokenRefreshed(QString)), this, SLOT(startTransfer()));
    }
}

void TransferItem::performUpload(const QUrl &url) {
    if (!m_file.open(QIODevice::ReadOnly)) {
        this->setStatusInfo(tr("Unable to read file"));
        this->setStatus(Transfers::Failed);
        return;
    }

    this->setStatus(Transfers::Uploading);
    m_transferTime.start();
    m_progressTimer.start();
    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", QString("cuteTube/%1 (Qt)").arg(Utils::versionNumberString()).toUtf8());
    request.setRawHeader("Host", "uploads.gdata.youtube.com");
    request.setHeader(QNetworkRequest::ContentTypeHeader, "application/octet-stream");
    request.setHeader(QNetworkRequest::ContentLengthHeader, m_file.size());
    m_reply = this->networkAccessManager()->put(request, &m_file);
    this->connect(m_reply, SIGNAL(uploadProgress(qint64,qint64)), this, SLOT(onUploadProgressChanged(qint64,qint64)));
    this->connect(m_reply, SIGNAL(finished()), this, SLOT(onUploadFinished()));
}

void TransferItem::onUploadProgressChanged(qint64 sent, qint64 total) {
    this->setProgress(sent * 100 / total);
    this->setSpeed(sent * 1000 / m_transferTime.elapsed());
}

void TransferItem::onUploadFinished() {
    if (!m_reply) {
        this->setStatusInfo(tr("Network error"));
        this->setStatus(Transfers::Failed);
        return;
    }

    m_progressTimer.stop();
    m_file.close();

    if (m_reply->error()) {
        if (m_reply->error() == QNetworkReply::OperationCanceledError) {
            m_reply->deleteLater();
            emit statusChanged(Transfers::Cancelled);
        }
        else {
            this->setStatusInfo(m_reply->attribute(QNetworkRequest::HttpReasonPhraseAttribute).toString());
            m_reply->deleteLater();
            this->setStatus(Transfers::Failed);
        }
    }
    else {
        m_reply->deleteLater();
        this->setStatus(Transfers::Completed);
    }
}

void TransferItem::downloadThumbnail() {
    QNetworkRequest request(this->thumbnailUrl());
    m_reply = this->networkAccessManager()->get(request);
    this->connect(m_reply, SIGNAL(finished()), this, SLOT(onThumbnailDownloadFinished()));
}

void TransferItem::onThumbnailDownloadFinished() {
    if (!m_reply) {
        this->setStatus(Transfers::Completed);
        return;
    }

    if (m_reply->error()) {
        if (m_reply->error() != QNetworkReply::OperationCanceledError) {
            this->setStatusInfo(m_reply->attribute(QNetworkRequest::HttpReasonPhraseAttribute).toString());
            m_reply->deleteLater();
            this->setStatus(Transfers::Failed);
            return;
        }
        else {
            switch (this->status()) {
            case Transfers::Cancelled:
                m_file.remove();
                m_reply->deleteLater();
                emit statusChanged(Transfers::Cancelled);
                return;
            case Transfers::Paused:
                m_reply->deleteLater();
                emit statusChanged(Transfers::Paused);
                return;
            default:
                break;
            }
        }
    }

    QDir dir;
    dir.mkpath(Settings::instance()->downloadPath() + ".thumbnails/");
    QFile file(QString("%1.thumbnails/%2.jpg").arg(Settings::instance()->downloadPath()).arg(this->fileName().section('.', 0, -2)));
    int num = 1;

    while ((file.exists()) && (num < 100)) {
        file.setFileName(QString("%1(%2).jpg").arg(m_file.fileName().section('.', 0, -2)).arg(num));
        num++;
    }

    if (file.open(QIODevice::WriteOnly)) {
        file.write(m_reply->readAll());
        file.close();
    }

    m_reply->deleteLater();

    if ((this->convertibleToAudio()) && (this->saveAsAudio())) {
        this->convertVideoToAudio();
    }
    else {
        this->moveDownloadedFiles();
    }
}

void TransferItem::convertVideoToAudio() {
    this->setStatus(Transfers::Converting);

    if (!m_process) {
        m_process = new QProcess(this);
        this->connect(m_process, SIGNAL(finished(int,QProcess::ExitStatus)), this, SLOT(onConvertToAudioFinished(int,QProcess::ExitStatus)));
    }

    m_process->setWorkingDirectory(this->downloadPath());
    m_process->start(QString("ffmpeg -i \"%1\" -acodec copy -y -vn \"%2.m4a\"").arg(this->tempFileName()).arg(this->tempFileName()));
}

void TransferItem::onConvertToAudioFinished(int exitCode, QProcess::ExitStatus exitStatus) {
    if ((exitCode == 0) && (exitStatus == QProcess::NormalExit)) {
        QFile::remove(this->downloadPath() + this->tempFileName());
        this->setTempFileName(this->tempFileName() + ".m4a");
        this->setFileName(this->fileName().section('.', 0, -2) + ".m4a");
    }
    else {
        qWarning() << m_process->readAllStandardError();
    }

    this->moveDownloadedFiles();
}

void TransferItem::onUrlError(const QString &errorString) {
    this->setStatusInfo(errorString);
    this->setStatus(Transfers::Failed);
}

void TransferItem::moveDownloadedFiles() {
    QString oldFileName = this->downloadPath() + this->tempFileName();
    QString newFileName = Settings::instance()->downloadPath() + this->fileName();
    qDebug() << "Old fileName: " + oldFileName;
    qDebug() << "New fileName: " + newFileName;
    int num = 1;
    bool fileSaved = QFile::rename(oldFileName, newFileName);

    while ((!fileSaved) && (num < 100)) {
        newFileName = QString("%1(%2).%3").arg(newFileName.section('.', 0, -2)).arg(num).arg(newFileName.section('.', -1));
        qDebug() << "New fileName: " + newFileName;
        fileSaved = QFile::rename(oldFileName, newFileName);
        num++;
    }

    if (fileSaved) {
        this->setStatus(Transfers::Completed);
    }
    else {
        this->setStatusInfo(tr("Unable to rename temporary downloaded file"));
        this->setStatus(Transfers::Failed);
    }
}
