#include "youtubesubscriptionstab.h"
#include "youtubeuserlistdelegate.h"
#include "youtubechannelwindow.h"
#include "youtubevideolistwindow.h"
#include "../base/youtubesubscriptionlistmodel.h"
#include "../base/session.h"
#include "listview.h"
#include <QMenu>
#include <QAction>
#include <QLabel>
#include <QGridLayout>

YouTubeSubscriptionsTab::YouTubeSubscriptionsTab(Session *session, QWidget *parent) :
    AbstractTab(parent),
    m_session(session),
    m_model(new YouTubeSubscriptionListModel(session, this)),
    m_listView(new ListView(this)),
    m_contextMenu(new QMenu(this)),
    m_unsubscribeAction(m_contextMenu->addAction(tr("Unsubscribe"), this, SLOT(unsubscribe()))),
    m_videosAction(new QAction(tr("Latest videos"), this)),
    m_noResultsLabel(new QLabel(QString("<p style='color:#4d4d4d;font-size:40px;'><center><b>%1</b></center></p>").arg(tr("No subscriptions found")), this)),
    m_grid(new QGridLayout(this))
{
    this->addAction(m_videosAction);

    m_grid->setContentsMargins(0, 0, 0, 0);
    m_grid->addWidget(m_listView, 0, 0);
    m_listView->setItemDelegate(new YouTubeUserListDelegate(m_listView));
    m_noResultsLabel->hide();

    if (m_session->youtube()->subscriptionsLoaded()) {
        this->onSubscriptionsLoaded();
    }
    else {
        m_session->toggleBusy(true);
        this->connect(m_session->youtube(), SIGNAL(allSubscriptionsLoaded()), this, SLOT(onSubscriptionsLoaded()));
    }

    this->connect(m_model, SIGNAL(countChanged(int)), this, SLOT(onCountChanged(int)));
    this->connect(m_videosAction, SIGNAL(triggered()), this, SLOT(showSubscriptionVideos()));

    this->onOrientationChanged();
}

void YouTubeSubscriptionsTab::setLandscapeLayout() {
    m_noResultsLabel->setFixedSize(800, 400);
}

void YouTubeSubscriptionsTab::setPortraitLayout() {
    m_noResultsLabel->setFixedSize(480, 700);
}

void YouTubeSubscriptionsTab::onSubscriptionsLoaded() {
    m_session->toggleBusy(false);
    m_listView->setModel(m_model);
    m_noResultsLabel->setVisible(m_model->rowCount() == 0);

    this->connect(m_listView, SIGNAL(clicked(QModelIndex)), this, SLOT(onItemClicked(QModelIndex)));
    this->connect(m_listView, SIGNAL(customContextMenuRequested(QPoint)), this, SLOT(showContextMenu(QPoint)));
}

void YouTubeSubscriptionsTab::onCountChanged(int count) {
    m_noResultsLabel->setVisible(count == 0);
}

void YouTubeSubscriptionsTab::onItemClicked(const QModelIndex &index) {
    if (index.isValid()) {
        this->showUserChannel(index.data(YouTubeSubscriptionListModel::IdRole).toString());
    }
}

void YouTubeSubscriptionsTab::showUserChannel(const QString &userId) {
    YouTubeChannelWindow *window = new YouTubeChannelWindow(m_session, userId, m_session->currentWindow());
    window->show();
}

void YouTubeSubscriptionsTab::showSubscriptionVideos() {
    YouTubeVideoListWindow *window = new YouTubeVideoListWindow(m_session, m_session->currentWindow());
    window->setWindowTitle(tr("Latest subscription videos"));
    window->setVideoFeed(YOUTUBE_NEW_SUBSCRIPTION_VIDEOS_FEED);
    window->show();
}

void YouTubeSubscriptionsTab::showContextMenu(const QPoint &pos) {
    m_contextMenu->popup(pos, m_unsubscribeAction);
}

void YouTubeSubscriptionsTab::unsubscribe() {
    m_session->youtube()->unsubscribe(m_model->data(m_listView->currentIndex(), YouTubeSubscriptionListModel::SubscriptionIdRole).toString());
}
