#include "vimeovideolistwindow.h"
#include "vimeovideoinfowindow.h"
#include "vimeoalbumdialog.h"
#include "vimeovideoselectionwindow.h"
#include "videolistdelegate.h"
#include "videoplaybackwindow.h"
#include "../base/session.h"
#include "../base/vimeovideolistmodel.h"
#include "listview.h"
#include <QAction>
#include <QLabel>
#include <QMenuBar>
#include <QTimer>

VimeoVideoListWindow::VimeoVideoListWindow(Session *session, QWidget *parent) :
    StackedWindow(session, parent),
    m_listView(new ListView(this)),
    m_model(new VimeoVideoListModel(m_session, this)),
    m_group(0),
    m_groupAction(this->menuBar()->addAction(tr("Join group"), this, SLOT(toggleGroupMembership()))),
    m_deleteFromUploadsAction(this->menuBar()->addAction(tr("Delete from uploads"), this, SLOT(deleteVideosFromUploads()))),
    m_deleteFromWatchLaterAction(this->menuBar()->addAction(tr("Delete from album"), this, SLOT(deleteVideosFromWatchLaterAlbum()))),
    m_addToLikesAction(this->menuBar()->addAction(tr("Add to likes"), this, SLOT(addVideosToLikes()))),
    m_deleteFromLikesAction(this->menuBar()->addAction(tr("Delete from likes"), this, SLOT(deleteVideosFromLikes()))),
    m_addToAlbumAction(this->menuBar()->addAction(tr("Add to album"), this, SLOT(addVideosToAlbum()))),
    m_addToWatchLaterAction(this->menuBar()->addAction(tr("Watch later"), this, SLOT(addVideosToWatchLaterAlbum()))),
    m_addToQueueAction(this->menuBar()->addAction(tr("Add to playback queue"), this, SLOT(queueVideos()))),
    m_videoDownloadAction(this->menuBar()->addAction(tr("Download"), this, SLOT(addVideoDownloads()))),
    m_audioDownloadAction(this->menuBar()->addAction(tr("Download (audio)"), this, SLOT(addAudioDownloads()))),
    m_contextMenu(new QMenu(this)),
    m_deleteFromUploadsContextAction(m_contextMenu->addAction(tr("Delete from uploads"), this, SLOT(deleteVideoFromUploads()))),
    m_deleteFromWatchLaterContextAction(m_contextMenu->addAction(tr("Delete from album"), this, SLOT(deleteVideoFromWatchLaterAlbum()))),
    m_addToLikesContextAction(m_contextMenu->addAction(tr("Add to Likes"), this, SLOT(addVideoToLikes()))),
    m_deleteFromLikesContextAction(m_contextMenu->addAction(tr("Delete from likes"), this, SLOT(deleteVideoFromLikes()))),
    m_addToAlbumContextAction(m_contextMenu->addAction(tr("Add to album"), this, SLOT(showAlbumDialog()))),
    m_addToWatchLaterContextAction(m_contextMenu->addAction(tr("Watch later"), this, SLOT(addVideoToWatchLaterAlbum()))),
    m_addToQueueContextAction(m_contextMenu->addAction(tr("Add to playback queue"), this, SLOT(queueVideo()))),
    m_videoDownloadContextAction(m_contextMenu->addAction(tr("Download"), this, SLOT(addVideoDownload()))),
    m_audioDownloadContextAction(m_contextMenu->addAction(tr("Download (audio)"), this, SLOT(addAudioDownload()))),
    m_noResultsLabel(new QLabel(QString("<p style='color:#4d4d4d;font-size:40px;'><center><b>%1</b></center></p>").arg(tr("No videos found")), this))
{
    this->setCentralWidget(m_listView);

    VideoListDelegate *delegate = new VideoListDelegate(m_listView);

    m_listView->setModel(m_model);
    m_listView->setItemDelegate(delegate);
    m_noResultsLabel->hide();

    this->connect(m_model, SIGNAL(loadingChanged(bool)), this, SLOT(onLoadingChanged(bool)));
    this->connect(m_model, SIGNAL(countChanged(int)), this, SLOT(onCountChanged(int)));
    this->connect(m_listView, SIGNAL(atEnd()), m_model, SLOT(getMoreVideos()));
    this->connect(m_listView, SIGNAL(customContextMenuRequested(QPoint)), this, SLOT(showContextMenu(QPoint)));
    this->connect(delegate, SIGNAL(clicked(QModelIndex)), this, SLOT(onItemClicked(QModelIndex)));
    this->connect(delegate, SIGNAL(play(QModelIndex)), this, SLOT(playVideo(QModelIndex)));

    this->onOrientationChanged();
}

void VimeoVideoListWindow::showEvent(QShowEvent *event) {
    StackedWindow::showEvent(event);

    if ((m_model) && (m_model->refreshRequired())) {
        QTimer::singleShot(1000, m_model, SLOT(reset()));
    }
}

void VimeoVideoListWindow::setLandscapeLayout() {
    m_noResultsLabel->setFixedSize(800, 400);
}

void VimeoVideoListWindow::setPortraitLayout() {
    m_noResultsLabel->setFixedSize(480, 700);
}

void VimeoVideoListWindow::setVideoFeed(const QString &feed) {
    m_model->getVideos(feed);
    this->setMenuActions();
}

void VimeoVideoListWindow::getGroupVideos(QSharedPointer<GroupItem> group) {
    m_group = group;
    m_model->getGroupVideos(m_group.data()->id());
    this->setMenuActions();

    this->connect(m_group.data(), SIGNAL(memberChanged(bool)), this, SLOT(setMenuActions()));
}

void VimeoVideoListWindow::search(const QString &query, Queries::QueryOrder order) {
    this->setWindowTitle(tr("Search") + QString(" ('%1')").arg(query));
    m_model->search(query, order);
    this->setMenuActions();
}

void VimeoVideoListWindow::setMenuActions() {
    bool signedIn = m_session->vimeo()->userSignedIn();

    m_groupAction->setEnabled((signedIn) && (!m_group.isNull()));

    if (m_groupAction->isEnabled()) {
        m_groupAction->setText(m_group.data()->member() ? tr("Leave group") : tr("Join group"));
    }

    m_deleteFromUploadsAction->setEnabled((signedIn) && (m_model->feed() == VIMEO_UPLOADS_FEED));
    m_addToLikesAction->setEnabled((signedIn) && (m_model->feed() != VIMEO_LIKES_FEED));
    m_deleteFromLikesAction->setEnabled((signedIn) && (m_model->feed() == VIMEO_LIKES_FEED));
    m_addToAlbumAction->setEnabled(signedIn);
    m_addToWatchLaterAction->setEnabled((signedIn) && (m_model->feed() != VIMEO_WATCH_LATER_FEED));
    m_deleteFromWatchLaterAction->setEnabled((signedIn) && (m_model->feed() == VIMEO_WATCH_LATER_FEED));
    m_deleteFromUploadsContextAction->setVisible((signedIn) && (m_model->feed() == VIMEO_UPLOADS_FEED));
    m_addToLikesContextAction->setVisible((signedIn) && (m_model->feed() != VIMEO_LIKES_FEED));
    m_deleteFromLikesContextAction->setVisible((signedIn) && (m_model->feed() == VIMEO_LIKES_FEED));
    m_addToAlbumContextAction->setVisible(signedIn);
    m_addToWatchLaterContextAction->setVisible((signedIn) && (m_model->feed() != VIMEO_WATCH_LATER_FEED));
    m_deleteFromWatchLaterContextAction->setVisible((signedIn) && (m_model->feed() == VIMEO_WATCH_LATER_FEED));
}

void VimeoVideoListWindow::onLoadingChanged(bool loading) {
    this->toggleBusy(loading);
    m_noResultsLabel->setVisible((!loading) && (m_model->rowCount() == 0));
}

void VimeoVideoListWindow::onCountChanged(int count) {
    m_noResultsLabel->setVisible((count == 0) && (!m_model->loading()));
}

void VimeoVideoListWindow::toggleGroupMembership() {
    if (m_group.data()->member()) {
        m_session->vimeo()->leaveGroup(m_group.data()->id());
    }
    else {
        m_session->vimeo()->joinGroup(m_group.data()->id());
    }
}

void VimeoVideoListWindow::deleteVideosFromUploads() {
    VimeoVideoSelectionWindow *window = new VimeoVideoSelectionWindow(m_session, m_model, tr("Delete from uploads"), SLOT(deleteVideosFromUploads()), this);
    window->show();
}

void VimeoVideoListWindow::addVideosToLikes() {
    VimeoVideoSelectionWindow *window = new VimeoVideoSelectionWindow(m_session, m_model, tr("Add to likes"), SLOT(addVideosToLikes()), this);
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void VimeoVideoListWindow::deleteVideosFromLikes() {
    VimeoVideoSelectionWindow *window = new VimeoVideoSelectionWindow(m_session, m_model, tr("Delete from likes"), SLOT(deleteVideosFromLikes()), this);
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void VimeoVideoListWindow::addVideosToAlbum() {
    VimeoVideoSelectionWindow *window = new VimeoVideoSelectionWindow(m_session, m_model, tr("Add to album"), SLOT(showAlbumDialog()), this);
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void VimeoVideoListWindow::addVideosToWatchLaterAlbum() {
    VimeoVideoSelectionWindow *window = new VimeoVideoSelectionWindow(m_session, m_model, tr("Watch later"), SLOT(addVideosToWatchLaterAlbum()), this);
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void VimeoVideoListWindow::deleteVideosFromWatchLaterAlbum() {
    VimeoVideoSelectionWindow *window = new VimeoVideoSelectionWindow(m_session, m_model, tr("Delete from album"), SLOT(deleteVideosFromWatchLaterAlbum()), this);
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void VimeoVideoListWindow::addVideoDownloads() {
    VimeoVideoSelectionWindow *window = new VimeoVideoSelectionWindow(m_session, m_model, tr("Download"), SLOT(addVideoDownloads()), this);
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void VimeoVideoListWindow::addAudioDownloads() {
    VimeoVideoSelectionWindow *window = new VimeoVideoSelectionWindow(m_session, m_model, tr("Download"), SLOT(addAudioDownloads()), this);
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void VimeoVideoListWindow::queueVideos() {
    VimeoVideoSelectionWindow *window = new VimeoVideoSelectionWindow(m_session, m_model, tr("Add to queue"), SLOT(queueVideos()), this);
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void VimeoVideoListWindow::showContextMenu(const QPoint &pos) {
    m_contextMenu->popup(pos, m_deleteFromUploadsContextAction);
}

void VimeoVideoListWindow::deleteVideoFromUploads() {
    m_session->vimeo()->deleteFromUploads(QStringList(m_model->data(m_listView->currentIndex(), AbstractVideoListModel::VideoIdRole).toString()));
}

void VimeoVideoListWindow::addVideoToLikes() {
    m_session->vimeo()->addToLikes(QStringList(m_model->data(m_listView->currentIndex(), AbstractVideoListModel::VideoIdRole).toString()));
}

void VimeoVideoListWindow::deleteVideoFromLikes() {
    m_session->vimeo()->deleteFromLikes(QStringList(m_model->data(m_listView->currentIndex(), AbstractVideoListModel::VideoIdRole).toString()));
}

void VimeoVideoListWindow::showAlbumDialog() {
    VimeoAlbumDialog *dialog = new VimeoAlbumDialog(m_session, this);
    dialog->open();

    this->connect(dialog, SIGNAL(albumChosen(QString)), this, SLOT(addVideoToAlbum(QString)));
    this->connect(dialog, SIGNAL(addToNewAlbum(QVariantMap)), this, SLOT(addVideoToNewAlbum(QVariantMap)));
}

void VimeoVideoListWindow::addVideoToAlbum(const QString &id) {
    m_session->vimeo()->addToAlbum(QStringList(m_model->data(m_listView->currentIndex(), AbstractVideoListModel::VideoIdRole).toString()), id);
}

void VimeoVideoListWindow::addVideoToNewAlbum(const QVariantMap &album) {
    m_session->vimeo()->createAlbum(album, QStringList(m_model->data(m_listView->currentIndex(), AbstractVideoListModel::VideoIdRole).toString()));
}

void VimeoVideoListWindow::addVideoToWatchLaterAlbum() {
    m_session->vimeo()->addToWatchLaterAlbum(QStringList(m_model->data(m_listView->currentIndex(), AbstractVideoListModel::VideoIdRole).toString()));
}

void VimeoVideoListWindow::deleteVideoFromWatchLaterAlbum() {
    m_session->vimeo()->deleteFromWatchLaterAlbum(QStringList(m_model->data(m_listView->currentIndex(), AbstractVideoListModel::VideoIdRole).toString()));
}

void VimeoVideoListWindow::addVideoDownload() {
    m_session->addDownloadTransfer(m_model->get(m_listView->currentIndex().row()));
}

void VimeoVideoListWindow::addAudioDownload() {
    m_session->addDownloadTransfer(m_model->get(m_listView->currentIndex().row()), true);
}

void VimeoVideoListWindow::queueVideo() {
    m_session->playbackQueue()->addVideo(m_model->get(m_listView->currentIndex().row()));
}

void VimeoVideoListWindow::onItemClicked(const QModelIndex &index) {
    if (index.isValid()) {
        this->showVideoInfo(m_model->get(index.row()));
    }
}

void VimeoVideoListWindow::showVideoInfo(QSharedPointer<VideoItem> video) {
    VimeoVideoInfoWindow *window = new VimeoVideoInfoWindow(m_session, video, this);
    window->show();
}

void VimeoVideoListWindow::playVideo(const QModelIndex &index) {
    if (m_session->settings()->mediaPlayer() == "cutetube") {
        VideoPlaybackWindow *window = new VideoPlaybackWindow(m_session, QList< QSharedPointer<VideoItem> >() << m_model->get(index.row()), this);
        window->show();
    }
    else {
        m_session->urlGrabber()->getVimeoVideoUrl(index.data(AbstractVideoListModel::VideoIdRole).toString());
    }
}
