#include "youtubevideolistmodel.h"
#include "feedurls.h"
#include "session.h"
#include <QNetworkReply>
#include <QNetworkRequest>

YouTubeVideoListModel::YouTubeVideoListModel(Session *session, QObject *parent) :
    AbstractVideoListModel(!session ? 0 : session->newThumbnailCache(), parent),
    m_session(session),
    m_loading(true),
    m_moreResults(true),
    m_error(false),
    m_offset(1),
    m_refresh(false)
{
}

void YouTubeVideoListModel::setSession(Session *session) {
    m_session = session;
}

void YouTubeVideoListModel::clear() {
    AbstractVideoListModel::clear();
    this->setLoading(false);
    this->setOffset(1);
    this->setMoreResults(true);
    this->setError(false);
    this->setRefreshRequired(false);
}

void YouTubeVideoListModel::reset() {
    if (!this->loading()) {
        this->clear();
        this->getMoreVideos();
    }
}

void YouTubeVideoListModel::setFeed(const QString &feed) {
    if (feed != this->feed()) {
        m_feed = feed;

        if (feed == YOUTUBE_UPLOADS_FEED) {
            this->connect(this->session()->youtube(), SIGNAL(deletedFromUploads(QString)), this, SLOT(onDeletedFromUploads(QString)));
        }
        else if (feed == YOUTUBE_FAVOURITES_FEED) {
            this->connect(this->session()->youtube(), SIGNAL(favouriteChanged(QString,bool)), this, SLOT(onFavouriteChanged(QString,bool)));
        }
        else if (feed == YOUTUBE_WATCH_LATER_FEED) {
            this->connect(this->session()->youtube(), SIGNAL(addedToWatchLaterPlaylist(QString)), this, SLOT(onAddedToWatchLaterPlaylist(QString)));
            this->connect(this->session()->youtube(), SIGNAL(deletedFromWatchLaterPlaylist(QString)), this, SLOT(onDeletedFromWatchLaterPlaylist(QString)));
        }
        else if (feed.startsWith(YOUTUBE_PLAYLISTS_BASE_URL)) {
            this->setPlaylistId(feed.section('/', -1).section('?', 0, 0));
            this->connect(this->session()->youtube(), SIGNAL(addedToPlaylist(QString,QString)), this, SLOT(onAddedToPlaylist(QString,QString)));
            this->connect(this->session()->youtube(), SIGNAL(deletedFromPlaylist(QString,QString)), this, SLOT(onDeletedFromPlaylist(QString,QString)));
        }

        emit feedChanged(feed);
    }
}

void YouTubeVideoListModel::getVideos(const QString &feed) {
    if (!feed.isEmpty()) {
        this->setFeed(feed);
    }

    this->setLoading(true);
    QNetworkReply *reply = this->session()->youtube()->createReply(QString("%1&fields=%2").arg(this->feed()).arg(QString(YOUTUBE_VIDEO_FIELDS)), this->offset());
    this->connect(reply, SIGNAL(finished()), this, SLOT(addVideos()));
}

void YouTubeVideoListModel::getMoreVideos() {
    if ((this->moreResults()) && (!this->loading())) {
        if (!this->query().isEmpty()) {
            this->search();
        }
        else {
            this->getVideos();
        }
    }
}

void YouTubeVideoListModel::getRelatedVideos(const QString &id) {
    this->setFeed(QString("%1/%2/related?v=2.1&max-results=30").arg(QString(YOUTUBE_VIDEOS_BASE_URL)).arg(id));
    this->getVideos();
}

void YouTubeVideoListModel::getPlaylistVideos(const QString &id) {
    this->setFeed(QString("%1/%2?v=2.1&max-results=50").arg(QString(YOUTUBE_PLAYLISTS_BASE_URL)).arg(id));
    this->getVideos();
}

void YouTubeVideoListModel::search(const QString &query, int order, int time, int duration, const QString &language) {
    if (!query.isEmpty()) {
        this->setQuery(query);
        this->setOrder(order);
        this->setTimeFilter(time);
        this->setDurationFilter(duration);
        this->setLanguageFilter(language);
    }

    this->setLoading(true);
    QNetworkReply *reply = this->session()->youtube()->createSearchReply(Queries::Videos, this->query(), this->offset(), this->order(), this->timeFilter(), this->durationFilter(), this->languageFilter());
    this->connect(reply, SIGNAL(finished()), this, SLOT(addVideos()));
}

void YouTubeVideoListModel::addVideos() {
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        this->setLoading(false);
        this->setError(true);
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    if (statusCode == 401) {
        this->connect(this->session()->youtube(), SIGNAL(accessTokenRefreshed(QString)), this, SLOT(getVideos()));
        this->connect(this->session()->youtube(), SIGNAL(refreshError()), this, SLOT(onError()));
        this->session()->youtube()->refreshAccessToken();
    }
    else {
        QDomDocument doc;
        doc.setContent(reply->readAll());
        QDomNodeList entries = doc.elementsByTagName("entry");

        for (int i = 0; i < entries.size(); i++) {
            VideoItem *video = new VideoItem;
            video->loadYouTubeVideo(entries.at(i), this->feed() == YOUTUBE_FAVOURITES_FEED);
            this->connect(this->session()->youtube(), SIGNAL(favouriteChanged(QString,bool,QString)), video, SLOT(onFavouriteChanged(QString,bool,QString)));
            this->connect(this->session()->youtube(), SIGNAL(videoLiked(QString)), video, SLOT(onVideoLiked(QString)));
            this->connect(this->session()->youtube(), SIGNAL(videoDisliked(QString)), video, SLOT(onVideoDisliked(QString)));
            this->appendVideo(QSharedPointer<VideoItem>(video));
        }

        this->setLoading(false);
        int totalResults = doc.namedItem("feed").firstChildElement("openSearch:totalResults").text().toInt();
        this->setMoreResults((totalResults > this->rowCount()) && (!entries.isEmpty()));

        if ((!this->playlistId().isEmpty()) && (this->moreResults())) {
            this->setOffset(this->offset() + 50);
            this->getVideos();
        }
        else {
            this->setOffset(this->offset() + 30);
        }

        this->disconnect(this->session()->youtube(), SIGNAL(accessTokenRefreshed(QString)), this, SLOT(getVideos()));
        this->disconnect(this->session()->youtube(), SIGNAL(refreshError()), this, SLOT(onError()));
    }

    reply->deleteLater();
}

void YouTubeVideoListModel::onDeletedFromUploads(const QString &videoId) {
    if (this->removeVideo(videoId)) {
        emit countChanged(this->rowCount());
    }
}

void YouTubeVideoListModel::onFavouriteChanged(const QString &videoId, bool favourite) {
    if (favourite) {
        this->setRefreshRequired(true);
    }
    else if (this->removeVideo(videoId, FavouriteIdRole)) {
        emit countChanged(this->rowCount());
    }
}

void YouTubeVideoListModel::onAddedToWatchLaterPlaylist(const QString &videoId) {
    Q_UNUSED(videoId)

    this->setRefreshRequired(true);
}

void YouTubeVideoListModel::onDeletedFromWatchLaterPlaylist(const QString &playlistVideoId) {
    if (this->removeVideo(playlistVideoId, PlaylistVideoIdRole)) {
        emit countChanged(this->rowCount());
    }
}

void YouTubeVideoListModel::onAddedToPlaylist(const QString &videoId, const QString &playlistId) {
    Q_UNUSED(videoId)

    if (playlistId == this->playlistId()) {
        this->setRefreshRequired(true);
    }
}

void YouTubeVideoListModel::onDeletedFromPlaylist(const QString &playlistVideoId, const QString &playlistId) {
    if ((playlistId == this->playlistId()) && (this->removeVideo(playlistVideoId, PlaylistVideoIdRole))) {
        emit countChanged(this->rowCount());
    }
}

void YouTubeVideoListModel::deleteSelectedVideosFromUploads() {
    if (this->feed() == YOUTUBE_UPLOADS_FEED) {
        QStringList videoIds;

        foreach (int row, this->selectedRows()) {
            videoIds << this->data(this->index(row), VideoIdRole).toString();
        }

        this->session()->youtube()->deleteFromUploads(videoIds);
    }

    this->selectNone();
}

void YouTubeVideoListModel::addSelectedVideosToFavourites() {
    QStringList videoIds;

    foreach (int row, this->selectedRows()) {
        videoIds << this->data(this->index(row), VideoIdRole).toString();
    }

    this->session()->youtube()->addToFavourites(videoIds);
    this->selectNone();
}

void YouTubeVideoListModel::deleteSelectedVideosFromFavourites() {
    if (this->feed() == YOUTUBE_FAVOURITES_FEED) {
        QStringList videoIds;

        foreach (int row, this->selectedRows()) {
            videoIds << this->data(this->index(row), VideoIdRole).toString();
        }

        this->session()->youtube()->deleteFromFavourites(videoIds);
    }

    this->selectNone();
}

void YouTubeVideoListModel::addSelectedVideosToPlaylist(const QString &playlistId) {
    QStringList videoIds;

    foreach (int row, this->selectedRows()) {
        videoIds << this->data(this->index(row), VideoIdRole).toString();
    }

    this->session()->youtube()->addToPlaylist(videoIds, playlistId);
    this->selectNone();
}

void YouTubeVideoListModel::addSelectedVideosToNewPlaylist(const QVariantMap &playlist) {
    QStringList videoIds;

    foreach (int row, this->selectedRows()) {
        videoIds << this->data(this->index(row), VideoIdRole).toString();
    }

    this->session()->youtube()->createPlaylist(playlist, videoIds);
    this->selectNone();
}

void YouTubeVideoListModel::deleteSelectedVideosFromPlaylist(const QString &playlistId) {
    QStringList videoIds;

    foreach (int row, this->selectedRows()) {
        videoIds << this->data(this->index(row), PlaylistVideoIdRole).toString();
    }

    this->session()->youtube()->deleteFromPlaylist(videoIds, playlistId);
    this->selectNone();
}

void YouTubeVideoListModel::addSelectedVideosToWatchLaterPlaylist() {
    QStringList videoIds;

    foreach (int row, this->selectedRows()) {
        videoIds << this->data(this->index(row), VideoIdRole).toString();
    }

    this->session()->youtube()->addToWatchLaterPlaylist(videoIds);
    this->selectNone();
}

void YouTubeVideoListModel::deleteSelectedVideosFromWatchLaterPlaylist() {
    if (this->feed() == YOUTUBE_WATCH_LATER_FEED) {
        QStringList videoIds;

        foreach (int row, this->selectedRows()) {
            videoIds << this->data(this->index(row), PlaylistVideoIdRole).toString();
        }

        this->session()->youtube()->deleteFromWatchLaterPlaylist(videoIds);
    }

    this->selectNone();
}

void YouTubeVideoListModel::downloadSelectedVideos(bool saveAsAudio) {
    this->session()->addDownloadTransfers(this->selectedItems(), saveAsAudio);
    this->selectNone();
}

void YouTubeVideoListModel::queueSelectedVideos() {
    this->session()->playbackQueue()->addVideos(this->selectedItems());
    this->selectNone();
}

VideoItem* YouTubeVideoListModel::getFromQML(int row) const {
    if (VideoItem *video = AbstractVideoListModel::getFromQML(row)) {
        this->connect(this->session()->youtube(), SIGNAL(favouriteChanged(QString,bool,QString)), video, SLOT(onFavouriteChanged(QString,bool,QString)));
        this->connect(this->session()->youtube(), SIGNAL(videoLiked(QString)), video, SLOT(onVideoLiked(QString)));
        this->connect(this->session()->youtube(), SIGNAL(videoDisliked(QString)), video, SLOT(onVideoDisliked(QString)));

        return video;
    }

    return 0;
}
