#include "transferitem.h"
#include "utils.h"
#include "session.h"
#include <QNetworkAccessManager>
#include <QNetworkReply>
#include <QNetworkRequest>
#include <QDir>
#include <QRegExp>
#include <QPixmap>
#include <QTimer>
#include <QDebug>
#ifdef MEEGO_EDITION_HARMATTAN
#include <TransferUI/Transfer>
#endif

#define ILLEGAL_CHARS "[\"@&~=\\/:?#!|<>*^]"

#ifdef MEEGO_EDITION_HARMATTAN
using namespace TransferUI;
#endif

TransferItem::TransferItem(QObject *parent) :
    QObject(parent),
    m_transfer(0),
    m_session(0),
    m_grabber(0),
    m_nam(0),
    m_reply(0),
    m_process(0),
    m_progressTimer(new QTimer(this)),
    m_status(Transfers::Queued),
    m_priority(Transfers::NormalPriority),
    m_progress(0),
    m_speed(0.0),
    m_size(0),
    m_retries(0),
    m_row(0),
    m_convertible(false),
    m_checkedIfConvertible(false),
    m_saveAsAudio(false)
{
    m_progressTimer->setInterval(2000);
    m_progressTimer->setSingleShot(false);

    this->connect(m_progressTimer, SIGNAL(timeout()), this, SLOT(updateProgress()));
}

TransferItem::TransferItem(const QString &id, VideoItem *video, Transfers::TransferStatus status, bool saveAsAudio, QObject *parent) :
    QObject(parent),
    m_transfer(0),
    m_session(0),
    m_grabber(0),
    m_nam(0),
    m_reply(0),
    m_process(0),
    m_progressTimer(new QTimer(this)),
    m_id(id),
    m_videoId(video->videoId()),
    m_title(video->title()),
    m_thumbnailUrl(video->thumbnailUrl()),
    m_service(video->service()),
    m_filename(this->title().replace(QRegExp(ILLEGAL_CHARS), "_") + ".mp4"),
    m_tempFileName(this->id().replace(QRegExp(ILLEGAL_CHARS), "_")),
    m_status(status),
    m_priority(Transfers::NormalPriority),
    m_progress(0),
    m_speed(0.0),
    m_size(0),
    m_retries(0),
    m_row(0),
    m_convertible(false),
    m_checkedIfConvertible(false),
    m_saveAsAudio(saveAsAudio),
    m_transferType(Transfers::Download)
{
    video->setDownloaded(true);
    m_progressTimer->setInterval(2000);
    m_progressTimer->setSingleShot(false);
    this->connect(m_progressTimer, SIGNAL(timeout()), this, SLOT(updateProgress()));
}

TransferItem::TransferItem(const QVariantMap &metadata, QObject *parent) :
    QObject(parent),
    m_transfer(0),
    m_session(0),
    m_grabber(0),
    m_nam(0),
    m_reply(0),
    m_process(0),
    m_progressTimer(new QTimer(this)),
    m_title(metadata.value("title").toString()),
    m_service(Services::YouTube),
    m_filename(metadata.value("filePath").toString().section('/', -1)),
    m_status(Transfers::Queued),
    m_priority(Transfers::HighPriority),
    m_progress(0.0),
    m_size(0),
    m_convertible(false),
    m_checkedIfConvertible(false),
    m_saveAsAudio(false),
    m_uploadMetadata(metadata),
    m_transferType(Transfers::Upload)
{
    m_progressTimer->setInterval(2000);
    m_progressTimer->setSingleShot(false);
    this->connect(m_progressTimer, SIGNAL(timeout()), this, SLOT(updateProgress()));
}

TransferItem::~TransferItem() {
    if (m_reply) {
        delete m_reply;
    }
}

void TransferItem::setHarmattanTransfer(Transfer *transfer) {
#ifdef MEEGO_EDITION_HARMATTAN
    m_transfer = transfer;

    if (this->harmattanTransfer()) {
        this->harmattanTransfer()->waitForCommit();
        this->harmattanTransfer()->setTargetName(Utils::serviceString(this->service()));
        this->harmattanTransfer()->setIcon("icon-m-content-videos-inverse");
        this->harmattanTransfer()->setCanPause(this->transferType() == Transfers::Download);

        if ((this->transferType() == Transfers::Download) && (this->status() == Transfers::Paused)) {
            this->harmattanTransfer()->markPaused();
        }

        this->harmattanTransfer()->commit();
        this->connect(this->harmattanTransfer(), SIGNAL(start()), this, SLOT(startTransfer()));
        this->connect(this->harmattanTransfer(), SIGNAL(pause()), this, SLOT(onTransferPauseRequest()));
        this->connect(this->harmattanTransfer(), SIGNAL(cancel()), this, SLOT(onTransferCancelRequest()));
        this->connect(this->harmattanTransfer(), SIGNAL(repairError()), this, SLOT(startTransfer()));
    }
#else
    Q_UNUSED(transfer)
    qWarning() << "TransferUI::Transfer is for Harmattan only";
#endif
}

void TransferItem::onHarmattanTransferPauseRequest() {
#ifdef MEEGO_EDITION_HARMATTAN
    this->setStatus(Transfers::Paused);
#else
    qWarning() << "TransferUI::Transfer is for Harmattan only";
#endif
}

void TransferItem::onHarmattanTransferCancelRequest() {
#ifdef MEEGO_EDITION_HARMATTAN
    this->setStatus(Transfers::Cancelled);
#else
    qWarning() << "TransferUI::Transfer is for Harmattan only";
#endif
}

void TransferItem::setFileName(const QString &name) {
    if (name != this->fileName()) {
        m_filename = name;

        if (this->status() != Transfers::Converting) {
            emit fileNameChanged(name);
        }
    }
}

void TransferItem::setPriority(Transfers::TransferPriority priority) {
    if (priority != this->priority()) {
        m_priority = priority;
        emit priorityChanged(priority);
    }
}

void TransferItem::setSize(qint64 size) {
    if (size != this->size()) {
        m_size = size;
        emit sizeChanged(size);
#ifdef MEEGO_EDITION_HARMATTAN
        if (this->harmattanTransfer()) {
            this->harmattanTransfer()->setSize(size);
        }
#endif
    }
}

void TransferItem::setRow(uint row) {
    if (row != this->row()) {
        m_row = row;
        emit rowChanged(row);
    }
}

bool TransferItem::convertibleToAudio() const {
    if (!m_checkedIfConvertible) {
        m_checkedIfConvertible = true;
        m_convertible = (this->transferType() == Transfers::Download) && (QFile::exists("/usr/bin/ffmpeg")); // Should probably check other locations
    }

    return m_convertible;
}

void TransferItem::setSaveAsAudio(bool saveAsAudio) {
    if (saveAsAudio != this->saveAsAudio()) {
        m_saveAsAudio = saveAsAudio;
        emit saveAsAudioChanged(saveAsAudio);
    }
}

QString TransferItem::statusText() const {
    switch (status()) {
    case Transfers::Queued:
        return tr("Queued");
    case Transfers::Paused:
        return tr("Paused");
    case Transfers::Connecting:
        return tr("Connecting");
    case Transfers::Downloading:
        return tr("Downloading");
    case Transfers::Uploading:
        return tr("Uploading");
    case Transfers::Cancelled:
        return tr("Cancelled");
    case Transfers::Failed:
        return tr("Failed");
    case Transfers::Completed:
        return tr("Completed");
    case Transfers::Converting:
        return tr("Converting");
    default:
        return tr("Unknown");
    }
}

QString TransferItem::priorityText() const {
    switch (priority()) {
    case Transfers::HighPriority:
        return tr("High");
    case Transfers::NormalPriority:
        return tr("Normal");
    case Transfers::LowPriority:
        return tr("Low");
    default:
        return QString();
    }
}

void TransferItem::setStatus(Transfers::TransferStatus status) {
    if (status != this->status()) {
        if (status == Transfers::Paused) {
            if ((this->transferType() == Transfers::Upload) && (this->status() == Transfers::Uploading)) {
                this->session()->onAlert(tr("Uploads cannot be paused once in progress"));
                return;
            }

            this->pauseTransfer();
        }
        else if (status == Transfers::Cancelled) {
            this->cancelTransfer();
        }
#ifdef MEEGO_EDITION_HARMATTAN
        else if ((status == Transfers::Connecting) && (this->harmattanTransfer())) {
            this->harmattanTransfer()->markResumed();
        }
        else if ((status == Transfers::Completed) && (this->harmattanTransfer())) {
            this->harmattanTransfer()->markCompleted(false);
        }
        else if ((status == Transfers::Failed) && (this->harmattanTransfer())) {
            this->harmattanTransfer()->markRepairableFailure(this->statusText(), this->statusInfo(), tr("Retry"));
        }
#endif
        m_status = status;
        emit statusChanged(status);
    }
}

void TransferItem::setStatusInfo(const QString &info) {
    if (info != this->statusInfo()) {
        m_statusInfo = info;
        emit statusInfoChanged(info);
    }
}

void TransferItem::startTransfer(bool resetRetries) {
    this->setStatus(Transfers::Connecting);

    if (resetRetries) {
        m_retries = 0;
    }

    if (this->transferType() == Transfers::Upload) {
        qDebug() << "starting upload: " + this->title();
        this->startUpload();
    }
    else {
        qDebug() << "starting download: " + this->title();

        if (!m_grabber) {
            m_grabber = new UrlGrabber(UrlGrabber::DownloadMode, this);
            m_grabber->setNetworkAccessManager(this->networkAccessManager());
            this->connect(m_grabber, SIGNAL(gotVideoUrl(QUrl)), this, SLOT(performDownload(QUrl)));
            this->connect(m_grabber, SIGNAL(error(QString)), this, SLOT(onUrlError(QString)));
        }

        m_grabber->setYouTubeQualitySet(this->session()->settings()->youtubeDownloadSet());
        m_grabber->setDailymotionQualitySet(this->session()->settings()->dailymotionDownloadSet());
        m_grabber->setVimeoQualitySet(this->session()->settings()->vimeoDownloadSet());
        m_grabber->getVideoUrl(this->service(), this->videoId());
    }
}

void TransferItem::performDownload(const QUrl &url) {
    QDir dir;
    dir.mkpath(this->downloadPath());

    m_file.setFileName(this->downloadPath() + this->tempFileName());

    if (m_file.exists()) {
        if (!m_file.open(QIODevice::Append)) {
            this->setStatusInfo(tr("Cannot write to file"));
            this->setStatus(Transfers::Failed);
            return;
        }
    }
    else if (!m_file.open(QIODevice::WriteOnly)) {
        this->setStatusInfo(tr("Cannot write to file"));
        this->setStatus(Transfers::Failed);
        return;
    }

    this->setStatus(Transfers::Downloading);
    QNetworkRequest request(url);

    if (m_file.size() > 0) {
        request.setRawHeader("Range", "bytes=" + QByteArray::number(m_file.size()) + "-"); // Set 'Range' header if resuming a download
    }

    m_reply = this->networkAccessManager()->get(request);
    m_transferTime.start();
    m_progressTimer->start();

    this->connect(m_reply, SIGNAL(metaDataChanged()), this, SLOT(onDownloadMetadataChanged()));
    this->connect(m_reply, SIGNAL(downloadProgress(qint64,qint64)), this, SLOT(onProgressChanged(qint64,qint64)));
    this->connect(m_reply, SIGNAL(readyRead()), this, SLOT(onDownloadReadyRead()));
    this->connect(m_reply, SIGNAL(finished()), this, SLOT(onDownloadFinished()));
}

void TransferItem::pauseTransfer() {
    if (m_reply) {
        m_reply->abort();
    }
#ifdef MEEGO_EDITION_HARMATTAN
    if (this->harmattanTransfer()) {
        this->harmattanTransfer()->markPaused();
    }
#endif
}

void TransferItem::cancelTransfer() {
    if (m_reply) {
        m_reply->abort();
    }

    if (m_file.exists()) {
        m_file.remove();
    }
#ifdef MEEGO_EDITION_HARMATTAN
    if (this->harmattanTransfer()) {
        this->harmattanTransfer()->markCancelled();
    }
#endif
}

void TransferItem::onDownloadMetadataChanged() {
    if (m_reply) {
        qint64 size = m_reply->rawHeader("Content-Length").toLongLong();
        QString fileName = QString(m_reply->rawHeader("Content-Disposition")).section("filename=", -1).remove(QRegExp("\'|\"")).replace(QRegExp(ILLEGAL_CHARS), "_");

        if (size > 0) {
            this->setSize(size + m_file.size());
        }

        if ((fileName.contains('.')) && (fileName != this->fileName())) {
            this->setFileName(fileName);
        }
    }
}

void TransferItem::onProgressChanged(qint64 received, qint64 total) {
    if (received) {
        qint64 size = m_file.size();
        this->setProgress((size + received) * 100 / (size + total));
#ifndef MEEGO_EDITION_HARMATTAN
        this->setSpeed(received * 1000 / m_transferTime.elapsed());
#endif
    }
}

void TransferItem::updateProgress() {
#ifdef MEEGO_EDITION_HARMATTAN
    if (this->harmattanTransfer()) {
        this->harmattanTransfer()->setProgress(float (this->progress()) / 100);
    }
#else
    emit progressChanged(this->progress());
    emit speedChanged(this->speed());
#endif
}

void TransferItem::onDownloadReadyRead() {
    if (m_reply) {
        m_file.write(m_reply->readAll());
    }
}

void TransferItem::onDownloadFinished() {
    if (!m_reply) {
        return;
    }

    m_progressTimer->stop();
    m_file.close();
    QUrl redirect = m_reply->attribute(QNetworkRequest::RedirectionTargetAttribute).toUrl();

    if (!redirect.isEmpty()) {
        this->performDownload(redirect);
    }
    else if (m_reply->error()) {
        if (m_reply->error() != QNetworkReply::OperationCanceledError) {
            if (m_retries < 3) {
                m_retries++;
                this->startTransfer(false);
            }
            else {
                QString statusText = m_reply->attribute(QNetworkRequest::HttpReasonPhraseAttribute).toString();
                this->setStatusInfo(statusText);
                this->setStatus(Transfers::Failed);
            }
        }
    }
    else {
        this->downloadThumbnail();
    }
}

void TransferItem::startUpload() {
    m_file.setFileName(this->uploadMetadata().value("filePath").toString());

    if (!m_file.exists()) {
        this->setStatusInfo(tr("File not found"));
        this->setStatus(Transfers::Failed);
        return;
    }

    emit sizeChanged(m_file.size());

    QUrl url("http://uploads.gdata.youtube.com/resumable/feeds/api/users/default/uploads");
    QByteArray xml("<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n" \
                   "<entry xmlns=\"http://www.w3.org/2005/Atom\"\n" \
                   "xmlns:media=\"http://search.yahoo.com/mrss/\"\n" \
                   "xmlns:yt=\"http://gdata.youtube.com/schemas/2007\">\n" \
                   "<media:group>\n" \
                   "<media:title>" + this->uploadMetadata().value("title").toString().toUtf8() + "</media:title>\n" \
                   "<media:description>\n" + this->uploadMetadata().value("description").toString().toUtf8() + "\n\n" + this->uploadMetadata().value("uploadAttribute").toString().toUtf8() + "\n</media:description>\n" \
                   "<media:category scheme=\"http://gdata.youtube.com/schemas/2007/categories.cat\">\n" + this->uploadMetadata().value("category").toString().toUtf8() + "\n</media:category>\n" \
                   "<media:keywords>" + this->uploadMetadata().value("tags").toString().toUtf8() + "</media:keywords>\n" \
                   "</media:group>\n" \
                   "</entry>");

    if (this->uploadMetadata().value("isPrivate").toBool()) {
        xml.insert(xml.indexOf("</media:group>"), "<yt:private/>\n");
    }

    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", QString("cuteTube/%1 (Qt)").arg(Utils::versionNumberString()).toUtf8());
    request.setRawHeader("Host", "uploads.gdata.youtube.com");
    request.setHeader(QNetworkRequest::ContentTypeHeader, "application/atom+xml; charset=UTF-8");
    request.setHeader(QNetworkRequest::ContentLengthHeader, xml.length());
    request.setRawHeader("Authorization", "Bearer " + this->session()->youtube()->accessToken().toUtf8());
    request.setRawHeader("GData-Version", "2");
    request.setRawHeader("X-Gdata-Key", "key=" + this->session()->youtube()->developerKey().toUtf8());
    request.setRawHeader("Slug", this->uploadMetadata().value("filePath").toString().section('/', -1).toUtf8());
    QNetworkReply *reply = this->networkAccessManager()->post(request, xml);
    this->connect(reply, SIGNAL(finished()), this, SLOT(setUploadUrl()));
}

void TransferItem::setUploadUrl() {
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        this->setStatusInfo(tr("Network error"));
        this->setStatus(Transfers::Failed);
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    if (statusCode == 401) {
        this->connect(this->session()->youtube(), SIGNAL(accessTokenRefreshed(QString)), this, SLOT(startTransfer()));
        this->session()->youtube()->refreshAccessToken();
    }
    else {
        if (statusCode == 200) {
            this->performUpload(QUrl(reply->rawHeader("Location")));
        }
        else {
            QString statusText = reply->attribute(QNetworkRequest::HttpReasonPhraseAttribute).toString();
            this->setStatusInfo(statusText);
            this->setStatus(Transfers::Failed);
        }

        this->disconnect(this->session()->youtube(), SIGNAL(accessTokenRefreshed(QString)), this, SLOT(startTransfer()));
    }

    reply->deleteLater();
}

void TransferItem::performUpload(const QUrl &url) {
    if (!m_file.open(QIODevice::ReadOnly)) {
        this->setStatusInfo(tr("Unable to read file"));
        this->setStatus(Transfers::Failed);
        return;
    }

    this->setStatus(Transfers::Uploading);
    m_transferTime.start();
    m_progressTimer->start();
    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", QString("cuteTube/%1 (Qt)").arg(Utils::versionNumberString()).toUtf8());
    request.setRawHeader("Host", "uploads.gdata.youtube.com");
    request.setHeader(QNetworkRequest::ContentTypeHeader, "application/octet-stream");
    request.setHeader(QNetworkRequest::ContentLengthHeader, m_file.size());
    m_reply = this->networkAccessManager()->put(request, &m_file);
    this->connect(m_reply, SIGNAL(uploadProgress(qint64,qint64)), this, SLOT(onProgressChanged(qint64,qint64)));
    this->connect(m_reply, SIGNAL(finished()), this, SLOT(onUploadFinished()));
}

void TransferItem::onUploadFinished() {
    m_progressTimer->stop();
    m_file.close();

    if (m_reply->error()) {
        if (m_reply->error() == QNetworkReply::OperationCanceledError) {
            this->setStatus(Transfers::Cancelled);
        }
        else {
            QString statusText = m_reply->attribute(QNetworkRequest::HttpReasonPhraseAttribute).toString();
            this->setStatusInfo(statusText);
            this->setStatus(Transfers::Failed);
        }
    }
    else {
        this->setStatus(Transfers::Completed);
    }
}

void TransferItem::downloadThumbnail() {
    QNetworkRequest request(this->thumbnailUrl());
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(onThumbnailDownloadFinished()));
}

void TransferItem::onThumbnailDownloadFinished() {
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        this->setStatus(Transfers::Completed);
        return;
    }

    QDir dir;
    dir.mkpath(this->session()->settings()->downloadPath() + ".thumbnails/");
    m_file.setFileName(QString("%1.thumbnails/%2.jpg").arg(this->session()->settings()->downloadPath()).arg(this->fileName().section('.', 0, -2)));

    if (m_file.open(QIODevice::WriteOnly)) {
        m_file.write(reply->readAll());
        m_file.close();
    }

    reply->deleteLater();

    if ((this->convertibleToAudio()) && (this->saveAsAudio())) {
        this->convertVideoToAudio();
    }
    else {
        this->moveDownloadedFiles();
    }
}

void TransferItem::convertVideoToAudio() {
    this->setStatus(Transfers::Converting);

    if (!m_process) {
        m_process = new QProcess(this);
        this->connect(m_process, SIGNAL(finished(int,QProcess::ExitStatus)), this, SLOT(onConvertToAudioFinished(int,QProcess::ExitStatus)));
    }

    m_process->setWorkingDirectory(this->downloadPath());
    m_process->start(QString("ffmpeg -i \"%1\" -acodec copy -y -vn \"%2.m4a\"").arg(this->tempFileName()).arg(this->tempFileName()));
}

void TransferItem::onConvertToAudioFinished(int exitCode, QProcess::ExitStatus exitStatus) {
    if ((exitCode == 0) && (exitStatus == QProcess::NormalExit)) {
        QFile::remove(this->downloadPath() + this->tempFileName());
        this->setTempFileName(this->tempFileName() + ".m4a");
        this->setFileName(this->fileName().section('.', 0, -2) + ".m4a");
    }
    else {
        qWarning() << m_process->readAllStandardError();
    }

    this->moveDownloadedFiles();
}

void TransferItem::onUrlError(const QString &errorString) {
    this->setStatusInfo(errorString);
    this->setStatus(Transfers::Failed);
}

void TransferItem::moveDownloadedFiles() {
    QString oldFileName = this->downloadPath() + this->tempFileName();
    QString newFileName = this->session()->settings()->downloadPath() + this->fileName();

    int num = 1;
    bool fileSaved = QFile::rename(oldFileName, newFileName);

    while ((!fileSaved) && (num < 100)) {
        if (num == 1) {
            newFileName = newFileName.insert(newFileName.lastIndexOf("."), "(" + QByteArray::number(num) + ")");
        }
        else {
            newFileName = newFileName.replace(newFileName.lastIndexOf("(" + QByteArray::number(num - 1) + ")"), 3, "(" + QByteArray::number(num) + ")");
        }

        fileSaved = QFile::rename(oldFileName, newFileName);
        num++;
    }

    if (fileSaved) {
        this->setStatus(Transfers::Completed);
    }
    else {
        this->setStatusInfo(tr("Unable to rename temporary downloaded file"));
        this->setStatus(Transfers::Failed);
    }
}
