#ifndef SESSION_H
#define SESSION_H

#include <qplatformdefs.h>
#include "settings.h"
#include "database.h"
#include "videolauncher.h"
#include "playbacklistmodel.h"
#include "youtube.h"
#include "dailymotion.h"
#include "vimeo.h"
#include "sharing.h"
#include "urlgrabber.h"
#include "cookiejar.h"
#include "transferitem.h"
#include "thumbnailcache.h"
#ifdef MEEGO_EDITION_HARMATTAN
#include <TransferUI/Client>
#include <TransferUI/Transfer>
#endif

class ThumbnailDownloader;
class QProgressDialog;
class QThread;
#ifndef MEEGO_EDITION_HARMATTAN
class Client;
#endif

class Session : public QObject
{
    Q_OBJECT

    Q_PROPERTY(QString clipboardUrl
               READ clipboardUrl
               NOTIFY clipboardUrlChanged)

public:
    explicit Session(QObject *parent = 0);
    ~Session();
    inline Settings* settings() const { return m_settings; }
    inline Database* database() const { return m_database; }
    inline VideoLauncher* videoLauncher() const { return m_launcher; }
    inline PlaybackListModel* playbackQueue() const { return m_queue; }
    inline YouTube* youtube() const { return m_youtube; }
    inline Dailymotion* dailymotion() const { return m_dailymotion; }
    inline Vimeo* vimeo() const { return m_vimeo; }
    inline Sharing* sharing() const { return m_sharing; }
    inline UrlGrabber* urlGrabber() const { return m_grabber; }
    inline CookieJar* cookieJar() const { return m_jar; }
    inline Client* harmattanTransferUI() const { return m_transferUI; }
    inline ThumbnailDownloader* thumbnailDownloader() const { return m_thumbnailDownloader; }
    inline ThumbnailCache* thumbnailCache() const { return m_thumbnailCache; }
    ThumbnailCache* newThumbnailCache();
    inline QNetworkAccessManager* networkAccessManager() const { return m_nam; }
    void setNetworkAccessManager(QNetworkAccessManager *manager);
    QWidget* currentWindow() const;
    Q_INVOKABLE int concurrentTransfers() const;
    inline QList< QSharedPointer<TransferItem> >* transferList() const { return m_transfers; }
    inline QString clipboardUrl() const { return m_clipboardUrl; }
    void restoreStoredDownloads();
    Q_INVOKABLE void addDownloadTransfer(VideoItem *video, bool saveAsAudio = false, bool notify = true);
    void addDownloadTransfer(QSharedPointer<VideoItem> video, bool saveAsAudio = false, bool notify = true);
    Q_INVOKABLE void addDownloadTransfers(QList<VideoItem*> videos, bool saveAsAudio = false);
    void addDownloadTransfers(QList< QSharedPointer<VideoItem> >videos, bool saveAsAudio = false);
    Q_INVOKABLE void addUploadTransfer(const QVariantMap &metadata);

public slots:
    void onAlert(const QString &message);
    void onInfo(const QString &message);
    void onError(const QString &message);
    void showProgressDialog(const QString &message, int numberOfOperations);
    void updateProgressDialog(int progress);
    void cancelProgressDialog();
    void toggleBusy(bool busy) const;
    void addWindowToStack(QWidget *window);
    void removeWindowFromStack(QObject *window);
    void unwindWindowStack();
    void preventDisturb(bool prevent);

private:
    QString generateTransferId(QString seedFileName) const;
    QSharedPointer<TransferItem> getNextTransfer() const;
    QSharedPointer<TransferItem> getTransfer(TransferItem *transfer) const;
    bool removeTransfer(QSharedPointer<TransferItem> transfer);
    void setClipboardUrl(const QString &url);

private slots:
    void onTransferStatusChanged(Transfers::TransferStatus status);
    void onTransferFileNameChanged(const QString &fileName);
    void onTransferPriorityChanged(Transfers::TransferPriority priority);
    void onTransferSaveAsAudioChanged(bool saveAsAudio);
    void onTransferRowChanged(uint row);
    void onTransferCompleted(QSharedPointer<TransferItem> transfer);
    void onTransferCancelled(QSharedPointer<TransferItem> transfer);
    void onMaximumConcurrentTransfersChanged(int oldMaximum, int newMaximum);
    void onClipboardTextChanged();
    void onMediaPlayerChanged(const QString &player);

signals:
    void alert(const QString &message);
    void info(const QString &message);
    void error(const QString &message);
    void busy(const QString &message, int numberOfOperations);
    void busyProgressChanged(int progress);
    void clipboardUrlChanged(const QString &url);
    void transferRemoved(int row);
    
private:
    Settings *m_settings;
    Database *m_database;
    VideoLauncher *m_launcher;
    YouTube *m_youtube;
    Dailymotion *m_dailymotion;
    Vimeo *m_vimeo;
    Sharing *m_sharing;
    UrlGrabber *m_grabber;
    CookieJar *m_jar;
    ThumbnailDownloader *m_thumbnailDownloader;
    QCache<QUrl, Thumbnail> *m_cache;
    ThumbnailCache *m_thumbnailCache;
    QNetworkAccessManager *m_nam;
    QThread *m_thread;
    PlaybackListModel *m_queue;
    QList<QWidget*> m_windowStack;
    QProgressDialog *m_progressDialog;
    QList< QSharedPointer<TransferItem> > *m_transfers;
    QTimer *m_clipboardTimer; // For Harmattan
    Client *m_transferUI; // For Harmattan;
    QString m_clipboardUrl;
};

#endif // SESSION_H
